/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "AddExistingDocumentDialogImpl.h"

#include "DocumentFormatComboboxController.h"

#include <core_api/Settings.h>
#include <core_api/DocumentFormats.h>
#include <core_api/ProjectModel.h>
#include <core_api/DocumentFormatConfigurators.h>
#include <core_api/IOAdapter.h>
#include <core_api/AppContext.h>

#include <util_gui/DialogUtils.h>

#include <QtGui/QFileDialog>
#include <QtGui/QMessageBox>

#include <memory>

/* TRANSLATOR GB2::AddExistingDocumentDialogImpl */

namespace GB2 {


#define SETTINGS_LAST_FORMAT	    "add_existing_document/last_format"
#define SETTINGS_LAST_FILE_FILTER	"add_existing_document/last_file_filter"

AddExistingDocumentDialogImpl::AddExistingDocumentDialogImpl(QWidget* p,  AddExistingDocumentDialogModel& m) 
: QDialog(p), model(m), formatController(NULL), filter(NULL)
{
	setupUi(this);

	if (model.format.isEmpty()) {
		model.format = BaseDocumentFormats::PLAIN_TEXT;
	}

	documentURLEdit->setText(model.url);
	forceReadOnlyCheck->setChecked(model.readOnly);
    save2DiskCheck->setDisabled( true );
	DocumentFormatConstraints c;
	formatController = new DocumentFormatComboboxController(this, forceFormatCombo, c, model.format);
    formatController->setActiveFormatId(AppContext::getSettings()->getValue(SETTINGS_LAST_FORMAT, BaseDocumentFormats::PLAIN_GENBANK).toString());
	model.successful = false;

	connect(documentURLButton, SIGNAL(clicked()), SLOT(sl_documentURLButtonClicked()));
    connect(formatSettingsButton, SIGNAL(clicked()), SLOT(sl_customFormatSettingsClicked()));
	connect(addButton, SIGNAL(clicked()), SLOT(sl_addButtonClicked()));
	connect(documentURLEdit, SIGNAL(editingFinished()), SLOT(sl_documentURLEditFinished()));
    connect(documentURLEdit, SIGNAL(textEdited(const QString&)), SLOT(sl_documentURLTextEdited(const QString&)));
	connect(forceFormatCombo, SIGNAL(currentIndexChanged(int)), SLOT(sl_formatComboCurrentChanged(int)));

	updateState();
}

void AddExistingDocumentDialogImpl::run(QWidget* p, AddExistingDocumentDialogModel& m, bool runFileDialog ) {
    m.successful = false;
    Project* proj = AppContext::getProject();
    if (proj->isStateLocked()) {
        QMessageBox::critical(NULL, tr("error"), tr("project_is_locked"));
        return;
    }
    if (runFileDialog) {
        m.url = selectFile(p);
        if (m.url.isEmpty()) {
            return;
        }
    }

	AddExistingDocumentDialogImpl d(p, m);
	d.exec();
	m = d.model;
    assert(!m.successful || proj->findDocumentByURL(m.url) == NULL);
}

QString AddExistingDocumentDialogImpl::selectFile(QWidget* p) {
    QString filter = DialogUtils::prepareDocumentsFileFilter(true);
    QString allDocsFilter = filter.split('\n').first();
    QString selectedFilter = AppContext::getSettings()->getValue(SETTINGS_LAST_FILE_FILTER, allDocsFilter).toString();
    LastOpenDirHelper lod;
    lod.url = QFileDialog::getOpenFileName(p, tr("open_file_dialog_caption"), lod.dir, filter, &selectedFilter);
    if (!lod.url.isEmpty()) {
        AppContext::getSettings()->setValue(SETTINGS_LAST_FILE_FILTER, selectedFilter);
    }
    return lod.url;
}

static bool isHyperLink( const QString & str ) {
    return str.startsWith("http://") || str.startsWith("https://");
}

void AddExistingDocumentDialogImpl::updateState() {
    const QString& url = documentURLEdit->text();
    bool isNetLink = isHyperLink(url);
    bool ready = formatController->hasSelectedFormat();

    DocumentFormatId format = formatController->getActiveFormatId();
	formatSettingsButton->setEnabled(ready && AppContext::getDocumentFormatConfigurators()->findConfigurator(format)!=NULL);
    save2DiskCheck->setEnabled(isNetLink);
}

void AddExistingDocumentDialogImpl::updateAvailableFormats() {
    QString url = documentURLEdit->text();
    DocumentFormatConstraints c;
    if (isHyperLink(url) || url.isEmpty()) {
	    formatController->updateConstraints(c); //empty constraints -> all formats
        return;
    }

    c.rawData = BaseIOAdapters::readFileHeader(url);
	c.checkRawData = true;
	formatController->updateConstraints(c);

    //try auto select format by ext
    DocumentFormatId formatToSelect;
    QString ext = DialogUtils::getDocumentExtension(url);
    QList<DocumentFormatId> formats = formatController->getFormatsInCombo();
	foreach(DocumentFormatId id, formats) {
		DocumentFormat* f = AppContext::getDocumentFormatRegistry()->getFormatById(id);
		if (f->getSupportedDocumentFileExtensions().contains(ext)) {
            formatToSelect = id;
			break;
		}
	}
    if (formatToSelect.isEmpty() && !formats.isEmpty()) {
        formatToSelect = AppContext::getDocumentFormatRegistry()->sortByDetectPriority(formats).first();
    }
    if (!formatToSelect.isEmpty()) {
        formatController->setActiveFormatId(formatToSelect);
    }
}

void AddExistingDocumentDialogImpl::sl_documentURLButtonClicked() {
    QString file = selectFile(this);
    if (!file.isEmpty()) {
        documentURLEdit->setText(file);	
        updateAvailableFormats();
        updateState();
    }
}

void AddExistingDocumentDialogImpl::sl_addButtonClicked() {
    QString url = documentURLEdit->text();
    bool isNetLink = isHyperLink(url);
    
    if (url.isEmpty()) {
        QMessageBox::critical(this, tr("error"), tr("url_is_empty"));
        documentURLEdit->setFocus();
        return;
    }
    Project* p = AppContext::getProject();
    if (p->findDocumentByURL(url) != NULL) {
        QMessageBox::critical(this, tr("error"), tr("document_is_already_added"));
        documentURLEdit->setFocus();
        return;
    }
    DocumentFormatId format = formatController->getActiveFormatId();
    if (format.isEmpty()) {
        QMessageBox::critical(this, tr("error"), tr("no_format_selected"));
        forceFormatCombo->setFocus();
        return;
    }
    IOAdapterId io = BaseIOAdapters::url2io(url);
    if (io.isEmpty()) {
        QMessageBox::critical(this, tr("error"), tr("no_io_adapter_found"));
        documentURLEdit->setFocus();
        return;
    }

    if ( save2DiskCheck->isChecked() ) { //copy-paste from above
        QString filter = DialogUtils::prepareDocumentsFileFilter(true);
        QString allDocsFilter = filter.split('\n').first();
        QString selectedFilter = AppContext::getSettings()->getValue(SETTINGS_LAST_FILE_FILTER, allDocsFilter).toString();

        LastOpenDirHelper lod;
        lod.url = model.dwnldPath = QFileDialog::getSaveFileName( this, tr("dwnld_file_dialog_caption"), lod, allDocsFilter, &selectedFilter );
    }
	model.format = format;
	model.url = url;
	model.io = io;
	model.successful = true;
	
    QFileInfo f(model.url);
	model.readOnly = forceReadOnlyCheck->isChecked() || isNetLink || !f.isWritable();
    
    AppContext::getSettings()->setValue(SETTINGS_LAST_FORMAT, model.format);

	accept();
}


void AddExistingDocumentDialogImpl::sl_documentURLTextEdited(const QString&) {
    bool isNetLink = isHyperLink(documentURLEdit->text());
    save2DiskCheck->setEnabled( isNetLink);
}

void AddExistingDocumentDialogImpl::sl_documentURLEditFinished() {
    updateState();
    updateAvailableFormats();
}

void AddExistingDocumentDialogImpl::sl_formatComboCurrentChanged(int i) {
    Q_UNUSED(i);
    model.formatSettings.clear();
	updateState();
}


void AddExistingDocumentDialogImpl::sl_customFormatSettingsClicked() {
    DocumentFormatId format =  formatController->getActiveFormatId();
    DocumentFormatConfigurator* c = AppContext::getDocumentFormatConfigurators()->findConfigurator(format);
    if (c == NULL) {
        return;
    }
    c->configure(model.formatSettings);
}

}//namespace
