/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

U_GLOBAL = new Object();

U_GLOBAL.name     = 'BLASTn';
U_GLOBAL.type     = 'http_annotator';
U_GLOBAL.alphabet = 'nucleic';
U_GLOBAL.strand   = 'both';
U_GLOBAL.annotations = [];
U_GLOBAL.max_query_len = 2000;

HOST_URL  = 'http://www.ncbi.nlm.nih.gov/blast/Blast.cgi';

QUERY_URL = HOST_URL + '?CMD=Put&DATABASE=nr&PROGRAM=blastn&QUERY=';
RES_URL   = HOST_URL + '?CMD=Get&FORMAT_TYPE=XML&RID=';

RID = '';

function parseHitParams( xml ) {
    var working = 3;
    var result = [];
    while( working && 'Hit_hsps' != xml.name() ) {
        xml.readNext();
        if( 'Hit_id' == xml.name() && xml.isStartElement() ) {
            result['id'] = xml.readElementText();
            working--;
        }
        if( 'Hit_def' == xml.name() && xml.isStartElement() ) {
            result['def'] = xml.readElementText();
            working--;
        }
        if( 'Hit_accession' == xml.name() && xml.isStartElement() ) {
            result['accession'] = xml.readElementText();
            working--;
        }
    }
    if( working ) {
        throw 'hit parameters parsing failed';
    }
    return result;
}

function parseHsp( xml, ad ) {
    var working = 7;

    var from = -1;
    var to = -1;
    var identities = -1;
    var gaps = -1;
    var align_len = -1;

    while( 'Hsp' != xml.name() || !xml.isEndElement() ) {
        xml.readNext();

        if( 'Hsp_bit-score' == xml.name() && xml.isStartElement() ) {
            ad.addQualifier( 'bit-score', xml.readElementText() );
            working--;
        }
        if( 'Hsp_score' == xml.name() && xml.isStartElement() ) {
            ad.addQualifier( 'score', xml.readElementText() );
            working--;
        }
        if( 'Hsp_evalue' == xml.name() && xml.isStartElement() ) {
            ad.addQualifier( 'E-value', xml.readElementText() );
            working--;
        }            

        if( 'Hsp_query-from' == xml.name() && xml.isStartElement() ) {
            from = parseInt( xml.readElementText(), 10 );
            working--;
        }

        if( 'Hsp_query-to' == xml.name() && xml.isStartElement() ) {
            to = parseInt( xml.readElementText(), 10 );            
            working--;
        }

        if( 'Hsp_hit-from' == xml.name() && xml.isStartElement() ) {
            ad.addQualifier( 'hit-from', xml.readElementText() );
            working--;
        }

        if( 'Hsp_hit-to' == xml.name() && xml.isStartElement() ) {
            ad.addQualifier( 'hit-to', xml.readElementText() );
            working--;
        }
        if( 'Hsp_query-frame' == xml.name() && xml.isStartElement() ) {
            var frame = parseInt( xml.readElementText(), 10 );
            ad.complement = (frame < 0);
        }
        if( 'Hsp_hit-frame' == xml.name() && xml.isStartElement() ) {
            var frame = parseInt( xml.readElementText(), 10 );
            var frame_txt = (frame < 0) ? 'complement' : 'direct';
            ad.addQualifier( 'source frame', frame_txt );
        }
        if( 'Hsp_identity' == xml.name() && xml.isStartElement() ) {
            identities = parseInt( xml.readElementText(), 10 );
        }
        if( 'Hsp_gaps' == xml.name() && xml.isStartElement() ) {
            gaps = parseInt( xml.readElementText(), 10 );
        }
        if( 'Hsp_align-len' == xml.name() && xml.isStartElement() ) {
            align_len = parseInt( xml.readElementText(), 10 );
        }
    }
    if( -1 != from && -1 != to ) {
        ad.addLocation( from-1, to - from + 1);
    } else {
        throw 'cannot parse hsp location';
    }

    if( -1 != align_len ) {
        if( -1 != gaps ) {
            var percent = gaps / align_len * 100;
            var str = '' + gaps + '/' + align_len + ' (' + percent + '%)';
            ad.addQualifier( 'gaps', str );
        }
        if( -1 != identities ) {
            var percent = identities / align_len * 100;
            var str = '' + identities + '/' + align_len + ' (' + percent + '%)';
            ad.addQualifier( 'identities', str );
        }
    }
}

function parseXml( xml_text ) {
    var result = new Array();
    var xml = new QXmlStreamReader( xml_text );
    var added = 0;
    var skipped = 0;
    while( !xml.atEnd() ) {
        xml.readNext();
        if( 'Hit' == xml.name() && xml.isStartElement() ) {
            var cur_hit_params = parseHitParams( xml );
        }
        if( 'Hsp' == xml.name() && xml.isStartElement() ) {
            var ad = new U_GLOBAL.AnnotationData();
            try {
                parseHsp( xml, ad );
            } catch( exc ) {
                U_GLOBAL.log.error( exc );
                continue;
            }
            var ad_len = ad.getSummaryLen();
            if( ad_len <= U_GLOBAL.max_res_len &&
                ad_len >= U_GLOBAL.min_res_len ) 
            {
                ad.setAlpha( false );
                ad.addQualifier( 'id', cur_hit_params['id'] );
                ad.addQualifier( 'def', cur_hit_params['def'] );
                ad.addQualifier( 'accession', cur_hit_params['accession'] );
                ad.name = 'HSP';
                result.push( ad );
                added++;
                U_GLOBAL.log.debug( 'new HSP added' );
                U_GLOBAL.log.debug( ad.toString() );
            } else {
                skipped++;
                U_GLOBAL.log.debug( 'HSP skipped due to length limitations, hsp length is: ' + ad_len );
            }
            
        }
    }
    if( xml.errorString().length ) {
        throw 'xml parsing error: ' + xml.errorString();
    }
    U_GLOBAL.log.info( 'TOTAL: ' + added + ' added ' + ', ' + skipped + ' skipped' );
    return result;
}

function tryGetResults() {
    try{
        var fullUrl = RES_URL + RID;
        U_GLOBAL.log.info( 'trying to get results...' );        
        var response = U_GLOBAL.get( fullUrl ).toString();
//        var response = U_GLOBAL.get( "http:\/\/localhost\/testa.xml" ).toString();
        U_GLOBAL.log.info( 'response is got, obtaining status...' );
        var status_regexp = /Status\s*=\s*(\w*)/;
        var status_res = status_regexp.exec( response );
        if( null != status_res ) {
            status = status_res[1];
            if( status ) {
                if( 'WAITING' == status ) {
                    U_GLOBAL.log.info( 'status: ' + status );
                    return;
                } else {
                    throw 'server error, status: ' + status;
                }
            }
        } else {
//            timer.timeout.disconnect( tryGetResults );
            timer.stop();
            U_GLOBAL.log.info( 'seems like data is ready, trying to parse response' );
            U_GLOBAL.stateInfo.stateDesc = 'parsing results';
            U_GLOBAL.annotations = parseXml( response );                    
            LOOP.exit();
        }
    } catch(e) {
//        U_GLOBAL.log.debug( 'exception thrown: ' + e.toString() );
//        U_GLOBAL.log.debug( 'exception backtrace: ' + e.backtrace );
        LOOP.exit();
        throw e;
    }
}


U_GLOBAL.main = function() {
    LOOP = new QEventLoop();

    var fullUrl = QUERY_URL + U_GLOBAL.query;
    U_GLOBAL.log.debug( 'query size: ' + U_GLOBAL.query.toString().length );
    U_GLOBAL.log.info( 'request queued to ' + HOST_URL + '...' );
    U_GLOBAL.stateInfo.stateDesc = 'waiting for server response';

    var response = U_GLOBAL.get( fullUrl );

    var rid_regexp = /RID\s*=\s*(\w*)/;
    var res = rid_regexp.exec( response );
    if( null != res && res.length > 1 ) {
        RID = res[1];
    } else {
        throw 'cannot obtain request id';
    }

    var rtoe_regexp = /RTOE\s*=\s*(\d*)/;
    res = rtoe_regexp.exec( response );
    if( null != res && res.length > 1 ) {
        var rtoe = res[1];
    } else {
        throw 'cannot obtain rtoe';
    }
    
    if( null == RID || null == rtoe ) {
        throw 'cannot get request parameters';
    }

    U_GLOBAL.log.info( 'estimated time of report: ' + rtoe + ' sec' );
    U_GLOBAL.stateInfo.stateDesc = 'waiting for search results';


    timer = new QTimer(this);
    timer.singleShot = false;
    timer.timeout.connect( tryGetResults );
    timer.start( rtoe * 1000 );   

    LOOP.exec();   
}
