/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "ProjectTasks.h"

#include "ProjectServiceImpl.h"
#include "ProjectImpl.h"
#include "ProjectLoaderImpl.h"
#include "../AppContextImpl.h"
#include "../CorePlugin.h"

#include <util_tasks/SaveDocumentTask.h>
#include <util_tasks/AddDocumentTask.h>
#include <util_tasks/LoadDocumentTask.h>

#include <document_format/DocumentFormatUtils.h>
#include <core_api/IOAdapter.h>
#include <core_api/Log.h>
#include <core_api/ObjectViewModel.h>
#include <core_api/DocumentModel.h>
#include <core_api/DocumentFormats.h>
#include <core_api/GHints.h>

#include <QtXml/qdom.h>
#include <QtGui/QMessageBox>

namespace GB2 {

static LogCategory log(ULOG_CAT_CORE_SERVICES);

//////////////////////////////////////////////////////////////////////////
///Close project
CloseProjectTask::CloseProjectTask() : Task(tr("close_project_task_name"), TaskFlags_NR_DWF_SSSOR)
{
}

void CloseProjectTask::prepare() {
	if (AppContext::getProject()==NULL) {
		stateInfo.error = tr("error_no_active_project");
		return;
	}
	/* TODO: this is done by project view. Need to cleanup this part! 
        addSubTask(new SaveProjectTask(SaveProjectTaskKind_SaveProjectAndDocumentsAskEach));
    */
	ServiceRegistry* sr = AppContext::getPluginSupport()->getServiceRegistry();
	QList<Service*> services = sr->findServices(Service_Project);
	assert(services.size() == 1);
	Service* projectService = services.first();
	addSubTask(sr->unregisterServiceTask(projectService));
}

Task::ReportResult CloseProjectTask::report() {
    propagateSubtaskError();
	return ReportResult_Finished;
}

//////////////////////////////////////////////////////////////////////////
/// OpenProjectTask
OpenProjectTask::OpenProjectTask(const QString& _url, const QString& _name) 
: Task(tr("open_project_task_name"), TaskFlags_NR_DWF_SSSOR), url(_url), name(_name)
{
}

OpenProjectTask::OpenProjectTask(const QList<QUrl>& list) 
: Task(tr("open_project_task_name"), TaskFlags_NR_DWF_SSSOR), urlList(list) {}

static Task* createRegisterProjectTask(Project* proj) {
	CorePlugin* corePlugin = AppContextImpl::getApplicationContext()->getCorePlugin();
    ProjectServiceImpl* ps = new ProjectServiceImpl(corePlugin, proj);
    corePlugin->addService(ps);
	Task* registerServiceTask = AppContext::getPluginSupport()->getServiceRegistry()->registerServiceTask(ps);
	return registerServiceTask;
}

void OpenProjectTask::prepare() {
    if (urlList.size() == 1 && url.isEmpty()) {
        url = urlList.takeFirst().toLocalFile();
    }
    
    if (url.endsWith(PROJECTFILE_EXT)) { // open another project
        QFileInfo f(url);
        if (f.exists() && !(f.isFile() && f.isReadable())) {
            stateInfo.error = tr("invalid_url%1").arg(url);
            return;
        }
        
        // close current
        if (AppContext::getProject()) {
            addSubTask(new CloseProjectTask());
        }

        if (f.exists()) {
            loadProjectTask = new LoadProjectTask(url);
            addSubTask(loadProjectTask);
        } else {
            ProjectImpl* p =  new ProjectImpl(name, url);
            addSubTask(new SaveProjectTask(SaveProjectTaskKind_SaveProjectOnly, p));
            addSubTask(createRegisterProjectTask(p));
        }
    } else { // load a (bunch of) documents
        if (!url.isEmpty()) {
            urlList << QUrl("file:///" + url);
        }
        Project* p = AppContext::getProject();
        if (!p) {
            // create anonymous project
            log.info(tr("Creating new project"));
            p = new ProjectImpl("", "");
            addSubTask(createRegisterProjectTask(p));
        }
        foreach(QUrl _url, urlList) {
            QString url = _url.toLocalFile();
            if (url.endsWith(PROJECTFILE_EXT) || p->findDocumentByURL(url)) {
                // skip project and duplicate files
                stateInfo.error = tr("ignore %1").arg(url);
                continue;
            }
            QList<DocumentFormat*> fs = DocumentFormatUtils::detectFormat(url);
            if (fs.isEmpty()) {
                stateInfo.error = tr("unsupported_document%1").arg(url);
                continue;
            }

            DocumentFormat* format = fs.first();
            assert(format);
            IOAdapterFactory* iof = AppContext::getIOAdapterRegistry()->getIOAdapterFactoryById(BaseIOAdapters::url2io(url));
            Document * newdoc = new Document(format, iof, url);
            addSubTask( new AddDocumentTask(newdoc) );
            if (urlList.size() == 1) {
                addSubTask(new LoadUnloadedDocumentAndOpenViewTask(newdoc));
            }
        }
    }
}

QList<Task*> OpenProjectTask::onSubTaskFinished(Task* subTask) {
	QList<Task*> res;
	if (subTask == loadProjectTask && !loadProjectTask->getStateInfo().hasErrors()) {
		Project* p =  loadProjectTask->detachProject();
		res.append(createRegisterProjectTask(p));
	}
	return res;
}

Task::ReportResult OpenProjectTask::report() {
    propagateSubtaskError();
	return ReportResult_Finished;
}

//////////////////////////////////////////////////////////////////////////
/// Save project
SaveProjectTask::SaveProjectTask(SaveProjectTaskKind _k, Project* p, const QString& _url) 
: Task(tr("save_project_task_name"), TaskFlag_DeleteWhenFinished), k(_k), proj(p), url(_url)
{
	lock = NULL;
}

SaveProjectTask::~SaveProjectTask () {
    assert(lock == NULL);
}

void SaveProjectTask::prepare() {
    if (proj == NULL) {
		proj = AppContext::getProject();
	}
	assert(proj!=NULL);
    if (url.isEmpty()) {
        url = proj->getProjectURL();
    }
    if (url.isEmpty() && (!proj->getGObjectViewStates().isEmpty() || proj->getDocuments().size() > 1)) {
        //ask if to save project
        int code = QMessageBox::question(NULL, tr("question"), tr("save_project_file_question"), QMessageBox::Yes | QMessageBox::No, QMessageBox::No);
        if (code == QMessageBox::Yes) {
            ProjectDialogController d(ProjectDialogController::Save_Project);
            d.projectFolderEdit->setText(QDir::home().absolutePath());
            d.projectNameEdit->setText(tr("My project"));
            int rc = d.exec();
            if (rc == QDialog::Accepted) {
                AppContext::getProject()->setProjectName(d.projectNameEdit->text());
                url = d.projectFolderEdit->text() + "/" + d.projectFileEdit->text();
                if (!url.endsWith(PROJECTFILE_EXT)) {
                    url.append(PROJECTFILE_EXT);
                }
                AppContext::getProject()->setProjectURL(url);
            }
        }
    }
    
    lock = new StateLock(getTaskName(), StateLockFlag_LiveLock);
	proj->lockState(lock);

	if (k!=SaveProjectTaskKind_SaveProjectOnly) {
		QList<Document*> modifiedDocs = SaveMiltipleDocuments::findModifiedDocuments(AppContext::getProject()->getDocuments());
		if (!modifiedDocs.isEmpty()) {
			addSubTask(new SaveMiltipleDocuments(modifiedDocs, k == SaveProjectTaskKind_SaveProjectAndDocumentsAskEach));		
		}
	}
}

void SaveProjectTask::run() {
    if (proj!=NULL && !url.isEmpty()) {
        log.info(tr("Saving project %1").arg(url));
		saveProjectFile(stateInfo, proj, url);
	}
}


Task::ReportResult SaveProjectTask::report() {
	if (!stateInfo.hasErrors() && url == proj->getProjectURL()) {
		proj->setModified(false);
	}
	proj->unlockState(lock);
    delete lock;
    lock = NULL;
	return Task::ReportResult_Finished;
}


static QString map2String(const QVariantMap& map) {
	QByteArray a;
	QVariant v(map);
	QDataStream s(&a, QIODevice::WriteOnly);
	s << v;
	QString res(a.toBase64());
	return res;
}

void SaveProjectTask::saveProjectFile(TaskStateInfo& ts, Project* p, QString url) {
	//TODO: use Project!
	ProjectImpl* pi = qobject_cast<ProjectImpl*>(p);
	assert(pi);
	QDomDocument xmlDoc("GB2PROJECT");

	QDomElement projectElement = xmlDoc.createElement("gb2project");
	projectElement.setAttribute("name", pi->getProjectName());

	//save documents
	foreach(Document* gbDoc, pi->getDocuments()) {
        gbDoc->getDocumentFormat()->updateFormatSettings(gbDoc);

		//save document info
		QDomElement docElement = xmlDoc.createElement("document");
		docElement.setAttribute("url", gbDoc->getURL());
		docElement.setAttribute("io-adapter", gbDoc->getIOAdapterFactory()->getAdapterId());
		DocumentFormat* f = gbDoc->getDocumentFormat();
		QString formatId = f->getFormatId();
		docElement.setAttribute("format", formatId);
        docElement.setAttribute("readonly", gbDoc->hasUserModLock() ? 1 : 0);
        StateLock* l = gbDoc->getDocumentModLock(DocumentModLock_FORMAT_AS_INSTANCE);
        if (l!=NULL) {
            docElement.setAttribute("format-lock", 1);
        }
        QString dataStr = map2String(gbDoc->getGHints()->getMap());
        QDomText textNode = xmlDoc.createCDATASection(dataStr);
        docElement.appendChild(textNode);

        projectElement.appendChild(docElement);
	}

	//save views
	foreach(GObjectViewState* view, pi->getGObjectViewStates()) {
		//save document info
		QDomElement viewElement = xmlDoc.createElement("view");
		viewElement.setAttribute("factory", view->getViewFactoryId());
		viewElement.setAttribute("viewName", view->getViewName());
		viewElement.setAttribute("stateName", view->getStateName());
		QString dataStr = map2String(view->getStateData());
		QDomText textNode = xmlDoc.createCDATASection(dataStr);
		viewElement.appendChild(textNode);

		projectElement.appendChild(viewElement);
	}

	xmlDoc.appendChild(projectElement);

	QByteArray rawData = xmlDoc.toByteArray();
	//	printf(">>%s", xmlDoc.toString().toStdString().c_str());

	QFile f(url);
	if (f.open(QIODevice::WriteOnly | QIODevice::Truncate)) {
		qint64 s = f.write(rawData);
		f.close();
		if (s!=rawData.size()) {
			ts.error = tr("error_writing_a_file_%1").arg(url);
		}
	} else {
		ts.error = tr("error_cant_open_file_%1").arg(url);
	}
}


//////////////////////////////////////////////////////////////////////////
/// LoadProjectTask
LoadProjectTask::LoadProjectTask(const QString& _url) 
: Task(tr("load_project_task_name"), TaskFlag_DeleteWhenFinished), proj(NULL), url(_url)
{
	xmlDoc = new QDomDocument();
}

LoadProjectTask::~LoadProjectTask() {
	if (proj) {
		delete proj;
	}
	delete xmlDoc;
}

void LoadProjectTask::run() {
    log.details(tr("Loading project from: %1").arg(url));
	loadXMLProjectModel(url, stateInfo, *xmlDoc);
}

Task::ReportResult LoadProjectTask::report() {
	if (!stateInfo.hasErrors()) {
        proj = createProjectFromXMLModel(url, *xmlDoc, stateInfo);
        if (proj!=NULL) {
            log.info(tr("Project loaded: %1").arg(url));
        }
	}
	return Task::ReportResult_Finished;
}


static QVariantMap string2Map(const QString& string, bool emptyMapIfError) {
    Q_UNUSED(emptyMapIfError);

	QDataStream s(QByteArray::fromBase64(string.toAscii()));
	QVariant res(QVariant::Map);
	s >> res;
    if (res.type() == QVariant::Map) {
        return res.toMap();
    }
    assert(emptyMapIfError);
    return QVariantMap();
}

void LoadProjectTask::loadXMLProjectModel(const QString& url, TaskStateInfo& si, QDomDocument& doc) {
	assert(doc.isNull());

	QFile f(url);
	if (!f.open(QIODevice::ReadOnly)) {
		si.error = tr("error_open_file %1").arg(url);
		return;
	}
	QByteArray  xmlData = f.readAll();
	f.close();

	bool res = doc.setContent(xmlData);
	if (!res) {
		si.error = tr("error_invalid_content %1").arg(url);
		doc.clear();
	}
	if (doc.doctype().name()!="GB2PROJECT") {
		si.error = tr("error_invalid_content %1").arg(url);
		doc.clear();
	}
}

Project* LoadProjectTask::createProjectFromXMLModel(const QString& pURL, const QDomDocument& xmlDoc, TaskStateInfo& si) {
    Q_UNUSED(si); //TODO: report about errors using si!!!

    
	QDomElement projectElement = xmlDoc.documentElement();
	QString name = projectElement.attribute("name");

	ProjectImpl* pi = new ProjectImpl(name, pURL);
	
	//read all documents
	QDomNodeList documents = projectElement.elementsByTagName("document");
	for(int i=0;i<documents.size(); i++) {
		QDomNode n = documents.item(i);
		assert(n.isElement());
		if (!n.isElement()) {
			continue;
		}
		QDomElement docElement = n.toElement();
		QString ioAdapterId = docElement.attribute("io-adapter");
		QString docURL = docElement.attribute("url");
		DocumentFormatId format = docElement.attribute("format");
        bool readonly = docElement.attribute("readonly").toInt() != 0;
        bool instanceLock = docElement.attribute("format-lock").toInt() != 0;
		IOAdapterFactory* iof = AppContext::getIOAdapterRegistry()->getIOAdapterFactoryById(ioAdapterId);
		DocumentFormat* df = AppContext::getDocumentFormatRegistry()->getFormatById(format);
		assert(df!=NULL);
        QVariantMap fs = string2Map(docElement.text(), true);
        Document* d = new Document(df, iof, docURL, fs, instanceLock ? tr("last_loaded_state_was_locked_by_format") : QString());
        d->setUserModLock(readonly);
		pi->addDocument(d);
	}

	// read all saved views
	QDomNodeList viewElements = projectElement.elementsByTagName("view");
	for(int i=0;i<viewElements.size(); i++) {
		QDomNode n = viewElements.item(i);
		assert(n.isElement());
		if (!n.isElement()) {
			continue;
		}
		QDomElement viewElement = n.toElement();
		GObjectViewFactoryId id = viewElement.attribute("factory");
		QString viewName = viewElement.attribute("viewName");
		QString stateName = viewElement.attribute("stateName");
		QVariantMap map  = string2Map(viewElement.text(), false);
		GObjectViewState* state = new GObjectViewState(id, viewName, stateName, map);
		pi->addState(state);
	}
	pi->setModified(false);
    return pi;
}

} //namespace
