/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "ServiceRegistryImpl.h"

#include <core_api/AppContext.h>
#include <core_api/Log.h>
#include <core_api/PluginModel.h>

#include <QtCore/QTimerEvent>

namespace GB2 {

/* TRANSLATOR GB2::ServiceRegistryImpl */

static LogCategory log(ULOG_CAT_USER_INTERFACE);

ServiceRegistryImpl::~ServiceRegistryImpl() {
	assert(services.isEmpty());
	assert(activeServiceTasks.isEmpty());
}

QList<Service*> ServiceRegistryImpl::findServices(ServiceType t ) const {
	QList<Service*> res;
    foreach(Service* s, services) {
        if (s->getType() == t) {
            res.append(s);
        }
    }
    return res;
}

void ServiceRegistryImpl::unregisterPluginServices(Plugin* p) {
	QList<Service*> servicesToRemove;
	foreach(Service* s, services) {
		if (s->getPlugin() == p) {
			assert(s->isDisabled());
			servicesToRemove.append(s);
		}
	}
	foreach(Service* s, servicesToRemove) {
		services.removeAll(s);
	}

	foreach(Service* s, servicesToRemove) {
		emit si_serviceUnregistered(s);
	}
}


/// Returns 'true' if a service with the specified ServiceType is registered and enabled
Task* ServiceRegistryImpl::registerServiceTask(Service* s) {
	return new RegisterServiceTask(this, s);
}

Task* ServiceRegistryImpl::unregisterServiceTask(Service* s) {
	return new UnregisterServiceTask(this, s);
}



Task* ServiceRegistryImpl::enableServiceTask(Service* s) {
	return new EnableServiceTask(this, s);
}

Task* ServiceRegistryImpl::disableServiceTask(Service* s) {
	return new DisableServiceTask(this, s, true);
}


void ServiceRegistryImpl::setServiceState(Service* s, ServiceState state) {
    if (s->getState() == state) {
        return;
    }
	_setServiceState(s, state);

	initiateServicesCheckTask();
}

void ServiceRegistryImpl::initiateServicesCheckTask() {
	if (!timerIsActive) {
		timerIsActive = true;
		startTimer(100);
	}
}

void ServiceRegistryImpl::timerEvent(QTimerEvent *event) {
	if (!activeServiceTasks.empty()) {
		return; //wait until no active service tasks left
	}
	killTimer(event->timerId());
	timerIsActive = false;
	Service* s = findServiceReadyToEnable();
	if (s!=NULL) {
		AppContext::getTaskScheduler()->registerTopLevelTask(new EnableServiceTask(this, s));
	}
}

Service* ServiceRegistryImpl::findServiceReadyToEnable() const {
	//TODO: recheck circular tasks too

	// look for new + parent_disabled services and check if a service can be run
	foreach(Service* s, services) {
		if (s->getState() == ServiceState_Disabled_New || s->getState() == ServiceState_Disabled_ParentDisabled) {
			bool allParentsEnabled = true;
			QList<ServiceType> parentTypes = s->getParentServiceTypes();
			foreach(ServiceType t, parentTypes) {
				QList<Service*> parentServices = findServices(t);
				bool parentIsEnabled = false;
				foreach(Service* ps, parentServices) {
					if (ps->isEnabled()) {
						parentIsEnabled = true;
						break;
					}
				}
				if (!parentIsEnabled) {
					allParentsEnabled = false;
					break;
				}
			}
			if (allParentsEnabled) {
				return s;
			}
		}
	}
	return NULL;
}

//////////////////////////////////////////////////////////////////////////
// Tasks

/// RegisterServiceTask

RegisterServiceTask::RegisterServiceTask(ServiceRegistryImpl* _sr, Service* _s) 
: Task(tr("register_service_task_%1").arg(_s->getName()), TaskFlags_NR_DWF), sr(_sr), s(_s)
{
	assert(s->getState() == ServiceState_Disabled_New);
	assert(!sr->services.contains(s));
}

void RegisterServiceTask::prepare() {
    if (sr->services.contains(s)) {
        stateInfo.error = tr("service_is_already_registered_%1").arg(s->getName());
        return;
    }
    if (s->getState() != ServiceState_Disabled_New) {
		stateInfo.error = tr("illegal_service_state_%1").arg(s->getName());
		return;
	}
	sr->services.append(s);
	emit sr->si_serviceRegistered(s);
	addSubTask(new EnableServiceTask(sr, s));
}

///EnableServiceTask

EnableServiceTask::EnableServiceTask(ServiceRegistryImpl* _sr, Service* _s) 
: Task(tr("enable_service_task_%1").arg(_s->getName()), TaskFlags_NR_DWF), sr(_sr), s(_s)
{
	assert(s->isDisabled());
}

static bool findCircular(ServiceRegistryImpl* sr, Service* s, int currentDepth =0);
static bool checkAllParentsEnabled(ServiceRegistryImpl* sr, Service* s);

void EnableServiceTask::prepare() {
//TODO: improve messaging. The service name is already mentined in task name!

	sr->activeServiceTasks.push_back(this);
	if (s->isEnabled()) {
		stateInfo.error = tr("service_is_enabled_%1").arg(s->getName());
		return;
	}
	bool circular = findCircular(sr, s);
	if (circular) {
		sr->setServiceState(s, ServiceState_Disabled_CircularDependency);
		stateInfo.error = tr("service_circular_dependency_%1").arg(s->getName());
		return;
	}
	bool noparent = !checkAllParentsEnabled(sr, s);
	if (noparent) {
		sr->setServiceState(s, ServiceState_Disabled_ParentDisabled);
		stateInfo.error = tr("service_parent_not_enabled_%1").arg(s->getName());
		return;
	}
	
	Task* t = sr->createServiceEnablingTask(s);
	if (t != NULL) {
		addSubTask(t);
	}

	sr->initiateServicesCheckTask();
}

Task::ReportResult EnableServiceTask::report() {
	sr->activeServiceTasks.removeAll(this);
	if (stateInfo.hasErrors() || s->isEnabled()) {
		return ReportResult_Finished;
	}
	bool success = !propagateSubtaskError();
	sr->setServiceState(s, success ? ServiceState_Enabled : ServiceState_Disabled_FailedToStart);
	return ReportResult_Finished;
}


static bool findCircular(ServiceRegistryImpl* sr, Service* s, int currentDepth) {
	currentDepth++;
	if (currentDepth > sr->getServices().size()) {
		return true;
	}
	foreach(ServiceType st, s->getParentServiceTypes()) {
		QList<Service*> parents = sr->findServices(st);
		foreach(Service* p, parents) {
			bool circular = findCircular(sr, p, currentDepth);
			if (circular) {
				return true;
			}
		}
	}
	return false;
}

static bool checkAllParentsEnabled(ServiceRegistryImpl* sr, Service* s) {
	foreach(ServiceType st, s->getParentServiceTypes()) {
		QList<Service*> parents = sr->findServices(st);
		if (parents.isEmpty()) {
			return false;
		}
		foreach(Service* p, parents) {
			if (p->isDisabled()) {
				return false;
			}
		}
	}
	return true;
}

/// UnregisterServiceTask
UnregisterServiceTask::UnregisterServiceTask(ServiceRegistryImpl* _sr, Service* _s) 
: Task(tr("unregister_service_task_%1").arg(_s->getName()), TaskFlags_NR_DWF), sr(_sr), s(_s)
{
	assert(sr->services.contains(s));
}

void UnregisterServiceTask::prepare() {
	if (!sr->services.contains(s)) {
		stateInfo.error = tr("service_is_not_registered_%1").arg(s->getName());
		return;
	}
	if (s->isEnabled()) {
		addSubTask(new DisableServiceTask(sr, s, false));
	}
}

Task::ReportResult UnregisterServiceTask::report() {
	if (stateInfo.hasErrors()) {
		return ReportResult_Finished;
	}
	if (s->isDisabled()) {
		assert(sr->services.count(s) == 1);
		sr->services.removeAll(s);
		emit sr->si_serviceUnregistered(s);
        delete s;//TODO: redesign real-time service registration/unregistration-> synchronize with plugin list
	}
	return ReportResult_Finished;
}


/// DisableServiceTask
DisableServiceTask::DisableServiceTask(ServiceRegistryImpl* _sr, Service* _s, bool _manual) 
: Task(tr("disable_service_task_%1").arg(_s->getName()), TaskFlags_NR_DWF_SSSOR), sr(_sr), s(_s), manual(_manual)
{
	assert(sr->services.contains(s) && s->isEnabled());
}

static QList<Service*> getDirectChilds(ServiceRegistryImpl* sr, ServiceType st);

void DisableServiceTask::prepare() {
	sr->activeServiceTasks.push_back(this);
	if (AppContext::getTaskScheduler()->getTopLevelTasks().size() > 1) {
		stateInfo.error = tr("error_active_tasks_found");
		return;
	}
	if (!sr->services.contains(s)) {
		stateInfo.error = tr("service_is_not_registered_%1").arg(s->getName());
		return;
	}
	if (!s->isEnabled()) {
		stateInfo.error = tr("service_is_not_enabled_%1").arg(s->getName());
		return;
	}
	bool stopChilds = sr->findServices(s->getType()).size() == 1;
	if (stopChilds) {
		QList<Service*> childsToDisable = getDirectChilds(sr, s->getType());
		foreach(Service* c, childsToDisable) {
			if (c->isEnabled()) {
				addSubTask(new DisableServiceTask(sr, c, false));
			}
		}
	}
	Task* disablingTask = sr->createServiceDisablingTask(s);
	if (disablingTask!=NULL) {
		addSubTask(disablingTask);
	}
}

Task::ReportResult DisableServiceTask::report() {
	sr->activeServiceTasks.removeAll(this);
	if (stateInfo.hasErrors() || s->isDisabled()) {
		return ReportResult_Finished;
	}
	if (propagateSubtaskError()) {
		return ReportResult_Finished;
	}
	ServiceState newState = manual ? ServiceState_Disabled_Manually : ServiceState_Disabled_ParentDisabled;
	sr->setServiceState(s, newState);
	return ReportResult_Finished;
}


static QList<Service*> getDirectChilds(ServiceRegistryImpl* sr, ServiceType st) {
	QList<Service*> res;
	foreach(Service* s, sr->getServices()) {
		if (s->getParentServiceTypes().contains(st)) {
			res.append(s);
		}
	}
	return res;
}


}//namespace
