/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "AppContextImpl.h"
#include "SettingsImpl.h"
#include "DocumentFormatRegistryImpl.h"
#include "IOAdapterRegistryImpl.h"
#include "CorePlugin.h"

#include "plugin_support/PluginSupportImpl.h"
#include "plugin_viewer/PluginViewerImpl.h"
#include "project_support/ProjectLoaderImpl.h"
#include "main_window/MainWindowImpl.h"
#include "project_view/ProjectViewImpl.h"
#include "logview/LogCache.h"
#include "logview/LogView.h"
#include "task_scheduler/TaskSchedulerImpl.h"
#include "task_view/TaskViewController.h"
#include "script_registry/ScriptManagerView.h"
#include "app_settings/AppSettingsImpl.h"
#include "app_settings/AppSettingsGUIImpl.h"

#include <core_api/Log.h>
#include <core_api/DNATranslation.h>
#include <core_api/ObjectViewModel.h>
#include <core_api/ResourceTracker.h>
#include <core_api/DocumentFormatConfigurators.h>
#include <core_api/ScriptRegistry.h>
#include <core_api/DBXRefRegistry.h>
#include <core_api/UserApplicationsSettings.h>

#include <document_format/DNAAlphabetRegistryImpl.h>
#include <gobjects/AnnotationSettings.h>
#include <test_framework/GTestFrameworkComponents.h>
#include <util_gui/BaseDocumentFormatConfigurators.h>

#include <workflow_support/WorkflowEnvImpl.h>
#include <workflow_library/BioActorLibrary.h>

#include <QtGui/QApplication>
#include <QtGui/QIcon>

static GB2::LogCategory logCat(ULOG_CAT_USER_INTERFACE);

/* TRANSLATOR GB2::CorePlugin */

namespace GB2 {

static AppContextImpl appContext;


AppContextImpl* AppContextImpl::getApplicationContext() { 
    return &appContext; 
}

static CorePlugin* createCorePlugin() {
    CorePlugin* plug = new CorePlugin();
    
    Service* s = new PluginViewerImpl(plug);
    plug->addService(s);

    s = new ProjectViewImpl(plug);
    plug->addService(s);

    s = new ScriptRegistryService( plug );
    plug->addService( s );

    return plug;
}

}//namespace

using namespace GB2;

static void openProject(const QString& fileName) {
    QFileInfo f (fileName);
    Task* t = AppContext::getProjectLoader()->openProjectTask(f.absoluteFilePath());
    AppContext::getTaskScheduler()->registerTopLevelTask(t);
}

int main(int argc, char **argv) 
{
    QT_REQUIRE_VERSION( argc, argv, "4.4" );
    QApplication app(argc, argv);
    QCoreApplication::addLibraryPath(QCoreApplication::applicationDirPath());


    LogCache logsCache;

    logCat.details(CorePlugin::tr("UGENE initialization started"));

	AppContextImpl* appContext = AppContextImpl::getApplicationContext();

    //1 create 
    SettingsImpl* globalSettings = new SettingsImpl(QSettings::SystemScope);
    appContext->setGlobalSettings(globalSettings);

	SettingsImpl* settings = new SettingsImpl(QSettings::UserScope);
	appContext->setSettings(settings);

    AppSettings* appSettings = new AppSettingsImpl();
    appContext->setAppSettings(appSettings);


	QTranslator translator;
    QString transFile[] = {
        AppContext::getAppSettings()->getUserAppsSettings()->getTranslationFile(),
        "transl_" + QLocale::system().name(),
        "transl_en"
    };
    bool trOK = false;
    for (int i = transFile[0].isEmpty() ? 1 : 0; i < 3; ++i) {
        if (!translator.load(transFile[i], QCoreApplication::applicationDirPath())) {
            fprintf(stderr, "Translation not found: %s\n", transFile[i].toAscii().constData());
        } else {
            trOK = true;
            break;
        }
    }
    if (!trOK) {
        fprintf(stderr, "No translations found, exiting\n");
	    return 1;   
	}
	
	app.installTranslator(&translator);

    ResourceTracker* resTrack = new ResourceTracker();
    appContext->setResourceTracker(resTrack);

	TaskSchedulerImpl* ts = new TaskSchedulerImpl();
	appContext->setTaskScheduler(ts);

    AnnotationSettingsRegistry* asr = new AnnotationSettingsRegistry(settings);
    appContext->setAnnotationSettingsRegistry(asr);

    TestFramework* tf = new TestFramework();
    appContext->setTestFramework(tf);

    MainWindowImpl* mw = new MainWindowImpl();
	appContext->setMainWindow(mw);
	mw->show();

    AppSettingsGUI* appSettingsGUI = new AppSettingsGUIImpl();
    appContext->setAppSettingsGUI(appSettingsGUI);

    AppContext::getMainWindow()->getDockManager()->registerDock(MWDockArea_Bottom, new TaskViewDockWidget(), QKeySequence(Qt::ALT | Qt::Key_2));
    AppContext::getMainWindow()->getDockManager()->registerDock(MWDockArea_Bottom, new LogViewDockWidget(&logsCache), QKeySequence(Qt::ALT | Qt::Key_3));
    
    GObjectViewFactoryRegistry* ovfr = new GObjectViewFactoryRegistry();
    appContext->setObjectViewFactoryRegistry(ovfr);

    CorePlugin* corePlug = createCorePlugin();
	appContext->setCorePlugin(corePlug);

	PluginSupportImpl* psp = new PluginSupportImpl(corePlug);
    appContext->setPluginSupport(psp);

	DocumentFormatRegistryImpl* dfr = new DocumentFormatRegistryImpl();
	appContext->setDocumentFormatRegistry(dfr);

    DocumentFormatConfigurators* dfc = new DocumentFormatConfigurators();
    appContext->setDocumentFormatConfigurators(dfc);
    BaseDocumentFormatConfigurators::initBuiltInConfigurators();
    

	IOAdapterRegistryImpl* io = new IOAdapterRegistryImpl();
	appContext->setIOAdapterRegistry(io);

	DNATranslationRegistry* dtr = new DNATranslationRegistry();
	appContext->setDNATranslationRegistry(dtr);

	DNAAlphabetRegistry* dal = new DNAAlphabetRegistryImpl(dtr);
	appContext->setDNAAlphabetRegistry(dal);

	/*ObjectRegistry* or = new ObjectRegistry();
	appContext->setObjectRegistry(or);*/

	ProjectLoaderImpl* pli = new ProjectLoaderImpl();
	appContext->setProjectLoader(pli);

    ScriptManagerView * smv = new ScriptManagerView();

	DBXRefRegistry* dbxr = new DBXRefRegistry();
	appContext->setDBXRefRegistry(dbxr);

    Workflow::WorkflowEnv::init(new Workflow::WorkflowEnvImpl());
    Workflow::BioActorLibrary::init();

    if (argc == 2) {
        openProject(argv[1]);
    } else if (AppContext::getAppSettings()->getUserAppsSettings()->openLastProjectAtStartup()) {
        QString lastProject = ProjectLoaderImpl::getLastProjectURL();
        if (!lastProject.isEmpty()) {
            openProject(lastProject);
        }
    }
    
    //2 run QT GUI
    logCat.info(CorePlugin::tr("UGENE started"));
	int rc = app.exec();

    //3. deallocate resources
    Workflow::WorkflowEnv::shutdown();
    delete smv;
   
    delete dbxr;
    appContext->setDBXRefRegistry(NULL);

    delete pli;
	appContext->setProjectLoader(NULL);

	delete psp;
    appContext->setPluginSupport(NULL);

	delete mw;
	appContext->setMainWindow(NULL);

    delete tf;
    appContext->setTestFramework(0);


    //delete or;
	//appContext->setObjectRegistry(NULL);

	delete ovfr;
	appContext->setObjectViewFactoryRegistry(NULL);

	delete dal;
	appContext->setDNAAlphabetRegistry(NULL);

	delete dtr;
	appContext->setDNATranslationRegistry(NULL);

	delete io;
	appContext->setIOAdapterRegistry(NULL);

    delete dfc;
    appContext->setDocumentFormatConfigurators(NULL);

    delete dfr;
	appContext->setDocumentFormatRegistry(NULL);

	delete ts;
	appContext->setTaskScheduler(NULL);

	delete asr;
    appContext->setAnnotationSettingsRegistry(NULL);

    delete resTrack;
    appContext->setResourceTracker(NULL);

    delete appSettingsGUI;
    appContext->setAppSettingsGUI(NULL);

    delete appSettings;
    appContext->setAppSettings(NULL);

    delete settings;
    appContext->setSettings(NULL);

	delete globalSettings;
	appContext->setGlobalSettings(NULL);
	
	return rc;
}

