/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "umuscleTests.h"
#include "MuscleTask.h"

#include <util_tasks/SaveDocumentTask.h>

#include <core_api/DocumentModel.h>
#include <core_api/DocumentFormats.h>
#include <core_api/AppContext.h>
#include <core_api/IOAdapter.h>

#include <gobjects/GObjectTypes.h>
#include <gobjects/MAlignmentObject.h>
#include <gobjects/DNASequenceObject.h>

/* TRANSLATOR GB2::GTest*/

namespace GB2 {

#define OUT_FILE_NAME_ATTR "out"
#define IN_FILE_NAME_ATTR "in"
#define INDEX_ATTR "index"
#define DOC1_ATTR "doc1"
#define DOC2_ATTR "doc2"

void GTest_uMuscle::init(XMLTestFormat *tf, const QDomElement& el) {
    Q_UNUSED(tf);

    ctxAdded = false;
    ma_result = NULL;

    inputDocCtxName = el.attribute(IN_FILE_NAME_ATTR);
    if (inputDocCtxName.isEmpty()) {
        stateInfo.error = GTest::tr("value not set %1").arg(IN_FILE_NAME_ATTR);
        return;
    }

    resultCtxName = el.attribute(INDEX_ATTR);

}

QList<Task*> GTest_uMuscle::onSubTaskFinished(Task* subTask) {
    Q_UNUSED(subTask);
    QList<Task*> res;
    if(subTask == mTask) {
        if(mTask->hasErrors()) {
            return res;
        }
        if(!stateInfo.cancelFlag) {
            IOAdapterFactory* iof = AppContext::getIOAdapterRegistry()->getIOAdapterFactoryById(BaseIOAdapters::LOCAL_FILE);
            res << new SaveDocumentTask(doc,iof, env->getVar("TEMP_DATA_DIR")+"/"+"muscle_aligned_doc.aln");
        }
    }
    return res;
}

void GTest_uMuscle::prepare() {
    doc = getContext<Document>(this, inputDocCtxName);
    if (doc == NULL) {
        stateInfo.error = GTest::tr("context not found %1").arg(inputDocCtxName);
        return;
    }

    QList<GObject*> list = doc->findGObjectByType(GObjectTypes::MULTIPLE_ALIGNMENT);
    if (list.size() == 0) {
        stateInfo.error = GTest::tr("container of object with type \"%1\" is empty").arg(GObjectTypes::MULTIPLE_ALIGNMENT);
        return;
    }

    GObject *obj = list.first();
    if(obj==NULL){
        stateInfo.error = GTest::tr("object with type \"%1\" not found").arg(GObjectTypes::MULTIPLE_ALIGNMENT);
        return;
    }
    assert(obj!=NULL);
    MAlignmentObject* ma = qobject_cast<MAlignmentObject*>(obj);
    if(ma==NULL){
        stateInfo.error = GTest::tr("error can't cast to multiple alignment from GObject");
        return;
    }
    MuscleTaskSettings s;
    s.stableMode = false; //default mode is 'group' like in MUSCLE
    //ma_result = (MAlignmentObject*)ma->clone();
    ma_result = ma;
    mTask = new MuscleGObjectTask(ma_result,s);
    addSubTask(mTask);
}

Task::ReportResult GTest_uMuscle::report() {
    if(mTask->hasErrors()) {
        stateInfo.error = mTask->getError();
        return ReportResult_Finished;
    }
    ma_result->getMAlignment().alignedSeqs;
    if(!resultCtxName.isEmpty()) {
        ctxAdded = true;
        addContext(resultCtxName, ma_result);
    }

    return ReportResult_Finished;
}

void GTest_uMuscle::cleanup() {
    //if(ma_result!=NULL)
    //    delete ma_result;
    if(ctxAdded) 
        removeContext(resultCtxName);
}

void GTest_CompareMAlignment::init(XMLTestFormat *tf, const QDomElement& el) {
    Q_UNUSED(tf);

    doc1CtxName = el.attribute(DOC1_ATTR);
    if (doc1CtxName.isEmpty()) {
        stateInfo.error = GTest::tr("value not set %1").arg(DOC1_ATTR);
        return;
    }
    doc2CtxName = el.attribute(DOC2_ATTR);
    if (doc2CtxName.isEmpty()) {
        stateInfo.error = GTest::tr("value not set %1").arg(DOC2_ATTR);
        return;
    }
}

Task::ReportResult GTest_CompareMAlignment::report() {
    Document* doc1 = getContext<Document>(this, doc1CtxName);
    if (doc1 == NULL) {
        stateInfo.error = GTest::tr("document not found %1").arg(doc1CtxName);
        return ReportResult_Finished;
    }
    Document* doc2 = getContext<Document>(this, doc2CtxName);
    if (doc2 == NULL) {
        stateInfo.error = GTest::tr("document not found %1").arg(doc2CtxName);
        return ReportResult_Finished;
    }

    QList<GObject*> objs1 = doc1->findGObjectByType(GObjectTypes::MULTIPLE_ALIGNMENT);
    QList<GObject*> objs2 = doc2->findGObjectByType(GObjectTypes::MULTIPLE_ALIGNMENT);

    if(objs1.size()!=objs2.size()) {
        stateInfo.error = GTest::tr("MAlignmentObjects count not matched %1, expected %2").arg(objs1.size()).arg(objs2.size());
        return ReportResult_Finished;
    }

    int listSize = objs1.size();
    for (int i=0;i<listSize;i++) {
        MAlignmentObject* ma1 = qobject_cast<MAlignmentObject*>(objs1.at(i));
        MAlignmentObject* ma2 = qobject_cast<MAlignmentObject*>(objs2.at(i));
        const QList<MAlignmentItem> &alignedSeqs1 = ma1->getMAlignment().alignedSeqs;
        const QList<MAlignmentItem> &alignedSeqs2 = ma2->getMAlignment().alignedSeqs;
        if(ma1->objectName()!=ma2->objectName()) {
            stateInfo.error = GTest::tr("MAlignmentObjects name not matched \"%1\", expected \"%2\"").arg(ma1->objectName()).arg(ma2->objectName());
            return ReportResult_Finished;
        }
        foreach(const MAlignmentItem &maItem1, alignedSeqs1) {    
            bool nameFound = false;
            foreach(const MAlignmentItem &maItem2, alignedSeqs2) {        
                if (maItem1.name == maItem2.name) {
                    nameFound = true;
                    if(maItem2.sequence.length() != maItem1.sequence.length()) {
                        stateInfo.error = GTest::tr("Aligned sequences \"%1\" length not matched \"%2\", expected \"%3\"").arg(maItem1.name).arg(QString(maItem1.sequence.length())).arg(QString(maItem2.sequence.length()));
                        return ReportResult_Finished;
                    }
                    if (maItem1.sequence != maItem2.sequence) {
                        stateInfo.error = GTest::tr("Aligned sequences \"%1\" not matched \"%2\", expected \"%3\"").arg(maItem1.name).arg(QString(maItem1.sequence)).arg(QString(maItem2.sequence));
                        return ReportResult_Finished;
                    }
                }
            }
            if (!nameFound) {
                stateInfo.error = GTest::tr("aligned sequence not found \"%1\"").arg(maItem1.name);
            }

        }

    }
    return ReportResult_Finished;
}

void GTest_uMuscleAddUnalignedSequenceToProfile::init(XMLTestFormat *tf, const QDomElement& el) {
    origAliSeqs = 0;
    aliObj = NULL;
    resultAliSeqs = 0;
    aliDocName = el.attribute("ali-doc");
    if (aliDocName.isEmpty()) {
        stateInfo.error = GTest::tr("value not set %1").arg("ali-doc");
        return;
    }
    seqDocName = el.attribute("seq-doc");
    if (seqDocName.isEmpty()) {
        stateInfo.error = GTest::tr("value not set %1").arg("seq-doc");
        return;
    }
    QString gaps = el.attribute("gap-map");
    QStringList gapsPerSeq = gaps.split('|');
    //gapsPerSeq.removeAll(QString());
    foreach (const QString& s, gapsPerSeq) {
        QList<int> seqGaps;
        QStringList nums = s.split(',');
        foreach (const QString& n, nums) {
            if (n.isEmpty()) {
                continue;
            }
            bool ok = false;
            int gapPos = n.toInt(&ok);
            if (!ok) {
                stateInfo.error = tr("error parsing gap value '%1', line %2").arg(n).arg(s);
                return;
            }
            seqGaps.append(gapPos);
        }
        gapPositionsForSeqs.append(seqGaps);
    }
    QString resultLen = el.attribute("result-ali-len");
    bool ok = false;
    resultAliLen = resultLen.toInt(&ok);
    if (!ok) {
        stateInfo.error = tr("error result-ali-len '%1'").arg(resultLen);
        return;
    }
}

void GTest_uMuscleAddUnalignedSequenceToProfile::prepare() {
    if (hasErrors()) {
        return;
    }
    Document* aliDoc = getContext<Document>(this, aliDocName);
    if (aliDoc == NULL) {
        stateInfo.error = tr("alignment document not found in context: %1").arg(aliDocName);
        return;
    }
    Document* seqDoc = getContext<Document>(this, seqDocName);
    if (seqDoc == NULL) {
        stateInfo.error = tr("sequence document not found in context: %1").arg(seqDocName);
        return;
    }
    QList<GObject*> aliObjs = aliDoc->findGObjectByType(GObjectTypes::MULTIPLE_ALIGNMENT);
    if (aliObjs.isEmpty()) {
        stateInfo.error = tr("no alignment obejct found in doc: %1").arg(aliDoc->getURL());
        return;
    }
    aliObj = qobject_cast<MAlignmentObject*>(aliObjs[0]);
    origAliSeqs = aliObj->getMAlignment().getNumSequences();
    
    QList<GObject*> seqObjs = seqDoc->findGObjectByType(GObjectTypes::DNA_SEQUENCE);
    if (seqObjs.isEmpty()) {
        stateInfo.error = tr("no sequence objects found in doc: %1").arg(seqDoc->getURL());
        return;
    }
    MAlignment unalignedMA;
    unalignedMA.alphabet = aliObj->getMAlignment().alphabet;
    foreach (GObject* obj, seqObjs) {
        DNASequenceObject* dnaObj = qobject_cast<DNASequenceObject*>(obj);
        unalignedMA.alignedSeqs.append(MAlignmentItem(dnaObj->getGObjectName(), dnaObj->getSequence()));
    }
    if (unalignedMA.getNumSequences()!=gapPositionsForSeqs.size()) {
        stateInfo.error = tr("number of sequences not matches number of gaps in test: %1 sequences and %2 gap lines")
            .arg(unalignedMA.getNumSequences()).arg(gapPositionsForSeqs.size());
        return;
    }
    resultAliSeqs = origAliSeqs + unalignedMA.getNumSequences();

    MuscleTaskSettings s;
    s.op = MuscleTaskOp_AddUnalignedToProfile;
    s.profile = unalignedMA;
    addSubTask(new MuscleGObjectTask(aliObj, s));
}

Task::ReportResult GTest_uMuscleAddUnalignedSequenceToProfile::report() {
    propagateSubtaskError();
    if (hasErrors()) {
        return ReportResult_Finished;
    }
    MAlignment ma = aliObj->getMAlignment();
    if (ma.getLength()!=resultAliLen) {
        stateInfo.error = tr("result alignment length notmatches: %1, expected: %2").arg(ma.getLength()).arg(resultAliLen);
        return ReportResult_Finished;
    }
    
    if (resultAliSeqs!=ma.getNumSequences()) {
        stateInfo.error = tr("unexpected number of sequences in result: %1, expected: %2").arg(ma.getNumSequences()).arg(resultAliSeqs);
        return ReportResult_Finished;
    }

    for (int i = origAliSeqs, j = 0; i < ma.getNumSequences(); i++, j++) {
        QByteArray seq = ma.alignedSeqs[i].sequence;
        QList<int> seqGaps = gapPositionsForSeqs[j];
        for (int pos = 0; pos < seq.size(); pos++) {
            char c = seq[pos];
            if (c == MAlignment_GapChar) {
                bool found = seqGaps.contains(pos);
                if (!found) {
                    stateInfo.error = tr("illegal gap found! pos: %1, sequence: %2").arg(pos).arg(ma.alignedSeqs[i].name);
                    return ReportResult_Finished;
                }
            }
        }
        for (int gap = 0; gap < seqGaps.size(); gap++) {
            int pos  = seqGaps[gap];
            char c = seq[pos];
            if (c != MAlignment_GapChar) {
                stateInfo.error = tr("gap not found! pos: %1, sequence: %2").arg(pos).arg(ma.alignedSeqs[i].name);
                return ReportResult_Finished;
            }
        }
    }
    return ReportResult_Finished;
}


QList<XMLTestFactory*> UMUSCLETests::createTestFactories() {
    QList<XMLTestFactory*> res;
    res.append(GTest_uMuscle::createFactory());
    res.append(GTest_CompareMAlignment::createFactory());
    res.append(GTest_uMuscleAddUnalignedSequenceToProfile::createFactory());
    return res;
}

}//namespace
