/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "MuscleTask.h"
#include "MuscleAdapter.h"
#include "TaskLocalStorage.h"

#include <core_api/AppContext.h>
#include <core_api/Log.h>
#include <core_api/StateLockableDataModel.h>
#include <core_api/DocumentModel.h>
#include <core_api/IOAdapter.h>
#include <gobjects/DNASequenceObject.h>
#include <util_tasks/LoadDocumentTask.h>

#include "muscle/muscle.h" 
#include "muscle/muscle_context.h" 


namespace GB2 {

static LogCategory log(ULOG_CAT_MUSCLE);


void MuscleTaskSettings::reset() {
    op = MuscleTaskOp_Align;
    maxIterations = 8;
    stableMode = true;
    regionToAlign.startPos = regionToAlign.len = 0;
    profile.clear();
    alignRegion = false;
}

MuscleTask::MuscleTask(const MAlignment& ma, const MuscleTaskSettings& _config) 
: Task(tr("MUSCLE alignment"), TaskFlags_FAIL_OSCOE | TaskFlag_DeleteWhenFinished), config(_config), inputMA(ma)
{
    tpm = Task::Progress_Manual;
}

void MuscleTask::run() {
    TaskLocalData::initializeMuscleTLSContext();
    MuscleContext* ctx = getMuscleContext();
    
    ctx->params.g_bStable = config.stableMode;
    ctx->params.g_uMaxIters = config.maxIterations;

    assert(!hasErrors());
    
    switch(config.op) {
        case MuscleTaskOp_Align:
            doAlign(false); 
            break;
        case MuscleTaskOp_Refine: 
            doAlign(true); 
            break;
        case MuscleTaskOp_AddUnalignedToProfile: 
            doAddUnalignedToProfile();
            break;
        case MuscleTaskOp_ProfileToProfile: 
            doProfile2Profile();
            break;
    }
    if (!hasErrors() && !isCanceled()) {
        assert(resultMA.alphabet!=NULL);
        assert(resultMA.isNormalized());
    }
    TaskLocalData::freeMuscleTLSContext();
}

void MuscleTask::doAlign(bool refine) {
    if (config.alignRegion && config.regionToAlign.len != inputMA.getLength()) {
        assert(config.regionToAlign.len > 0);
        MAlignment inputSubMA = inputMA.subAlignment(config.regionToAlign.startPos, config.regionToAlign.len);
        MAlignment resultSubMA;
        if (refine) {
            MuscleAdapter::refine(inputSubMA, resultSubMA, stateInfo);
        } else {
            MuscleAdapter::align(inputSubMA, resultSubMA, stateInfo, config.regionToAlign.startPos == 0);
        }

        resultMA.alphabet = inputMA.alphabet;
        QByteArray emptySeq;
        for(int i=0, n = inputMA.getNumSequences(); i < n; i++) {
            const MAlignmentItem& item = inputMA.alignedSeqs[i];
            resultMA.alignedSeqs.append(MAlignmentItem(item.name, emptySeq));
        }
        if (config.regionToAlign.startPos != 0) {
            resultMA += inputMA.subAlignment(0, config.regionToAlign.startPos);
        }
        resultMA +=resultSubMA;
        if (config.regionToAlign.endPos() != inputMA.getLength()) {
            resultMA += inputMA.subAlignment(config.regionToAlign.endPos(), inputMA.getLength() - config.regionToAlign.endPos());
        }
        //todo: check if there are GAP columns on borders and remove them
    } else {
        if (refine) {
            MuscleAdapter::refine(inputMA, resultMA, stateInfo);
        } else {
            MuscleAdapter::align(inputMA, resultMA, stateInfo, true);
        }
    }
}

void MuscleTask::doAddUnalignedToProfile() {
    MuscleAdapter::addUnalignedSequencesToProfile(inputMA, config.profile, resultMA, stateInfo);
}

void MuscleTask::doProfile2Profile() {
    MuscleAdapter::align2Profiles(inputMA, config.profile, resultMA, stateInfo);
}


//////////////////////////////////////////////////////////////////////////
// MuscleAddSequencesToProfileTask

MuscleAddSequencesToProfileTask::MuscleAddSequencesToProfileTask(MAlignmentObject* _obj, const QString& fileWithSequencesOrProfile, MMode _mode) 
: Task("", TaskFlags_NR_DWF_SSSOR), maObj(_obj), mode(_mode)
{
    setUseDescriptionFromSubtask(true);
    setVerboseLogMode(true);
    
    QString aliName = maObj->getDocument()->getName();
    QString fileName = QFileInfo(fileWithSequencesOrProfile).fileName();
    QString tn;
    if (mode == Profile2Profile) {
        tn = tr("MUSCLE align profiles '%1' vs '%2'").arg(aliName).arg(fileName);
    } else {
        tn = tr("MUSCLE align '%2' by profile '%1'").arg(aliName).arg(fileName);
    }
    setTaskName(tn);


    //todo: create 'detect file format task'
    DocumentFormatConstraints c;
    c.checkRawData = true;
    c.supportedObjectTypes.append(GObjectTypes::MULTIPLE_ALIGNMENT); //MA here comes first because for a sequence format raw sequence can be used by default
    c.rawData = BaseIOAdapters::readFileHeader(fileWithSequencesOrProfile);
    QList<DocumentFormatId> formats = AppContext::getDocumentFormatRegistry()->selectFormats(c);
    if (formats.isEmpty()) {
        c.supportedObjectTypes.clear();
        c.supportedObjectTypes.append(GObjectTypes::DNA_SEQUENCE);
        formats = AppContext::getDocumentFormatRegistry()->selectFormats(c);
        if (formats.isEmpty()) {
            stateInfo.error = tr("input_format_error");
            return;
        }
    }
    DocumentFormatId format = formats.first();
    IOAdapterFactory* iof = AppContext::getIOAdapterRegistry()->getIOAdapterFactoryById(BaseIOAdapters::url2io(fileWithSequencesOrProfile));
    loadTask = new LoadDocumentTask(format, fileWithSequencesOrProfile, iof);
    loadTask->setSubtaskProgressWeight(0.01f);
    addSubTask(loadTask);
}

QList<Task*> MuscleAddSequencesToProfileTask::onSubTaskFinished(Task* subTask) {
    QList<Task*> res;

    if (subTask != loadTask) {
        return res;
    }

    propagateSubtaskError();
    if (hasErrors()) {
        return res;
    }

    MuscleTaskSettings s;
    s.op = mode == Sequences2Profile ? MuscleTaskOp_AddUnalignedToProfile : MuscleTaskOp_ProfileToProfile;

    QList<GObject*> seqObjects = loadTask->getDocument()->findGObjectByType(GObjectTypes::DNA_SEQUENCE);
    //todo: move to utility alphabet reduction
    DNAAlphabet* al = NULL;
    foreach(GObject* obj, seqObjects) {
        DNASequenceObject* dnaObj = qobject_cast<DNASequenceObject*>(obj);
        DNAAlphabet* objAl = dnaObj->getAlphabet();
        if (al == NULL) {
            al = objAl;
        } else if (al != objAl) {
            al = DNAAlphabet::deriveCommonAlphabet(al, objAl);
            if (al == NULL) {
                stateInfo.error = tr("Sequences in file have different alphabets %1").arg(loadTask->getDocument()->getURL());
                return res;
            }
        }
        s.profile.alignedSeqs.append(MAlignmentItem(dnaObj->getGObjectName(), dnaObj->getSequence()));
    }
    s.profile.alphabet = al;

    if (seqObjects.isEmpty()) {
        QList<GObject*> maObjects = loadTask->getDocument()->findGObjectByType(GObjectTypes::MULTIPLE_ALIGNMENT);
        if (!maObjects.isEmpty()) {
            MAlignmentObject* maObj = qobject_cast<MAlignmentObject*>(maObjects.first());
            s.profile = maObj->getMAlignment();
        }
    }
    
    if (s.profile.isEmpty()) {
        if (mode == Sequences2Profile) {
            stateInfo.error = tr("No sequences found in file %1").arg(loadTask->getDocument()->getURL());
        } else {
            stateInfo.error = tr("No alignment found in file %1").arg(loadTask->getDocument()->getURL());
        }
        return res;
    }

    res.append(new MuscleGObjectTask(maObj, s));
    return res;
}

Task::ReportResult MuscleAddSequencesToProfileTask::report() {
    if (!hasErrors()) {
        propagateSubtaskError();
    }
    return ReportResult_Finished;
}

//////////////////////////////////////////////////////////////////////////
// MuscleGObjectTask

MuscleGObjectTask::MuscleGObjectTask(MAlignmentObject* _obj, const MuscleTaskSettings& _config) 
: Task("", TaskFlags_NR_DWF | TaskFlags_FAIL_OSCOE), obj(_obj), lock(NULL), muscleTask(NULL), config(_config)
{
    QString aliName = obj->getDocument()->getName();
    QString tn;
    switch(config.op) {
        case MuscleTaskOp_Align:
            tn = tr("MUSCLE align '%1'").arg(aliName);
            break;
        case MuscleTaskOp_Refine: 
            tn = tr("MUSCLE refine '%1'").arg(aliName);
            break;
        case MuscleTaskOp_AddUnalignedToProfile: 
            tn = tr("MUSCLE add to profile '%1'").arg(aliName);
            break;
        case MuscleTaskOp_ProfileToProfile: 
            tn = tr("MUSCLE align profiles");
            break;
        default: assert(0);
    }
    setTaskName(tn);
    setUseDescriptionFromSubtask(true);
    setVerboseLogMode(true);
}

MuscleGObjectTask::~MuscleGObjectTask() {
    assert(lock == NULL);
}

void MuscleGObjectTask::prepare() {
    if (obj.isNull()) {
        stateInfo.error = tr("object_removed");
        return;
    }
    if (obj->isStateLocked()) {
        stateInfo.error = tr("object_is_state_locked");
        return;
    }

    lock = new StateLock("muscle_lock");
    obj->lockState(lock);
    muscleTask = new MuscleTask(obj->getMAlignment(), config);

    addSubTask(muscleTask);
}

Task::ReportResult MuscleGObjectTask::report() {
    if (lock!=NULL) {
        obj->unlockState(lock);
        delete lock;
        lock = NULL;
    }
    propagateSubtaskError();
    if (hasErrors() || isCanceled()) {
        return ReportResult_Finished;
    }
    assert(!obj.isNull());
    if (obj->isStateLocked()) {
        stateInfo.error = tr("object_is_state_locked");
        return ReportResult_Finished;
    }
    obj->setMAlignment(muscleTask->resultMA);    

    return ReportResult_Finished;
}

} //namespace
