/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "BaseDocWorker.h"

#include "CoreLib.h"
#include <workflow/WorkflowEnv.h>
#include <workflow_support/WorkflowUtils.h>
#include <util_tasks/LoadDocumentTask.h>
#include <util_tasks/SaveDocumentTask.h>

#include <core_api/AppContext.h>
#include <core_api/ProjectModel.h>
#include <core_api/Log.h>
#include <core_api/DocumentModel.h>
#include <core_api/IOAdapter.h>
#include <util_gui/DialogUtils.h>

namespace GB2 {
namespace LocalWorkflow {

static LogCategory log(ULOG_CAT_WD);

LocalDocReader::LocalDocReader(Actor* a, const QString& tid, const DocumentFormatId& fid) : BaseWorker(a), ch(NULL), fid(fid), 
done(false), attachDoc2Proj(false) {
    mtype = WorkflowEnv::getDataTypeRegistry()->getById(tid);
}

void LocalDocReader::init() {
    QStringList urls = DesignerUtils::expandToUrls(actor->getParameter("URL")->value.toString());
    Project* p = AppContext::getProject();
    foreach(QString url, urls) {
        Document* doc = NULL;
        bool newDoc = true;
        if (p) {
            doc = p->findDocumentByURL(url);
            if (doc && doc->getDocumentFormatId() == fid) {
                newDoc = false;
            } else {
                doc = NULL;
            }
        }
        if (!doc) {
            DocumentFormat* format = AppContext::getDocumentFormatRegistry()->getFormatById(fid);
            assert(format);
            IOAdapterFactory* iof = AppContext::getIOAdapterRegistry()->getIOAdapterFactoryById(BaseIOAdapters::url2io(url));
            doc = new Document(format, iof, url, QVariantMap());
        }
        //TODO lock document???
        docs.insert(doc, newDoc);
    }

    assert(ports.size() == 1);
    ch = ports.values().first();
}

bool LocalDocReader::isReady() {
    return !isDone();
}

Task* LocalDocReader::tick() {
    if (!docs.isEmpty()) {
        Document* doc = docs.begin().key();
        if (!doc->isLoaded()) {
            return new LoadUnloadedDocumentTask(doc);
        } else {
            doc2data(doc);
            while (!cache.isEmpty()) {
                ch->put(cache.takeFirst());
            }
            if (docs.take(doc)) {
                doc->unload();
                delete doc;
            }
        }
    } 
    if (docs.isEmpty()) {
        done = true;
        ch->setEnded();
    }
    return NULL;
}

bool LocalDocReader::isDone() {
    return done && cache.isEmpty();
}

void LocalDocReader::cleanup() {
    QMapIterator<Document*, bool> it(docs);
    while (it.hasNext())
    {
        it.next();
        if (it.value()) {
            if (it.key()->isLoaded()) {
                it.key()->unload();
            }
            delete it.key();
        }
    }
}

bool LocalDocWriter::isDone() {
    return done;
}

void LocalDocWriter::cleanup() {
}

void LocalDocWriter::init() {
    url = actor->getParameter(CoreLib::URL_ATTR_ID)->value.toString();
    Attribute* a = actor->getParameter(CoreLib::APPEND_ATTR_ID);
    if (a) {
        append = a->value.toBool();
    }
    assert(ports.size() == 1);
    ch = ports.values().first();
}

bool LocalDocWriter::isReady() {
    return ch->hasMessage();
}

Task* LocalDocWriter::tick() {
    Document* doc = NULL;
    QString anUrl = url;
    DocumentFormat* format = AppContext::getDocumentFormatRegistry()->getFormatById(fid);
    while (ch->hasMessage()) {
        QVariantMap data = ch->get().getData().toMap();
        if (anUrl.isEmpty()) {
            anUrl = data.value(CoreLib::URL_SLOT_ID).toString();
        }
        if (anUrl.isEmpty()) {
            //TODO
        }
        doc = docs.value(anUrl);
        if (!doc) {
            IOAdapterFactory* iof = AppContext::getIOAdapterRegistry()->getIOAdapterFactoryById(BaseIOAdapters::url2io(url));
            doc = new Document(format, iof, anUrl, QVariantMap());
            doc->setLoaded(true);
            if (append) {
                docs.insert(anUrl, doc);
            }
        }
        data2doc(doc, data);
        if (!append) {
            break;
        }
    }
    done = ch->isEnded();
    if (append && !ch->isEnded()) {
        return NULL;
    }
    assert(doc && !anUrl.isEmpty());

    int count = ++counter[anUrl];
    if (!append) {
        anUrl = DialogUtils::prepareFileName(anUrl, count, format->getSupportedDocumentFileExtensions());
    } else {
        assert(count == 1);
        anUrl = DialogUtils::ensureFileExt(anUrl, format->getSupportedDocumentFileExtensions());
    }
    doc->setURL(anUrl);
    log.info(tr("Writing to %1 [%2]").arg(anUrl).arg(format->getFormatName()));
    return new SaveDocumentTask(doc, true);
}

} // Workflow namespace
} // GB2 namespace
