/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "WorkflowDocument.h"
#include "WorkflowViewController.h"
#include "SceneSerializer.h"

#include <core_api/AppContext.h>
#include <core_api/MainWindow.h>
#include <core_api/IOAdapter.h>
#include <core_api/Task.h>
#include <selection/SelectionUtils.h>
#include <workflow_support/SchemaSerializer.h>
#include <util_gui/GUIUtils.h>
#include <QtCore/QFile>
#include <QtXml/qdom.h>
#include <memory>

/* TRANSLATOR GB2::IOAdapter */

namespace GB2 {

const GObjectType WorkflowGObject::TYPE("workflow-obj");
const GObjectViewFactoryId WorkflowViewFactory::ID("workflow-view-factory");

bool WorkflowGObject::isTreeItemModified () const {
    if (view) {
        return view->getScene()->isModified();
    }
    return GObject::isItemModified();
}

void WorkflowGObject::setView(WorkflowView* _view) {
    view = _view;
}

void WorkflowGObject::setXML(QDomDocument _xml) {
    assert(view);
    assert(!view->getScene()->isModified());
    xml = _xml;
}

GObject* WorkflowGObject::clone() const {
    WorkflowGObject* copy = new WorkflowGObject(getGObjectName(), xml, getGHintsMap());
    assert(!view);
    return copy;
}

Document* WorkflowDocFormat::createNewDocument(IOAdapterFactory* io, const QString& url, const QVariantMap& fs) {
    Document* d = DocumentFormat::createNewDocument(io, url, fs);
    GObject* o = new WorkflowGObject(tr("Workflow Schema"), QDomDocument());
    d->addObject(o);
    return d;
}

Document* WorkflowDocFormat::loadExistingDocument(IOAdapterFactory* iof, const QString& url, TaskStateInfo& ti, const QVariantMap& fs) {
    assert(iof->getAdapterId() == BaseIOAdapters::LOCAL_FILE); //FIXME as utility
    QFile f(url);
    if (!f.open(QIODevice::ReadOnly)) {
        ti.error = Translations::errorOpeningFileRead(url);
        return NULL;
    }
    QByteArray  xmlData = f.readAll();
    f.close();

    QDomDocument xml;
    bool res = xml.setContent(xmlData);
    if (!res || xml.doctype().name() != Workflow::SchemaSerializer::WORKFLOW_DOC) {
        ti.error = tr("Invalid content: %1").arg(url);
        xml.clear();
    }

    //todo: check file-readonly status?

    GObject* o = new WorkflowGObject(tr("Workflow Schema"), xml);
    QList<GObject*> objects;
    objects.append(o);
    Document* d = new Document(this, iof, url, objects, fs);
    return d;
}

void WorkflowDocFormat::storeDocument(Document* d, TaskStateInfo& ti, IOAdapterFactory* iof, const QString& newDocURL) {
    assert(d->getDocumentFormat() == this);
    assert(d->getObjects().size() ==1);

    if (iof == NULL) {
        iof = d->getIOAdapterFactory();
    }
    assert(iof);
    std::auto_ptr<IOAdapter> io(iof->createIOAdapter());
    WorkflowGObject* wo = qobject_cast<WorkflowGObject*>(d->getObjects().first());
    assert(wo && wo->getView());

    QDomDocument xmlDoc(SchemaSerializer::WORKFLOW_DOC);
    SceneSerializer::scene2xml(wo->getView()->getScene(), xmlDoc);
    Metadata metadata = (wo->getView()->getMeta());
    QDomElement elem = xmlDoc.documentElement();
    SchemaSerializer::saveMeta((const Metadata* ) &metadata , elem);
    QByteArray rawData = xmlDoc.toByteArray();

    QString url = newDocURL.isEmpty() ? d->getURL() : newDocURL;
    if (!io->open(url, IOAdapterMode_Write)) {
        ti.error = ti.error = Translations::errorOpeningFileWrite(url);
        return;
    }
    int nWritten = 0;
    int nTotal = rawData.size();
    while(nWritten < nTotal) {
        int d = io->writeBlock(rawData.data() + nWritten, nTotal - nWritten);
        assert(d > 0);
        nWritten+= d;
    }
    wo->getView()->getScene()->setModified(false);
    wo->setXML(xmlDoc);
}


bool WorkflowDocFormat::isDataFormatSupported(const char* data, int size) const {
    Q_UNUSED(size);
    return QString(data).startsWith("<!DOCTYPE GB2WORKFLOW>"); //TODO improve
}

bool WorkflowDocFormat::isObjectOpSupported(const Document* d , DocumentFormat::DocObjectOp op, GObjectType t) const{
    Q_UNUSED(d);
    Q_UNUSED(op);
    Q_UNUSED(t);
    return false;//(t == GObjectTypes::TEXT && (op == DocObjectOp_Remove || d->getObjects().isEmpty()));
}

bool WorkflowDocFormat::checkConstraints(const DocumentFormatConstraints& c) const {
    foreach (GObjectType t, c.supportedObjectTypes) {
        if (t!=WorkflowGObject::TYPE) {
            return false;
        }
    }
    if (c.checkRawData) {
        return isDataFormatSupported(c.rawData.constData(), c.rawData.size());
    }
    return true;
}

bool WorkflowViewFactory::canCreateView(const MultiGSelection& multiSelection) {
    foreach(GObject* go, SelectionUtils::findObjects(WorkflowGObject::TYPE, &multiSelection)) {
        if (!qobject_cast<WorkflowGObject*>(go)->getView()) {
            return true;
        }
    }
    return false;
}

Task* WorkflowViewFactory::createViewTask(const MultiGSelection& multiSelection, bool single) {
    QSet<Document*> documents = SelectionUtils::findDocumentsWithObjects(WorkflowGObject::TYPE, &multiSelection, true, true);
    if (documents.size() == 0) {
        return NULL;
    }
    Task* result = (single || documents.size() == 1) ? NULL : new Task(tr("Open multiple views"), TaskFlags_NR_DWF);
    foreach(Document* d, documents) {
        Task* t = new OpenWorkflowViewTask(d);
        if (result == NULL) {
            return t;
        } 
        result->addSubTask(t);
    }
    return result;
}	

OpenWorkflowViewTask::OpenWorkflowViewTask(Document* doc) 
: ObjectViewTask(WorkflowViewFactory::ID)
{
    if (!doc->isLoaded()) {
        documentsToLoad.append(doc);
    } else {
        foreach(GObject* go, doc->findGObjectByType(WorkflowGObject::TYPE)) {
            selectedObjects.append(go) ;
        }
        assert(!selectedObjects.isEmpty());
    }
}

void OpenWorkflowViewTask::open() {
    if (stateInfo.hasErrors()) {
        return;
    }
    if (!documentsToLoad.isEmpty()) {
        foreach(GObject* go, documentsToLoad.first()->findGObjectByType(WorkflowGObject::TYPE)) {
            selectedObjects.append(go) ;
        }
    }
    foreach(QPointer<GObject> po, selectedObjects) {
        WorkflowGObject* o = qobject_cast<WorkflowGObject*>(po);
        assert(o && !o->getView());
        WorkflowView* view = new WorkflowView(o);
        AppContext::getMainWindow()->getMDIManager()->addMDIWindow(view);
        AppContext::getMainWindow()->getMDIManager()->activateWindow(view);
    }
}

}//namespace
