/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include <QtCore/QtCore>
#include <QtCore/QThread>
#include <QtCore/QDebug>

#include <core_api/Log.h>
#include <core_api/AppContext.h>
#include <core_api/DNATranslation.h>
#include <core_api/DocumentModel.h>
#include <util_text/TextUtils.h>
#include <util_tasks/CreateAnnotationTask.h>
#include <datatype/AnnotationData.h>

#include "ScriptEngineContext.h"
#include "RemoteRequestTask.h"

namespace GB2
{
/* TRANSLATOR GB2::RemoteRequestTask */ 

static LogCategory log(ULOG_CAT_PLUGIN_REMOTE_REQUEST);


RemoteRequestSubtask::RemoteRequestSubtask(RemoteRequestTask* _t) 
: Task(tr("remote_request_query_task"), TaskFlag_DeleteWhenFinished), t(_t) 
{
}

RemoteRequestTask::RemoteRequestTask( Script * _script, int _maxrl, int _minrl, SendSelectionStrand _strand, 
                                     DNATranslation * _complT, DNATranslation * _aminoT,  const QByteArray & _query, 
                                     int _qoffs, AnnotationTableObject* _ao, const QString & _group  ) :
Task( tr("remote_request_task"), TaskFlags_NR_DWF_SSSOR ), script( _script ), maxrl(_maxrl), minrl(_minrl),
strand(_strand), complT(_complT), aminoT(_aminoT), query(_query), offsInGlobalSeq(_qoffs), aobj(_ao), group(_group)
{
    queryTask = new RemoteRequestSubtask(this);
    addSubTask(queryTask);
}

QList<Task*> RemoteRequestTask::onSubTaskFinished(Task* subTask) {
    QList<Task*> res;
    if (aobj.isNull()) {
        stateInfo.error = tr("obj_was_removed\n");
        return res;
    }
    if (subTask == queryTask) {
        res.append(new CreateAnnotationsTask(aobj, group, resultAnnotations));
    }
    return res;
}

void RemoteRequestTask::prepareQueries() {
    bool dir = (strand == SendSelectionStrand_Both || strand == SendSelectionStrand_Direct);
    bool complement = (strand == SendSelectionStrand_Both || strand == SendSelectionStrand_Complement) && complT;
    if( complement ) {
        Query q;
        q.complement = true;

        QByteArray complQuery( query.size(), 0 );
        complT->translate( query.data(), query.size(), complQuery.data(), complQuery.size() );
        TextUtils::reverse( complQuery.data(), complQuery.size() );
        if( aminoT ) {
            q.amino = true;
            for( int i = 0; i < 3; ++i ) {
                QByteArray aminoQuery( query.size() / 3, 0 );
                aminoT->translate( complQuery.data() + i, complQuery.size()-i, aminoQuery.data(), aminoQuery.size() );
                q.seq = aminoQuery;
                q.offs = i;
                queries.push_back(q);
            }            
        } else {
            q.seq = complQuery;
            queries.push_back(q);
        }

    } 
    if( dir ) {
        Query q;
        if( aminoT ) {
            q.amino = true;
            for( int i = 0; i < 3; ++i ) {
                QByteArray aminoQuery( query.size() / 3, 0 );
                aminoT->translate( query.data()+i, query.size()-i, aminoQuery.data(), aminoQuery.size() );
                q.seq = aminoQuery;
                q.offs = i;
                queries.push_back(q);
            }
        } else {
            q.seq = query;
            queries.push_back(q);
        }
    } 
}

void RemoteRequestTask::prepareEngine() {
    engine = new QScriptEngine();
    script->init_engine( engine );
    ScriptHttpAnnotatorContext::setDefaultProperties( engine );
    ScriptHttpAnnotatorContext::setMaxResLen( engine, maxrl );
    ScriptHttpAnnotatorContext::setMinResLen( engine, minrl );
    ScriptHttpAnnotatorContext::setLog( engine, &log );
    ScriptHttpAnnotatorContext::setTaskStateInfo( engine, &stateInfo );
    script->callSetup(engine);
}

void RemoteRequestTask::_run() {
    prepareQueries();

    foreach( Query q, queries) {
        prepareEngine();
        ScriptHttpAnnotatorContext::setQuery( engine, QString(q.seq) );
        QScriptValue exception = engine->nullValue();
        script->callMain( engine, &exception );
        if( !exception.isNull() ) {
            QString exc_msg = exception.isError() ? exception.property("message").toString() : exception.toString();
            log.error( tr("script_reports_error: ") + exc_msg );
            QStringList backtrace = engine->uncaughtExceptionBacktrace();
            if( !backtrace.isEmpty() ) {
                log.details( tr("exception_backtrace:") );
                foreach( QString s, backtrace ) {
                    log.details(s);
                }
            }
            break;
        }
        createAnnotations(q);
    }
    delete engine;
}

void RemoteRequestTask::createAnnotations( const Query & q ) {
    QList<SharedAnnotationData> annotations = ScriptHttpAnnotatorContext::getAnnotations( engine );
    if ( annotations.isEmpty() ) {
        return;
    }
    for( QList<SharedAnnotationData>::iterator it = annotations.begin(), end = annotations.end(); end != it; ++it ) {
        for( QList<LRegion>::iterator jt = it->data()->location.begin(), eend = it->data()->location.end(); 
            eend != jt; ++jt ) {
            int & s = jt->startPos;
            int & l = jt->len;

            if( q.complement ) {
                s = q.seq.size() - s - l;
                it->data()->complement = !it->data()->complement;
            }
            if( q.amino ) {
                s = s * 3 + (q.complement ? 2 - q.offs : q.offs);
                l = l * 3;
            }
            s += offsInGlobalSeq;
        }
    }
    resultAnnotations << annotations;
}

} //namespace
