/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include <QtGui/QMouseEvent>
#include <QtGui/QMessageBox>
#include <QtGui/QColorDialog>
#include <QtGui/QFileDialog>
#include <QtGui/QImageWriter>
#include <QtCore/QTime>

#include <datatype/BioStruct3D.h>
#include <util_ov_annotated_dna/AnnotatedDNAView.h>
#include <util_ov_annotated_dna/ADVSequenceObjectContext.h>
#include <gobjects/AnnotationSettings.h>
#include <gobjects/AnnotationTableObject.h>
#include <gobjects/DNASequenceObject.h>
#include <gobjects/GObjectUtils.h>
#include <gobjects/GObjectRelationRoles.h>
#include <core_api/Log.h>
#include <core_api/AppContext.h>
#include <core_api/DocumentModel.h>
#include <selection/DNASequenceSelection.h>
#include <selection/AnnotationSelection.h>
#include <time.h>

#include "BioStruct3DGLWidget.h"
#include "BioStruct3DGLRender.h"
#include "BioStruct3DColorScheme.h"
#include "ExportImageDialog.h"


namespace GB2 { 

const float BioStruct3DGLWidget::DEFAULT_ZOOM = 45.0f;
static LogCategory log(ULOG_CAT_PLUGIN_BIOSTRUCT_3D);
int BioStruct3DGLWidget::widgetCount = 0;


BioStruct3DGLWidget::BioStruct3DGLWidget(const BioStruct3D&  struc, const Document* doc, const AnnotatedDNAView* view, QWidget *parent /* = 0*/)
    : QGLWidget(parent), biostruc(struc), biostrucDoc(doc), dnaView(view), currentModelID(struc.pdbId), spinAngle(0), displayMenu(NULL)
{
    
    setObjectName(struc.pdbId + "-" + QString("%1").arg(++widgetCount));
    //TODO: ? setFormat(QGLFormat(QGL::DoubleBuffer | QGL::DepthBuffer));
    yAxis.set(0,1,0);
    colorSchemeFactoryMap = BioStruct3DColorSchemeFactory::createFactories();
    rendererFactoryMap = BioStruct3DGLRendererFactory::createFactories();
    createActions();
    createMenus();
    connectExternalSignals();

    currentColorSchemeName =  BioStruct3DColorSchemeFactory::defaultFactoryName();
    currentGLRendererName =  BioStruct3DGLRendererFactory::defaultFactoryName();
    backgroundColor = Qt::black;

    chainIdCache.clear();
    
    // Load settings
    QList<QAction*>::iterator iter;
    QList<QAction*> schemeActions = colorSchemeActions->actions();
    for (iter = schemeActions.begin(); iter != schemeActions.end(); ++iter) {
        if ((*iter)->text() == currentColorSchemeName) {
            (*iter)->setChecked(true);
            break;
        }
    }
    Q_ASSERT(iter != schemeActions.end());
    colorScheme.reset(colorSchemeFactoryMap.value(currentColorSchemeName)->createInstance(this));
    
    QList<QAction*> renderActions = rendererActions->actions();
    for (iter = renderActions.begin(); iter != renderActions.end(); ++iter) {
        if ((*iter)->text() == currentGLRendererName) {
            (*iter)->setChecked(true);
            break;
        }
    }
    Q_ASSERT(iter != renderActions.end());
    const BioStruct3DColorScheme* scheme = colorScheme.get();  
    renderer.reset(rendererFactoryMap.value(currentGLRendererName)->createInstance(struc,scheme));
	
    // Init rotation matrix as identity
    rotMatrix.loadIdentity();
    // Set view settings
    float scaleFactor = 2.5;
    cameraDistance = scaleFactor * biostruc.getMaxDistFromCenter();
    cameraClipNear = (cameraDistance - biostruc.getMaxDistFromCenter()) * 0.66f;
    cameraClipFar = (cameraDistance + biostruc.getMaxDistFromCenter()) * 1.2f; 
    zoomFactor = DEFAULT_ZOOM;
    saveDefaultsSettings();
}

BioStruct3DGLWidget::~BioStruct3DGLWidget()
{
    // Clean up factories
    
    foreach(QString key, colorSchemeFactoryMap.keys()) {
         BioStruct3DColorSchemeFactory* f = colorSchemeFactoryMap.take(key);
         delete f;
    }
    foreach(QString key, rendererFactoryMap.keys()) {
        BioStruct3DGLRendererFactory* f = rendererFactoryMap.take(key);
        delete f;
    }
    
   log.trace("Biostruct3DGLWdiget "+objectName()+" deleted");
}



void BioStruct3DGLWidget::initializeGL()
{
    setLightPosition(Vector3D(0, 0.0, 1.0));
    GLfloat light_diffuse[] = { 0.8f, 0.8f, 0.8f, 1.0 };
    GLfloat light_specular[] = { 0.6f, 0.6f, 0.6f, 1.0 };
    GLfloat mat_specular[] = { 0.6f, 0.6f, 0.6f, 1.0 };
	GLfloat mat_shininess[] = { 90.0 };
    DisplayLists::createAtomDisplayList();
    qglClearColor(backgroundColor);
    glShadeModel (GL_SMOOTH);
    glMaterialfv(GL_FRONT, GL_SPECULAR, mat_specular);
    glMaterialfv(GL_FRONT, GL_SHININESS, mat_shininess);
    glLightfv(GL_LIGHT0, GL_DIFFUSE, light_diffuse);
    glLightfv(GL_LIGHT0, GL_SPECULAR, light_specular);
    glLightfv(GL_LIGHT0, GL_POSITION, lightPostion);
    //glEnable(GL_NORMALIZE);
    glEnable(GL_LIGHTING);
    glEnable(GL_LIGHT0);
    glEnable(GL_DEPTH_TEST);
    
}

void BioStruct3DGLWidget::resizeGL(int width, int height)
{
    glViewport(0, 0, width, height);
    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();
    // Set view settings
    GLfloat aspect = GLfloat(width) / height;
    gluPerspective(zoomFactor, aspect, cameraClipNear, cameraClipFar);

}

void BioStruct3DGLWidget::paintGL()
{
    if (!isVisible())
        return;
    glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
    draw();
}

void BioStruct3DGLWidget::mousePressEvent(QMouseEvent *event)
{
	lastPos = getTrackballMapping(event->x(), event->y());
}

void BioStruct3DGLWidget::mouseMoveEvent(QMouseEvent *event)
{
    Vector3D rotCenter = biostruc.getCenter();
	
	if (event->buttons() & Qt::LeftButton) {

		Vector3D curPos = getTrackballMapping(event->x(), event->y());
		
		Vector3D delta = curPos - lastPos;
		if (delta.x || delta.y || delta.z) {
			rotAngle = 90.0f*delta.length();
			rotAxis =  vector_cross(lastPos,curPos); 
			glMatrixMode(GL_MODELVIEW);
			glLoadIdentity();
            glRotatef(rotAngle, rotAxis.x, rotAxis.y, rotAxis.z );
            glMultMatrixf(rotMatrix.getData());
            glGetFloatv( GL_MODELVIEW_MATRIX, rotMatrix.getData());
        }
        
        lastPos = curPos;
		updateGL();
	}
    
	

}

void BioStruct3DGLWidget::draw()
{
    Vector3D rotCenter = biostruc.getCenter();
    glMatrixMode(GL_MODELVIEW);
	glLoadIdentity(); 
    gluLookAt(0.0, 0.0, cameraDistance, 0,  0, 0, 0.0, 1.0, 0.0);
    glMultMatrixf(rotMatrix.getData());
    glTranslatef(-rotCenter.x ,-rotCenter.y, -rotCenter.z);
    renderer->drawBioStruct3D();
    
}

Vector3D BioStruct3DGLWidget::getTrackballMapping(int x, int y)
{
    Vector3D pos;
	/* project x,y onto a hemisphere centered within width, height */
	pos.x = (2.0f*x - width()) / width();
	pos.y = (height() - 2.0f*y) / height();
	pos.z = 0;
	float d = pos.length();
	d = (d < 1.0) ? d : 1.0;
	pos.z = sqrtf(1.0f - d*d);
	pos.normalize();

	return pos;

} 


void BioStruct3DGLWidget::wheelEvent ( QWheelEvent * event )
{
	float numDegrees =  event->delta() / 8;
    zoom(numDegrees / 10);
}

void BioStruct3DGLWidget::createActions()
{

    QAction* action = NULL;
    
    animationTimer = new QTimer(this);
    animationTimer->setInterval(20); // fixed interval
    connect(animationTimer, SIGNAL(timeout()), this, SLOT(sl_updateAnnimation()));
    
    rendererActions = new QActionGroup(this);
    connect(rendererActions, SIGNAL(triggered(QAction *)), this, SLOT(sl_selectGLRenderer(QAction *)));
    foreach(QString key, rendererFactoryMap.keys()) {
        action  = new QAction(key, rendererActions);
        action->setCheckable(true);
    }

    colorSchemeActions = new QActionGroup(this);	
	connect(colorSchemeActions, SIGNAL(triggered(QAction *)), this, SLOT(sl_selectColorScheme(QAction *)));
    foreach(QString key, colorSchemeFactoryMap.keys()) {
        action = new QAction(key, colorSchemeActions);
        action->setCheckable(true);
    }   

    spinAction = new QAction(tr("Spin"), this);
    spinAction->setCheckable(true);
    connect(spinAction, SIGNAL(triggered()), this, SLOT(sl_acitvateSpin()));

    setBackgroundColorAction = new QAction(QIcon(":core/images/color_wheel.png"), tr("Set background color"), this);
    connect(setBackgroundColorAction, SIGNAL(triggered()), this, SLOT(sl_setBackgroundColor()));

    closeAction = new QAction(tr("close_action"), this);
    connect(closeAction, SIGNAL(triggered()), this, SLOT(sl_closeWidget()));

    exportImageAction = new QAction(tr("Export image"), this);
    connect(exportImageAction, SIGNAL(triggered()), this, SLOT(sl_exportImage()));


   	
}

void BioStruct3DGLWidget::createMenus()
{
    selectRendererMenu = new QMenu(tr("Render style"));
    selectRendererMenu->addActions(rendererActions->actions());
    
    selectColorSchemeMenu = new QMenu(tr("Color Scheme"));
	selectColorSchemeMenu->addActions(colorSchemeActions->actions());


}

void BioStruct3DGLWidget::connectExternalSignals()
{
	AnnotationSettingsRegistry* asr = AppContext::getAnnotationsSettingsRegistry();
	connect(asr, SIGNAL(si_annotationSettingsChanged(const QStringList& )), this, SLOT(sl_updateRenderSettings(const QStringList& )) );

    const QList<ADVSequenceObjectContext*> seqContexts = getSequenceContexts();
    foreach (ADVSequenceObjectContext* ctx, seqContexts) {
    connect(ctx->getSequenceSelection(), 
            SIGNAL(si_selectionChanged(LRegionsSelection*, const QList<LRegion>&, const QList<LRegion>&)), 
            SLOT(sl_onSequenceSelectionChanged(LRegionsSelection*, const QList<LRegion>& , const QList<LRegion>&)));
    
    connect(dnaView->getAnnotationsSelection(), 
        SIGNAL(si_selectionChanged(AnnotationSelection* , const QList<Annotation*>&, const QList<Annotation*>&)),
        SLOT(sl_onAnnotationSelectionChanged(AnnotationSelection* , const QList<Annotation*>&, const QList<Annotation*>&)));
    
    
    }


}



void BioStruct3DGLWidget::setBioStruct3DColorScheme( BioStruct3DColorScheme* clScheme )
{
	Q_ASSERT(clScheme != NULL);
	renderer->updateColorScheme(clScheme);
    colorScheme.reset(clScheme);
   
}

void BioStruct3DGLWidget::setBioStruct3DRenderer( BioStruct3DGLRenderer* r )
{
	Q_ASSERT(r != NULL);
	renderer.reset(r);

}

void BioStruct3DGLWidget::contextMenuEvent( QContextMenuEvent *event )
{
	QMenu menu;
    menu.addMenu(selectRendererMenu);
    menu.addMenu(selectColorSchemeMenu);
    menu.addAction(spinAction);
    menu.addAction(setBackgroundColorAction);
    menu.addAction(exportImageAction);
    menu.addAction(closeAction);

	menu.exec(event->globalPos());
}

void BioStruct3DGLWidget::sl_selectColorScheme(QAction* action)
{
	BioStruct3DColorScheme* colorScheme = NULL;
	QString schemeName = action->text();
    colorScheme = createCustomColorScheme(schemeName);
	Q_ASSERT(colorScheme != NULL);
	setBioStruct3DColorScheme(colorScheme);
    currentColorSchemeName = schemeName;
 	updateGL();
		
}

void BioStruct3DGLWidget::sl_updateRenderSettings(const QStringList& list )
{
    Q_UNUSED(list);
    sl_selectColorScheme(colorSchemeActions->checkedAction());
}


QMap<QString, QColor> BioStruct3DGLWidget::getSecStructAnnotationColors() const
{
    QMap<QString, QColor> colors;
    AnnotationSettingsRegistry* asr = AppContext::getAnnotationsSettingsRegistry();

    foreach (GObject* obj, biostrucDoc->getObjects() ) {
        if (obj->getGObjectType() == GObjectTypes::ANNOTATION_TABLE) {
                AnnotationTableObject* ao = qobject_cast<AnnotationTableObject*>(obj);
                foreach(Annotation* a, ao->getAnnotations()) {
                        QString name = a->getAnnotationName();
                        if ( (name == BioStruct3D::AlphaHelixAnnotationTag) || (name == BioStruct3D::BetaStrandAnnotationTag)
                                || (name == BioStruct3D::TurnAnnotationTag) ) {
                const AnnotationSettings* as = asr->getSettings(name);
                colors.insert(name, as->color);
				}
			}
        }
    }

    return colors;
}



const QMap<int, QColor> BioStruct3DGLWidget::getChainColors() const
{
    QMap<int, QColor> colorMap;
    AnnotationSettingsRegistry* asr = AppContext::getAnnotationsSettingsRegistry();
    
	foreach (GObject* obj, biostrucDoc->getObjects() ) {
		if (obj->getGObjectType() == GObjectTypes::ANNOTATION_TABLE) {
			AnnotationTableObject* ao = qobject_cast<AnnotationTableObject*>(obj);
 			foreach(Annotation* a, ao->getAnnotations()) {
 				QString name = a->getAnnotationName();
 				if (name.startsWith(BioStruct3D::MoleculeAnnotationTag)) {
					int index = getQualifierValueByName(a, BioStruct3D::ChainIdQualifierName).toInt();
					Q_ASSERT(index != 0);
                const AnnotationSettings* as = asr->getSettings(name);
                colorMap.insert(index, as->color);				
 				}
 			}
		}
 	}
	
    return colorMap;
}

void BioStruct3DGLWidget::sl_selectGLRenderer( QAction* action )
{
    QString rendererName = action->text();
    BioStruct3DGLRenderer* renderer = createCustomRenderer(rendererName);
    Q_ASSERT(renderer != NULL);
    setBioStruct3DRenderer(renderer);
    currentGLRendererName = rendererName;
    updateGL();
}

void BioStruct3DGLWidget::setLightPosition( const Vector3D& pos )
{
    lightPostion[0] = pos.x;
    lightPostion[1] = pos.y;
    lightPostion[2] = pos.z;
    lightPostion[3] = 1.0;
}

BioStruct3DColorScheme* BioStruct3DGLWidget::createCustomColorScheme( const QString& name )
{
    if (colorSchemeFactoryMap.contains(name)) {
        return colorSchemeFactoryMap.value(name)->createInstance(this);    
    } else {
        return NULL;
    }

}

BioStruct3DGLRenderer* BioStruct3DGLWidget::createCustomRenderer( const QString& name )
{
    if (rendererFactoryMap.contains(name)) {
        return rendererFactoryMap.value(name)->createInstance(biostruc, colorScheme.get());    
    } else {
        return NULL;
    }
}


const QList<ADVSequenceObjectContext*> BioStruct3DGLWidget::getSequenceContexts() const
{
    return dnaView->getSequenceContexts();
}

void BioStruct3DGLWidget::sl_onSequenceSelectionChanged( LRegionsSelection* s, const QList<LRegion>&  added, const QList<LRegion>&  removed)
{
    if (!isVisible())
        return;

    DNASequenceSelection* selection = qobject_cast<DNASequenceSelection*>(s);
    const DNASequenceObject* seqObj = selection->getSequenceObject();
    if (selection) {
        const QString seqName = seqObj->getGObjectName();
        if (!seqName.startsWith(currentModelID))
			return;
        
        QList<GObject*> relAnns = GObjectUtils::findObjectsRelatedToObjectByRole(seqObj, GObjectTypes::ANNOTATION_TABLE, GObjectRelationRole::SEQUENCE, biostrucDoc->getObjects());
        AnnotationTableObject* a = relAnns.isEmpty() ? NULL : qobject_cast<AnnotationTableObject*>(relAnns.first());
        if (a == NULL) {
            log.error("annotation object not found!");
            return;
        }
        int chainId = getChainIdFromAnnotationObject(a);
        Q_ASSERT(biostruc.moleculeMap.contains(chainId));
        colorScheme->updateSelectionRegion(chainId, added, removed);
        //TODO: unobvious! Refactor interface 
        renderer->updateColorScheme(colorScheme.get());
        update();
    }


}


int BioStruct3DGLWidget::getChainIdFromAnnotationObject(const AnnotationTableObject* ao) 
{
    if ( this->chainIdCache.contains(ao) ) {
        return chainIdCache.value(ao);
    } else {
        foreach (Annotation* a, ao->getAnnotations()) {
            if (a->getAnnotationName().startsWith(BioStruct3D::MoleculeAnnotationTag) ) {
                int chainId = getQualifierValueByName(a,BioStruct3D::ChainIdQualifierName).toInt();
                chainIdCache.insert(ao,chainId);
                return chainId;
            } 
        }
         return 0;
    }
}

void BioStruct3DGLWidget::sl_onAnnotationSelectionChanged( AnnotationSelection* as, const QList<Annotation*>& added, const QList<Annotation*>& removed )
{
    Q_UNUSED(as);
    if (!isVisible())
        return;
    
    QList<LRegion> empty;

	foreach (Annotation* annotation, added) {
		if (annotation->getLocation().isEmpty())
            continue;
		AnnotationTableObject* ao = annotation->getGObject(); 
		if (!ao->getGObjectName().startsWith(currentModelID))
			continue;
		int chainId  = getChainIdFromAnnotationObject(ao);
		Q_ASSERT(biostruc.moleculeMap.contains(chainId));
		colorScheme->updateSelectionRegion(chainId, annotation->getLocation(), empty );
    }
  
    foreach (Annotation* annotation, removed) {
       if (annotation->getLocation().isEmpty() )
           continue;
	   AnnotationTableObject* ao = annotation->getGObject(); 
	   if (!ao->getGObjectName().startsWith(currentModelID))
		   continue;
	   int chainId  = getChainIdFromAnnotationObject(ao);
       Q_ASSERT(biostruc.moleculeMap.contains(chainId));
       colorScheme->updateSelectionRegion(chainId, empty, annotation->getLocation());
    }
    
    renderer->updateColorScheme(colorScheme.get());
    update();
}

QString BioStruct3DGLWidget::getQualifierValueByName( const Annotation* annotation, const QString& qualifierName )
{
    foreach (Qualifier q, annotation->getQualifiers()) {
        if (q.getQualifierName() == qualifierName) {
            return QString(q.getQualifierValue());
        }
    }
    return QString("");
}

void BioStruct3DGLWidget::sl_acitvateSpin()
{
    if (spinAction->isChecked()) { 
        animationTimer->start();
    } else {
        animationTimer->stop();
    }

    updateGL();

}

void BioStruct3DGLWidget::sl_updateAnnimation()
{
    static float velocity = 0.05f;
    spinAngle = velocity* animationTimer->interval();
    glMatrixMode(GL_MODELVIEW);
    glLoadIdentity();
    glRotatef(spinAngle, 0, 1, 0 );
    glMultMatrixf(rotMatrix.getData());
    glGetFloatv( GL_MODELVIEW_MATRIX, rotMatrix.getData());
    updateGL();
}

#define ZOOM_FACTOR_ID "ZOOM_FACTOR"
#define ROTATION_MATRIX_ID "ROTATION_MATRIX"
#define PRODUCT_NAME "Unipro Ugene"
#define PLUGIN_NAME "BioStruct3D Viewer Plugin"
#define COLOR_SCHEME_NAME "ColorScheme"
#define RENDERER_NAME "GLRenderer"
#define OBJECT_ID_NAME "OBJECT_ID"


QVariantMap BioStruct3DGLWidget::getState()
{
    QVariantMap state;
    state[ZOOM_FACTOR_ID] = QVariant::fromValue(zoomFactor);
    state[ROTATION_MATRIX_ID] = rotMatrix.store();
    state[COLOR_SCHEME_NAME] = QVariant::fromValue(currentColorSchemeName);
    state[RENDERER_NAME] = QVariant::fromValue(currentGLRendererName);
    state[OBJECT_ID_NAME] = QVariant::fromValue(getBioStruct3DObjectName());

    return state;

}

void BioStruct3DGLWidget::setState( const QVariantMap& state )
{
    // TODO: draw current selection correctly
    if (state.isEmpty()) {
        return;
    }
    zoomFactor = state.value(ZOOM_FACTOR_ID, DEFAULT_ZOOM).value<float>();
    QVariantList rotML = state.value(ROTATION_MATRIX_ID).value<QVariantList>();
    if (!rotML.isEmpty()) {
        rotMatrix.load(rotML);
    }
    
    currentColorSchemeName = state.value(COLOR_SCHEME_NAME, BioStruct3DColorSchemeFactory::defaultFactoryName()).value<QString>();
    currentGLRendererName = state.value(RENDERER_NAME, BioStruct3DGLRendererFactory::defaultFactoryName()).value<QString>();
    
    setBioStruct3DColorScheme(createCustomColorScheme(currentColorSchemeName));
    setBioStruct3DRenderer(createCustomRenderer(currentGLRendererName));
    resizeGL(width(), height());
    updateGL();
}

const QString BioStruct3DGLWidget::getBioStruct3DObjectName() const
{
    foreach (GObject* obj, biostrucDoc->getObjects()) {
        if (obj->getGObjectType() == GObjectTypes::BIOSTRUCTURE_3D)
            return obj->getGObjectName();
    }
    Q_ASSERT(0);
    return ""; 
}

void BioStruct3DGLWidget::sl_setBackgroundColor()
{
    backgroundColor = QColorDialog::getColor();
    qglClearColor(backgroundColor);
    updateGL();

}

void BioStruct3DGLWidget::sl_closeWidget()
{
    hide();
    emit si_widgetClosed(this);
}

void BioStruct3DGLWidget::sl_exportImage() 
{
    
    ExportImageDialog dialog(this);
    dialog.exec();

//     QPixmap image = renderPixmap();
//         
//      QList<QByteArray> suppotedFormats = QImageWriter::supportedImageFormats();
//     
//     QString fileName = QFileDialog::getSaveFileName(this, tr("Export Image File"),
//         "", tr("PNG images (*.png );; JPEG images (*.jpg) "));
// 
//     QByteArray fileFormat("PNG");
//     image.save(fileName, fileFormat.constData());

}

void BioStruct3DGLWidget::zoom( float delta )
{
    const float maxZoom = 150.0;
    const float minZoom = 2.0;
    zoomFactor += delta;
    if (zoomFactor < minZoom) {
        zoomFactor = minZoom;
        return;
    }

    if (zoomFactor > maxZoom) {
        zoomFactor = maxZoom;
        return;
    }


    resizeGL(width(), height());
    updateGL();


}

void BioStruct3DGLWidget::saveDefaultsSettings()
{
    defaultsSettings[ZOOM_FACTOR_ID] = QVariant::fromValue(zoomFactor);
    defaultsSettings[ROTATION_MATRIX_ID] = rotMatrix.store();
    defaultsSettings[COLOR_SCHEME_NAME] = QVariant::fromValue(currentColorSchemeName);
    defaultsSettings[RENDERER_NAME] = QVariant::fromValue(currentGLRendererName);

}

void BioStruct3DGLWidget::restoreDefaultSettigns()
{
    setState(defaultsSettings);
}


QMenu* BioStruct3DGLWidget::getDisplayMenu()
{
    if (displayMenu == NULL) {
        displayMenu = new QMenu(this);
        displayMenu->addMenu(selectRendererMenu);
        displayMenu->addMenu(selectColorSchemeMenu);
        displayMenu->addAction(spinAction);
    }
    
    return displayMenu;
}


///////////////////////////////////////////////////////////////////////////////////////////
/// Matrix4x4

Matrix4x4::Matrix4x4()
{
	m[0] = 0; m[1] = 0; m[2] = 0; m[3] = 0;
	m[4] = 0; m[5] = 0; m[6] = 0; m[7] = 0;
	m[8] = 0; m[9] = 0; m[10] = 0; m[11] = 0;
	m[12] = 0; m[13] = 0; m[14] = 0; m[15] = 0;
}

void Matrix4x4::loadIdentity()
{
	m[0] = 1; m[1] = 0; m[2] = 0; m[3] = 0;
	m[4] = 0; m[5] = 1; m[6] = 0; m[7] = 0;
	m[8] = 0; m[9] = 0; m[10] = 1; m[11] = 0;
	m[12] = 0; m[13] = 0; m[14] = 0; m[15] = 1;
}

float& Matrix4x4::operator[]( unsigned int i )
{
	static float err = 0.0;
	if (i < 16)
		return m[i];
	else  {
		qCritical("Vector operator [] access out of range : %d", i);
		return err;
	}
}

Matrix4x4& Matrix4x4::operator=( const Matrix4x4& matrix )
{
    if (this == &matrix) { 
        return *this;
    }
    memcpy((void*) m, (void*) matrix.m, sizeof(matrix.m) ); 
    return *this;

}

void Matrix4x4::load( QVariantList values )
{
    Q_ASSERT(values.size() == 16);
    for (int i = 0; i < 16; ++i) {
        m[i] = values.at(i).value<float>(); 
    }

}

QVariantList Matrix4x4::store()
{
    QVariantList values;
    for (int i = 0; i < 16; ++i ) {
        values.append(QVariant::fromValue(m[i]));
    }
    return values;
}
} //namespace
