/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "TypeMapEditor.h"

#include <assert.h>
#include <QtGui/QListWidget>
#include <QtGui/QTableWidget>
#include <QtGui/QHeaderView>
#include <QtGui/QLabel>
#include <QtGui/QComboBox>
#include <QtGui/QVBoxLayout>
#include <QtGui/QTextEdit>
#include <QtGui/QStandardItemModel>
#include <workflow/IntegralBusModel.h>
#include <workflow/IntegralBusType.h>
#include <workflow_support/WorkflowUtils.h>

namespace GB2 {

TypeMapEditor::TypeMapEditor(Configuration* cfg, const QString& prop, 
    DataTypePtr from, DataTypePtr to) : cfg(cfg), propertyName(prop), from(from), to(to), table(NULL) 
{}

QWidget* TypeMapEditor::getWidget() {
    return createGUI(from, to);
}

static QList<Descriptor> filter(DataTypePtr set, DataTypePtr el) {
    QList<Descriptor> result;
    foreach(const Descriptor& d, set->getElements()) {
        if (set->getElement(d) == el) {
            result.append(d);
        }
    }
    return result;
}

#define KEY_COL 0
#define VAL_COL 1

QWidget* TypeMapEditor::createGUI(DataTypePtr from, DataTypePtr to) {
    /*if (from == to) {
        QLabel* label = new QLabel(trs("Identity mapping"));
        label->setAlignment(Qt::AlignHCenter);
        return label;
    }*/
    if (from && to && from->isMap() && to->isMap()) {
        bool infoMode = (to == from);
        if (infoMode) {
            table = new QTableWidget(0,1);
            table->setHorizontalHeaderLabels((QStringList() << tr("Data on the bus")));
        } else {
            table = new QTableWidget(0,2);
            table->setHorizontalHeaderLabels((QStringList() << tr("Accepted inputs") << tr("Supplied data") ));
            table->setItemDelegate(new DescriptorListDelegate(this));
        }
        table->horizontalHeader()->setResizeMode(QHeaderView::Interactive);
        table->horizontalHeader()->setStretchLastSection(true);
        table->verticalHeader()->hide();
        QSizePolicy sizePolicy5(QSizePolicy::Expanding, QSizePolicy::Expanding);
        sizePolicy5.setHorizontalStretch(0);
        sizePolicy5.setVerticalStretch(2);
        sizePolicy5.setHeightForWidth(table->sizePolicy().hasHeightForWidth());
        table->setSizePolicy(sizePolicy5);
        table->setBaseSize(QSize(0, 100));
        table->setEditTriggers(QAbstractItemView::CurrentChanged|QAbstractItemView::DoubleClicked|QAbstractItemView::EditKeyPressed|QAbstractItemView::SelectedClicked);
        table->setAlternatingRowColors(true);
        table->setSelectionMode(QAbstractItemView::SingleSelection);
        table->setTextElideMode(Qt::ElideMiddle);
        table->setShowGrid(false);
        //table->setWordWrap(false);
        table->setCornerButtonEnabled(false);
        int height = QFontMetrics(QFont()).height() + 6;


        const QList<Descriptor>& keys = to->getElements();
        table->setRowCount(keys.size());
        
        QMap<QString,QString> map = cfg->getParameter(propertyName)->value.value<QStrStrMap>();
        const int col = infoMode ? 0 : KEY_COL;

        for(int i = 0; i < keys.size(); i++) {     
            Descriptor key = keys.at(i);
            QTableWidgetItem* label = new QTableWidgetItem(key.getDisplayName());
            label->setToolTip(to->getElement(key)->getDisplayName());
            label->setData(Qt::UserRole, qVariantFromValue<Descriptor>(key));
            label->setFlags(Qt::ItemIsSelectable);
            table->setItem(i, col, label);
            table->setRowHeight(i, height);

            if (infoMode) continue;

            DataTypePtr eldt = to->getElement(key);
            QList<Descriptor> cands = filter(from, eldt);
            Descriptor current("", tr("<empty>"), tr("Default value"));
            if (eldt->isList()) {
                cands +=filter(from, eldt->getElement());
                QString currentVal = map.value(key.getId());
                if (!currentVal.isEmpty()) {
                    current = Descriptor(currentVal, tr("<List of values>"), tr("List of values"));
                }
            } else {
                cands.append(current);
                int idx = map.contains(key.getId()) ? cands.indexOf(map.value(key.getId())) : 0;
                current = (idx >= 0) ? cands.at(idx) : cands.first();
            }
            
            QTableWidgetItem* value = new QTableWidgetItem(current.getDisplayName());
            value->setData(Qt::UserRole, qVariantFromValue<Descriptor>(current));
            value->setData(Qt::UserRole+1, qVariantFromValue<QList<Descriptor> >(cands));
            if (eldt->isList()) {
                value->setData(Qt::UserRole+2, true);
            }
            table->setItem(i, VAL_COL, value);
        }

        QWidget* widget = new QWidget();
        QSizePolicy sizePolicy1(QSizePolicy::Ignored, QSizePolicy::Preferred);
        sizePolicy1.setHorizontalStretch(0);
        sizePolicy1.setVerticalStretch(0);
        sizePolicy1.setHeightForWidth(widget->sizePolicy().hasHeightForWidth());
        widget->setSizePolicy(sizePolicy1);

        QVBoxLayout* verticalLayout = new QVBoxLayout(widget);
        verticalLayout->setSpacing(0);
        verticalLayout->setMargin(0);
        verticalLayout->addWidget(table);
        verticalLayout->addWidget(doc = new QTextEdit(widget));
        doc->setEnabled(false);
        connect(table, SIGNAL(itemSelectionChanged()), SLOT(sl_showDoc()));

        return widget;
    }
    assert(0);
    return NULL;
}

static QString formatDoc(const Descriptor& s, const Descriptor& d) {
    return GB2::TypeMapEditor::tr("The input slot <b>%1</b><br>is bound to<br>the bus slot <b>%2</b>")
        .arg(s.getDisplayName())
        .arg(d.getDisplayName());
}

void TypeMapEditor::sl_showDoc() {
    QList<QTableWidgetItem *> list = table->selectedItems();
    if (list.size() == 1) {
        if (isInfoMode()) {
            doc->setText(DesignerUtils::getRichDoc(list.at(0)->data(Qt::UserRole).value<Descriptor>()));
        } else {
            int row = list.at(0)->row();
            Descriptor d = table->item(row, KEY_COL)->data(Qt::UserRole).value<Descriptor>();
            Descriptor s = table->item(row, VAL_COL)->data(Qt::UserRole).value<Descriptor>();
            doc->setText(formatDoc(d, s));
        }        
    } else {
        doc->setText("");
    }
}

void TypeMapEditor::commit() {
    QMap<QString,QString> map;
    if (table && !isInfoMode()) {
        for (int i = 0; i < table->rowCount(); i++) {
            QString key = table->item(i, KEY_COL)->data(Qt::UserRole).value<Descriptor>().getId();
            QString val = table->item(i, VAL_COL)->data(Qt::UserRole).value<Descriptor>().getId();
            map[key] = val;
        }
    }
    cfg->setParameter(propertyName, qVariantFromValue<QStrStrMap>(map));
}

QWidget *DescriptorListDelegate::createEditor(QWidget *parent,
                                              const QStyleOptionViewItem &/* option */,
                                              const QModelIndex &/* index */) const
{
    QComboBox* editor = new QComboBox(parent);
    editor->setSizeAdjustPolicy(QComboBox::AdjustToContentsOnFirstShow);
    return editor;
}

void DescriptorListDelegate::setEditorData(QWidget *editor,
                                           const QModelIndex &index) const
{
    QString current = index.model()->data(index, Qt::UserRole).value<Descriptor>().getId();
    QList<Descriptor> list = index.model()->data(index, Qt::UserRole+1).value<QList<Descriptor> >();
    QComboBox *combo = static_cast<QComboBox*>(editor);
    if (index.model()->data(index, Qt::UserRole+2).toBool()) {
        QStringList curList = current.split(";");
        QStandardItemModel* cm = new QStandardItemModel(list.size(), 1, combo);
        for (int i = 0; i < list.size(); ++i) {
            Descriptor d = list[i];
            QStandardItem* item = new QStandardItem(d.getDisplayName());
            item->setCheckable(true); item->setEditable(false); item->setSelectable(false);
            item->setCheckState(curList.contains(d.getId())? Qt::Checked: Qt::Unchecked);
            item->setData(qVariantFromValue<Descriptor>(d));
            cm->setItem(i, item);
        }
        combo->setModel(cm);
        QListView* vw = new QListView(combo);
        vw->setModel(cm);
        combo->setView(vw);
        //vw->setEditTriggers(QAbstractItemView::AllEditTriggers);
    } else {
        combo->clear();
        int currentIdx = 0;
        for (int i = 0; i < list.size(); ++i) {
            combo->addItem(list[i].getDisplayName(), qVariantFromValue<Descriptor>(list[i]));
            if (list[i] == current) {
                currentIdx = i;
            }
        }

        combo->setCurrentIndex(currentIdx);
    }
}

void DescriptorListDelegate::setModelData(QWidget *editor, QAbstractItemModel *model,
                                          const QModelIndex &index) const
{
    QComboBox *combo = static_cast<QComboBox*>(editor);
    QVariant value;
    if (index.model()->data(index, Qt::UserRole+2).toBool()) {
        QStandardItemModel* cm = qobject_cast<QStandardItemModel*>(combo->model());
        Descriptor res;
        QStringList ids;
        for(int i = 0; i< cm->rowCount();++i){
            if (cm->item(i)->checkState() == Qt::Checked) {
                res = cm->item(i)->data().value<Descriptor>();
                ids << res.getId();
            }
        }
        if (ids.isEmpty()) {
            value = qVariantFromValue<Descriptor>(Descriptor("", tr("<empty>"), tr("Default value")));
        } else if (ids.size() == 1) {
            value = qVariantFromValue<Descriptor>(res);
        } else {
            value = qVariantFromValue<Descriptor>(Descriptor(ids.join(";"), tr("<List of values>"), tr("List of values")));
        }
    } else {
        value = combo->itemData(combo->currentIndex());
    }
    model->setData(index, value, Qt::UserRole);
    model->setData(index, value.value<Descriptor>().getDisplayName(), Qt::DisplayRole);
}

using namespace Workflow;

BusPortEditor::BusPortEditor(BusPort* p) : TypeMapEditor(p, BusPort::BUS_MAP, DataTypePtr(), DataTypePtr()), port(p) {
    DataTypePtr t = to = p->getType();
    if (!t->isMap()) {
        QMap<Descriptor, DataTypePtr> map;
        map.insert(*p, t);
        to = new DataTypeSet(Descriptor(), map);
        //IntegralBusType* bt = new IntegralBusType(Descriptor(), QMap<Descriptor, DataTypePtr>());
        //bt->addOutput(t, p);
    }
    if (p->isOutput() || p->getWidth() == 0) {
        //nothing to edit, go info mode
        from = to;
    } else {
        //port is input and has links, go editing mode
        IntegralBusType* bt = new IntegralBusType(Descriptor(), QMap<Descriptor, DataTypePtr>());
        bt->addInputs(p);
        from = bt;
    }
}

QWidget* BusPortEditor::createGUI(DataTypePtr from, DataTypePtr to) {
    QWidget* w = TypeMapEditor::createGUI(from, to);
    if (table && port->getWidth() == 0) {
        if (port->isInput()) {
            table->setHorizontalHeaderLabels((QStringList() << GB2::TypeMapEditor::tr("Accepted inputs")));
        }
        else {
            table->setHorizontalHeaderLabels((QStringList() << GB2::TypeMapEditor::tr("Provided outputs")));
        }
    } else if (table) {
        connect(table->model(), SIGNAL(dataChanged(const QModelIndex&, const QModelIndex&)), 
                     this, SLOT(handleDataChanged(const QModelIndex&, const QModelIndex&)));
    }
    return w;
}

void BusPortEditor::handleDataChanged(const QModelIndex & topLeft, const QModelIndex & bottomRight) {
    Q_UNUSED(topLeft);
    Q_UNUSED(bottomRight);
    commit();
}

}//namespace GB2
