/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "SaveDocumentTask.h"

#include <core_api/DocumentModel.h>
#include <core_api/Log.h>
#include <core_api/ObjectViewModel.h>
#include <core_api/AppContext.h>

#include <QtGui/QMessageBox>

namespace GB2 {

static LogCategory log(ULOG_CAT_CORE_SERVICES);

SaveDocumentTask::SaveDocumentTask(Document* _doc, IOAdapterFactory* _io, const QString& _url) 
: Task(tr("save_document_task_name"), TaskFlags(TaskFlag_DeleteWhenFinished)), doc(_doc), io(_io), url(_url), destroy(false)
{
	assert(doc!=NULL);
	if (io == NULL) {
		io = doc->getIOAdapterFactory();
	}
	if (url.isEmpty()) {
		url = doc->getURL();
	}
	lock = new StateLock(getTaskName());
}

SaveDocumentTask::SaveDocumentTask(Document* _doc, bool d) 
: Task(tr("save_document_task_name"), TaskFlags(TaskFlag_DeleteWhenFinished)), doc(_doc), destroy(d)
{
    assert(doc!=NULL);
    io = doc->getIOAdapterFactory();
    url = doc->getURL();
    lock = new StateLock(getTaskName());
}

SaveDocumentTask::~SaveDocumentTask() {
    assert(lock == NULL);
}

void SaveDocumentTask::prepare() {
	doc->lockState(lock);
}

void SaveDocumentTask::run() {
    log.info(tr("Saving document %1\n").arg(url));

	DocumentFormat* df = doc->getDocumentFormat();
	df->storeDocument(doc, stateInfo, io, url);
}

Task::ReportResult SaveDocumentTask::report() {
    if (stateInfo.hasErrors()) {
        //TODO: show error?
    } else if (url == doc->getURL() && io == doc->getIOAdapterFactory()) {
		doc->makeClean();
	}
	doc->unlockState(lock);
    delete lock;
    lock = NULL;
    if (destroy) {
        doc->unload();
        delete doc;
    }
	return Task::ReportResult_Finished;
}

//////////////////////////////////////////////////////////////////////////
/// save multiple

SaveMiltipleDocuments::SaveMiltipleDocuments(const QList<Document*>& docs, bool askBeforeSave) 
: Task(tr("save_multiple_documents_task_name"), TaskFlags_NR_DWF_SS)
{
    bool saveAll = false;
    foreach(Document* doc, docs) {
        bool save=true;
        if (askBeforeSave) {
            QMessageBox::StandardButtons buttons = QMessageBox::StandardButtons(QMessageBox::Yes) | QMessageBox::No;
            if (docs.size() > 1) {
                buttons = buttons | QMessageBox::YesToAll | QMessageBox::NoToAll;
            }

            QMessageBox::StandardButton res = saveAll ? QMessageBox::YesToAll : QMessageBox::question(NULL, 
                tr("save_doc_title"), tr("save_doc_text: %1").arg(doc->getURL()), 
                buttons, QMessageBox::Yes);

            if (res == QMessageBox::NoToAll) {
                break;
            } 
            if (res == QMessageBox::YesToAll) {
                saveAll = true;
            }
            if (res == QMessageBox::No) {
                save = false;
            }
        }
        if (save) {
            addSubTask(new SaveDocumentTask(doc));
        }
    }
}


QList<Document*> SaveMiltipleDocuments::findModifiedDocuments(const QList<Document*>& docs) {
	QList<Document*> res;
	foreach(Document* doc, docs) {
		if (doc->isTreeItemModified()) {
			res.append(doc);
		}
	}
	return res;
}

//////////////////////////////////////////////////////////////////////////
// unload document
UnloadDocumentTask::UnloadDocumentTask(Document* _doc, bool save) 
: Task(tr("unload_document_%1").arg(_doc->getURL()), TaskFlags_NR_DWF_SS), doc(_doc), saveTask(NULL)
{
    if (save) {
        saveTask = new SaveDocumentTask(doc);
        addSubTask(saveTask);
    }
}

Task::ReportResult UnloadDocumentTask::report() {
    if (doc.isNull() || !doc->isLoaded()) {
        return Task::ReportResult_Finished;
    }
    propagateSubtaskError();
    QString errPrefix = tr("Document '%1' can't be unloaded: ").arg(doc->getName());
    if (hasErrors()) {
        assert(saveTask!=NULL);
        log.error(errPrefix +  tr("save failed!"));
        return Task::ReportResult_Finished;
    }
    QString error = checkSafeUnload(doc);
    if (!error.isEmpty()) {
        stateInfo.error = errPrefix + error;
        log.error(stateInfo.error);
        return Task::ReportResult_Finished;
    }
    bool ok = doc->unload();
    if (!ok) {
        stateInfo.error = errPrefix + tr("unexpected error");
        log.error(stateInfo.error);
    }
    return Task::ReportResult_Finished;
}

void UnloadDocumentTask::runUnloadTaskHelper(const QList<Document*>& docs, UnloadDocumentTask_SaveMode sm) {
    QMap<Document*, QString> failedToUnload;

    // document can be unloaded if there are no active view with this doc + it's not state locked by user
    TriState saveAll = sm == UnloadDocumentTask_SaveMode_Ask ? TriState_Unknown : 
        (sm == UnloadDocumentTask_SaveMode_NotSave ? TriState_No : TriState_Yes);

    foreach(Document* doc, docs) {
        QString err = checkSafeUnload(doc);
        if (!err.isEmpty()) {
            failedToUnload[doc] = err;
            continue;
        }
        bool saveCurrentDoc = doc->isModified() && saveAll == TriState_Yes;
        if (doc->isModified() && saveAll == TriState_Unknown) {

            QMessageBox::StandardButtons buttons = QMessageBox::StandardButtons(QMessageBox::Yes) | QMessageBox::No;
            if (docs.size() > 1) {
                buttons = buttons | QMessageBox::YesToAll | QMessageBox::NoToAll;
            }

            QMessageBox::StandardButton res = saveAll ? QMessageBox::YesToAll : QMessageBox::question(NULL, 
                tr("save_doc_title"), tr("save_doc_text: %1").arg(doc->getURL()), 
                buttons, QMessageBox::Yes);

            if (res == QMessageBox::NoToAll) {
                saveAll = TriState_No;
            } else  if (res == QMessageBox::YesToAll) {
                saveAll = TriState_Yes;
                saveCurrentDoc = true;
            } else if (res == QMessageBox::No) {
                saveCurrentDoc = false;
            } else {
                assert(res == QMessageBox::Yes);
                saveCurrentDoc = true;
            }
        }
        AppContext::getTaskScheduler()->registerTopLevelTask(new UnloadDocumentTask(doc, saveCurrentDoc));
    }

    if (!failedToUnload.isEmpty()) {
        QString text = tr("document_failed_to_unload:")+"<br>";
        foreach(Document* doc, failedToUnload.keys()) {
            text+=doc->getName()+" : " + failedToUnload[doc] + "<br>";
        }
        QMessageBox::warning(NULL, tr("Warning"), text);
    }
}

QString UnloadDocumentTask::checkSafeUnload(Document* doc) {
    bool hasViews = !GObjectViewUtils::findViewsWithAnyOfObjects(doc->getObjects()).isEmpty();
    if (hasViews) {
        return tr("document has active views");
    } 

    bool liveLocked = doc->hasLocks(StateLockableTreeFlags_ItemAndChildren, StateLockFlag_LiveLock);
    if (liveLocked) {
        return tr("document is live-locked");
    } 
    
    return QString();
}

}//namespace
