/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "LoadDocumentTask.h"

#include <core_api/AppContext.h>
#include <core_api/ProjectModel.h>
#include <core_api/Log.h>
#include <core_api/ResourceTracker.h>
#include <core_api/DocumentModel.h>
#include <core_api/ObjectViewModel.h>
#include <core_api/GObjectReference.h>
#include <core_api/GObject.h>

#include <selection/GObjectSelection.h>

#include <gobjects/GObjectTypes.h>
#include <gobjects/GObjectRelationRoles.h>

#include <util_ov_textview/SimpleTextObjectView.h>

#include <QtCore/QFileInfo>
#include <QtGui/QApplication>

namespace GB2 {

/* TRANSLATOR GB2::LoadUnloadedDocumentTask */    

static LogCategory log(ULOG_CAT_IO);


//////////////////////////////////////////////////////////////////////////
// LoadUnloadedDocumentTask

//TODO: support subtask sharing!
//TODO: avoid multiple load tasks when opening view for unloaded doc!

LoadUnloadedDocumentTask::LoadUnloadedDocumentTask(Document* d) :
Task("", TaskFlags_NR_DWF), subtask(NULL), unloadedDoc(d)
{
	setVerboseLogMode(true);
    setTaskName(tr("Load '%1'").arg(d->getName()));
    setUseDescriptionFromSubtask(true);
    assert(d!=NULL);
}

void LoadUnloadedDocumentTask::prepare() {
	if (unloadedDoc == NULL) {
		stateInfo.error=tr("document_removed");
		return;
	}
	if (unloadedDoc->isLoaded()) {
		return;
	}

    DocumentFormatId format = unloadedDoc->getDocumentFormatId();
    QString formatName = AppContext::getDocumentFormatRegistry()->getFormatById(format)->getFormatName();
	IOAdapterFactory* iof = unloadedDoc->getIOAdapterFactory();
	const QString& url = unloadedDoc->getURL();
    log.details(tr("Starting load document from %1, document format %2").arg(url).arg(formatName));
	subtask = new LoadDocumentTask(format, url, iof, unloadedDoc->getGHintsMap());
	addSubTask(subtask);

    resName = getResourceName(unloadedDoc);
    AppContext::getResourceTracker()->registerResourceUser(resName, this);
}

void LoadUnloadedDocumentTask::clearResourceUse() {
    if (!resName.isEmpty()) {
        AppContext::getResourceTracker()->unregisterResourceUser(resName, this);
        resName.clear();
    }
}

Task::ReportResult LoadUnloadedDocumentTask::report() {
    Task::ReportResult res = ReportResult_Finished;
    Project* p = AppContext::getProject();
	
	if (unloadedDoc == NULL) {
        stateInfo.error=tr("document_removed");
    } else {
        propagateSubtaskError();
    }

    if (hasErrors()) {
        log.error(tr("Error: %1").arg(stateInfo.error));
		if (!resName.isEmpty()) {
			clearResourceUse();
			resName.clear();
		}
    } else if (isCanceled() || (subtask!=NULL && subtask->isCanceled())) {
        //do nothing
    } else if (unloadedDoc->isLoaded()) {
        //do nothing
    } else if (p && p->isStateLocked()) {
        res = ReportResult_CallMeAgain; //wait until project is unlocked
    } else {
        assert(unloadedDoc->isStateLocked()); // all unloaded docs are always state locked
        
        //todo: move to utility method?
        const QList<StateLock*>& locks = unloadedDoc->getStateLocks();
        bool readyToLoad = true;
        foreach(StateLock* lock, locks) {
            if  (  lock != unloadedDoc->getDocumentModLock(DocumentModLock_IO)
                && lock != unloadedDoc->getDocumentModLock(DocumentModLock_USER) 
                && lock != unloadedDoc->getDocumentModLock(DocumentModLock_FORMAT_AS_CLASS)
                && lock != unloadedDoc->getDocumentModLock(DocumentModLock_FORMAT_AS_INSTANCE)
                && lock != unloadedDoc->getDocumentModLock(DocumentModLock_UNLOADED_STATE))
            {
                readyToLoad = false;
            }
        }
        if (!readyToLoad) {
            stateInfo.error = tr("document_is_locked"); //todo: wait instead?
        }  else {
            Document* doc = subtask->getDocument();
            unloadedDoc->loadFrom(doc); // doc was load in a separate thread -> clone all GObjects
            assert(!unloadedDoc->isTreeItemModified());
            assert(unloadedDoc->isLoaded());
        }
    }
    if (res == ReportResult_Finished) {
        clearResourceUse();
    }
    return res;
}


Document* LoadUnloadedDocumentTask::getDocument() const {
    return unloadedDoc;
}


QString LoadUnloadedDocumentTask::getResourceName(Document* d) {
    return QString(LoadUnloadedDocumentTask::tr("project_doc_resource") + ":" + d->getURL());
}

LoadUnloadedDocumentTask* LoadUnloadedDocumentTask::findActiveLoadingTask(Document* d) {
    QString res = getResourceName(d);
    QList<Task*> tasks = AppContext::getResourceTracker()->getResourceUsers(res);
    foreach(Task* t, tasks) {
        LoadUnloadedDocumentTask* lut = qobject_cast<LoadUnloadedDocumentTask*>(t);
        if (lut!=NULL) {
            return lut;
        }
    }
    return false;
}


//////////////////////////////////////////////////////////////////////////
// Load Document


LoadDocumentTask::LoadDocumentTask(DocumentFormatId f, const QString& u, IOAdapterFactory* i, const QVariantMap& map)
: Task("", TaskFlag_DeleteWhenFinished), format(f), url(u), iof(i), hints(map), result(NULL)
{
    setTaskName(tr("Read document: '%1'").arg(QFileInfo(u).fileName()));
 
    tpm = Progress_Manual;
	assert(iof!=NULL);
}

LoadDocumentTask::~LoadDocumentTask() {
    cleanup();
}

void LoadDocumentTask::cleanup() {
	if (result!=NULL) {
		delete result;
        result = NULL;
	}
}

void LoadDocumentTask::run() {
	DocumentFormat* f = AppContext::getDocumentFormatRegistry()->getFormatById(format);
	if (f == NULL) {
		stateInfo.error = tr("invalid_format_%1").arg(format);
		return;
	}
    result = f->loadExistingDocument(iof, url, stateInfo, hints);
    assert(isCanceled() || result!=NULL || hasErrors());
    assert(result == NULL || result->isLoaded());
}


Task::ReportResult LoadDocumentTask::report() {
	if (stateInfo.hasErrors() || isCanceled()) {
		return ReportResult_Finished;
	}
	assert(result!=NULL);
	return ReportResult_Finished;
}


//////////////////////////////////////////////////////////////////////////
// LoadUnloadedDocumentAndOpenViewTask

LoadUnloadedDocumentAndOpenViewTask::LoadUnloadedDocumentAndOpenViewTask(Document* d) :
Task("", TaskFlags_RASF_DWF_SSSOR)
{
    loadUnloadedTask = new LoadUnloadedDocumentTask(d);
    setUseDescriptionFromSubtask(true);

    setVerboseLogMode(true);
    setTaskName(tr("Load document: '%1'").arg(d->getName()));

    addSubTask(loadUnloadedTask);
}

static Task* createOpenViewTask(const MultiGSelection& ms) {
    QList<GObjectViewFactory*> fs = AppContext::getObjectViewFactoryRegistry()->getAllFactories();
    QList<GObjectViewFactory*> ls;

    foreach(GObjectViewFactory* f, fs) {
        //check if new view can be created
        if (f->canCreateView(ms)) {
            ls.append(f);
        }
    }
    
    if (ls.size() > 1) {
        GObjectViewFactory* f = AppContext::getObjectViewFactoryRegistry()->getFactoryById(SimpleTextObjectViewFactory::ID);
        if (ls.contains(f)) {
            // ignore auxiliary text data
            ls.removeAll(f);
        }
    }

    if (ls.size() == 1) {
        GObjectViewFactory* f = ls.first();
        Task* t = f->createViewTask(ms, true);
        return t;
    }
    return NULL;
}

QList<Task*> LoadUnloadedDocumentAndOpenViewTask::onSubTaskFinished(Task* subTask) {
    QList<Task*> res;
    
	propagateSubtaskError();
    if (subTask != loadUnloadedTask || hasErrors() || subTask->isCanceled() || isCanceled()) {
        return res;
    }
    
    // look if saved state can be loaded
    Document* doc = loadUnloadedTask->getDocument();
    assert(doc->isLoaded());
    
    //if any of existing views has added an object from the document -> do not open new view
    const QList<GObject*>& docObjects = doc->getObjects();
    if (!GObjectViewUtils::findViewsWithAnyOfObjects(docObjects).isEmpty()) {
        return res;
    }
    
    //try open new view
    GObjectSelection os; os.addToSelection(docObjects);
    MultiGSelection ms; ms.addSelection(&os);
    QList<GObjectViewState*> sl = GObjectViewUtils::selectStates(ms, AppContext::getProject()->getGObjectViewStates());
    if (sl.size() == 1) {
        GObjectViewState* state = sl.first();
        GObjectViewFactory* f = AppContext::getObjectViewFactoryRegistry()->getFactoryById(state->getViewFactoryId());
        assert(f!=NULL);
        res.append(f->createViewTask(state->getViewName(), state->getStateData()));
    } else {
        Task* openViewTask = createOpenViewTask(ms);
        if (openViewTask!=NULL) {
            openViewTask->setSubtaskProgressWeight(0);
            res.append(openViewTask);
        }
    }

    if (res.isEmpty()) { 
        // no view can be opened -> check special case: loaded object contains annotations associated with sequence
        // -> load sequence and open view for it;
        foreach(GObject* obj, doc->findGObjectByType(GObjectTypes::ANNOTATION_TABLE)) {
            QList<GObjectRelation> rels = obj->findRelatedObjectsByRole(GObjectRelationRole::SEQUENCE);
            if (rels.isEmpty()) {
                continue;
            }
            const GObjectRelation& rel = rels.first();
            Document* seqDoc = AppContext::getProject()->findDocumentByURL(rel.ref.docUrl);
            if (seqDoc!=NULL) {
                if (seqDoc->isLoaded()) { //try open sequence view 
                    GObject* seqObj = seqDoc->findGObjectByName(rel.ref.objName);
                    if (seqObj!=NULL && seqObj->getGObjectType() == GObjectTypes::DNA_SEQUENCE) {
                        GObjectSelection os2; os2.addToSelection(seqObj);
                        MultiGSelection ms2; ms2.addSelection(&os2);
                        Task* openViewTask = createOpenViewTask(ms2);
                        if (openViewTask!=NULL) {
                            openViewTask->setSubtaskProgressWeight(0);
                            res.append(openViewTask);
                        }
                    }
                } else { //try load doc and open sequence view 
                    AppContext::getTaskScheduler()->registerTopLevelTask(new LoadUnloadedDocumentAndOpenViewTask(seqDoc));
                }
            } 
            if (!res.isEmpty()) { //one view is ok
                break;
            }
        }
    }
    return res;
}


}//namespace
