/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "MSAEditorNameList.h"
#include "MSAEditor.h"

#include <gobjects/MAlignmentObject.h>
#include <datatype/MAlignment.h>
#include <util_gui/GUIUtils.h>

#include <QtGui/QApplication>
#include <QtGui/QClipboard>

namespace GB2 {

MSAEditorNameList::MSAEditorNameList(MSAEditorUI* _ui) : editor(_ui->editor), ui(_ui) {
    //setup fonts
    QFont nameFont;
    nameFont.setFamily("Times");
    nameFont.setPointSize(12);
    nameFont.setBold(true);
    setFont(nameFont);
    QFontMetrics nameFM(nameFont);
    int spacing = nameFM.height() / 5;
    setSpacing(spacing);

    ui->consList->setFont(nameFont);
    ui->consList->setSpacing(spacing);
    ui->consList->setVerticalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
    ui->consList->setHorizontalScrollBarPolicy(Qt::ScrollBarAsNeeded);
    ui->consList->addItem(""); // for graph
    ui->consList->addItem(""); // for graph
    ui->consList->addItem(tr("CONSENSUS"));
    ui->consList->addItem(""); // for ruler & readonly message
    updateStateLockInfo();
    
    Qt::ItemFlags flags(Qt::NoItemFlags);
    ui->consList->item(0)->setFlags(flags);
    ui->consList->item(1)->setFlags(flags);
    //3 item with doc state manages flags by itself

    connect(ui->consList, SIGNAL(currentItemChanged(QListWidgetItem*, QListWidgetItem*)), SLOT(sl_consCurrentItemChanged(QListWidgetItem*, QListWidgetItem*)));
    
    setVerticalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
    setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOn);
    setVerticalScrollMode(QAbstractItemView::ScrollPerItem);
    
    //init default vals
    populateList();

    connect(editor, SIGNAL(si_buildStaticMenu(GObjectView*, QMenu*)), SLOT(sl_buildStaticMenu(GObjectView*, QMenu*)));
    connect(this, SIGNAL(itemSelectionChanged()), SLOT(sl_onItemSelectionChanged()));

    copyCurrentSequenceAction = new QAction(tr("copy_current_sequence"), this);
    connect(copyCurrentSequenceAction, SIGNAL(triggered()), SLOT(sl_copyCurrentSequence()));

    removeCurrentSequenceAction = new QAction("Remove current sequence", this);
    connect(removeCurrentSequenceAction, SIGNAL(triggered()), SLOT(sl_removeCurrentSequence()));
        
    connect(editor, SIGNAL(si_buildPopupMenu(GObjectView* , QMenu*)), SLOT(sl_buildContextMenu(GObjectView*, QMenu*)));
    connect(editor->getMSAObject(), SIGNAL(si_sequenceListModified()), SLOT(sl_sequenceListModified()));
    connect(editor->getMSAObject(), SIGNAL(si_lockedStateChanged()), SLOT(sl_lockedStateChanged()));
    
    updateActions();
}

void MSAEditorNameList::updateActions() {
    copyCurrentSequenceAction->setEnabled(currentItem()!=NULL);    
    
    MAlignmentObject* maObj = editor->getMSAObject();
    removeCurrentSequenceAction->setEnabled(currentItem()!=NULL && !maObj->isStateLocked() && maObj->getMAlignment().getNumSequences() > 1);
}


void MSAEditorNameList::sl_consCurrentItemChanged(QListWidgetItem* current, QListWidgetItem* prev) {
    Q_UNUSED(prev);
    //if (current!=ui->consList->item(2)) {
    //    ui->consList->setCurrentItem(ui->consList->item(2));
    //}
    if (current!=NULL) {
        ui->consList->setCurrentItem(NULL);
    }
    updateActions();
}

int MSAEditorNameList::getFirstVisibleSequence() const {
    //todo: optimize
    QRect visibleRect = viewport()->visibleRegion().boundingRect();
    for (int i=0, n = count(); i<n; i++) {
        QListWidgetItem* it = item(i);
        QRect rect = visualItemRect(it);
        bool visible = rect.intersects(visibleRect);
        if (visible) {
            return i;
        }
    }
    assert(0);
    return 0;
}

int MSAEditorNameList::getLastVisibleSequence(bool countClipped) const {
    //todo: optimize
    for (int i = count() - 1; i>=0; i--) {
        LRegion r = getSequenceYRange(i, false);
        if (!r.isEmpty()) {
            if (!countClipped) {
                QListWidgetItem* it = item(i);
                QRect rect = visualItemRect(it);
                int sp = spacing();
                if (r.len < rect.height() + sp * 2) {
                    continue;
                }
            }
            return i;
        }
    }
    assert(0);
    return 0;
}

bool MSAEditorNameList::isSequenceVisible(int n, bool countClipped) const {
    assert(n >=0 && n < editor->getNumSequences());
    return n >= getFirstVisibleSequence() &&  n<=getLastVisibleSequence(countClipped);
}

void MSAEditorNameList::sl_onItemSelectionChanged() {
    updateActions();
}

void MSAEditorNameList::setFirstVisibleSequence(int n) {
    int numSeqs = editor->getNumSequences();
    assert(n >=0 && n < numSeqs);
    
    QScrollBar* sc = verticalScrollBar();
    int scMax = sc->maximum();
    assert(sc->minimum() == 0 && scMax <= numSeqs && sc->singleStep() == 1);
    
    int nItemsCanBeFirst = numSeqs - (numSeqs - scMax);
    int effectiveFirst = qMin(nItemsCanBeFirst, n);
    sc->setValue(effectiveFirst);
}

LRegion MSAEditorNameList::getSequenceYRange(int n, bool useVirtualCoords) const {
    assert( n >= 0 && n < editor->getNumSequences());
    QListWidgetItem* it = item(n);
    QRect rect = visualItemRect(it);
    LRegion res(rect.top(), rect.height());
    
    int sp = spacing();
    res.startPos -= sp;
    res.len += 2*sp;

    if (!useVirtualCoords) {
        LRegion viewPortYRange(0, viewport()->height());
        res = res.intersect(viewPortYRange);
    }
    res.startPos += viewport()->geometry().top();
    return res;
}

void MSAEditorNameList::sl_buildStaticMenu(GObjectView* v, QMenu* m) {
    Q_UNUSED(v);
    buildMenu(m);
}

void MSAEditorNameList::sl_buildContextMenu(GObjectView* v, QMenu* m) {
    Q_UNUSED(v);
    buildMenu(m);
}

void MSAEditorNameList::buildMenu(QMenu* m) {    
    QMenu* copyMenu = GUIUtils::findSubMenu(m, MSAE_MENU_COPY);
    assert(copyMenu!=NULL);
    copyMenu->addAction(copyCurrentSequenceAction);

    QMenu* editMenu = GUIUtils::findSubMenu(m, MSAE_MENU_EDIT);
    assert(editMenu!=NULL);
    editMenu->addAction(removeCurrentSequenceAction);
}


void MSAEditorNameList::sl_copyCurrentSequence() {
    QListWidgetItem* lwi = currentItem();
    assert(lwi!=NULL);
    int n = row(lwi);
    const MAlignmentItem& item = editor->getMSAObject()->getMAlignment().alignedSeqs[n];
    QApplication::clipboard()->setText(QString(item.sequence));
}

void MSAEditorNameList::populateList() {
    const MAlignment& msa = editor->getMSAObject()->getMAlignment();
    foreach(const MAlignmentItem& item, msa.alignedSeqs) {
        addItem(item.name);
    }

}

void MSAEditorNameList::sl_sequenceListModified() {
    //todo: optimize and insert/del part of the sequences only
    int first = getFirstVisibleSequence();

    clear();
    populateList();
    
    first = qMin(first, editor->getNumSequences());
    setFirstVisibleSequence(first);
    
    updateActions();
}


void MSAEditorNameList::sl_removeCurrentSequence() {
    QListWidgetItem* lwi = currentItem();
    assert(lwi!=NULL);
    MAlignmentObject* maObj = editor->getMSAObject();
    assert(!maObj->isStateLocked());
    
    int n = row(lwi);
    MAlignment ma = maObj->getMAlignment();
    assert(ma.getNumSequences() > 1);
    ma.alignedSeqs.removeAt(n);
    
    maObj->setMAlignment(ma);
}

void MSAEditorNameList::sl_lockedStateChanged() {
    updateStateLockInfo();
    updateActions();
}

#define LOCK_ITEM_NUM 3
void MSAEditorNameList::updateStateLockInfo() {
    MAlignmentObject* obj = editor->getMSAObject();
    bool locked = obj->isStateLocked();
    QListWidgetItem* item = ui->consList->item(LOCK_ITEM_NUM);
    item->setFlags(locked ? Qt::ItemIsEnabled : Qt::NoItemFlags);
    if (locked) {
        item->setIcon(QIcon(":core/images/lock.png"));
        item->setText(tr("read_only"));
    } else {
        item->setIcon(QIcon());
        item->setText("");
    }
}

}//namespace
