/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "MSAEditorConsensusArea.h"
#include "MSAEditor.h"
#include "MSAEditorSequenceArea.h"

#include <gobjects/MAlignmentObject.h>
#include <util_gui/GraphUtils.h>
#include <util_gui/GUIUtils.h>
#include <util_algorithm/MSAUtils.h>
#include <core_api/DNAAlphabet.h>

#include <QtGui/QPainter>
#include <QtGui/QApplication>
#include <QtGui/QClipboard>

namespace GB2 {
#define RULER_LINE 3
#define CONS_LINE  2

MSAEditorConsensusArea::MSAEditorConsensusArea(MSAEditorUI* _ui) : editor(_ui->editor), ui(_ui) {
    rulerFont.setFamily("Arial");
    rulerFont.setPointSize(int(ui->seqArea->getSeqFont().pointSize() * 0.7));
    setSizePolicy(QSizePolicy::MinimumExpanding, QSizePolicy::Fixed);
    setMinimumWidth(100);
    connect(ui->seqArea, SIGNAL(si_startPosChanged(int)), SLOT(sl_startPosChanged(int)));
    connect(ui->seqArea, SIGNAL(si_cursorPosChanged(const QPoint&, const QPoint&)), SLOT(sl_cursorPosChanged(const QPoint&, const QPoint&)));
    connect(editor->getMSAObject(), SIGNAL(si_alignmentModified()), SLOT(sl_alignmentModified()));
    connect(editor->getMSAObject(), SIGNAL(si_sequenceListModified()), SLOT(sl_alignmentModified()));
    

    connect(editor, SIGNAL(si_buildStaticMenu(GObjectView*, QMenu*)), SLOT(sl_buildStaticMenu(GObjectView*, QMenu*)));
    copyConsensusAction = new QAction(tr("copy_consensus"), this);
    connect(copyConsensusAction, SIGNAL(triggered()), SLOT(sl_copyConsensusSequence()));
    connect(editor, SIGNAL(si_buildPopupMenu(GObjectView* , QMenu*)), SLOT(sl_buildContextMenu(GObjectView*, QMenu*)));

    updateStats();
    updateConsensus();
}

void MSAEditorConsensusArea::paintEvent(QPaintEvent *e) {
    QPainter p(this);
    p.fillRect(rect(), Qt::white);
    p.setFont(ui->seqArea->getSeqFont());
    
    drawCursor(p);

    drawConsensus(p);

    drawRuler(p);

    drawHistogram(p);

    QWidget::paintEvent(e);
}


void MSAEditorConsensusArea::drawCursor(QPainter& p) {
    LRegion yRange = getLineYRange(CONS_LINE);    
    LRegion xRange = ui->seqArea->getBaseXRange(ui->seqArea->getCursorPos().x(), false);
    QColor color(Qt::lightGray);
    color = color.lighter(115);
    p.fillRect(xRange.startPos, yRange.startPos, xRange.len, yRange.len, color);
}

void MSAEditorConsensusArea::drawConsensus(QPainter& p) {
    //draw consensus
    p.setPen(Qt::black);
    
    QFont f = ui->seqArea->getSeqFont();
    f.setWeight(QFont::DemiBold);
    p.setFont(f);

    int w = width();
    int h = height();
    int startPos = ui->seqArea->getFirstVisibleBase();
    int lastPos = ui->seqArea->getLastVisibleBase(true);
    assert(lastPos < consensusLine.size());
    LRegion yRange = getLineYRange(CONS_LINE);
    for (int pos = startPos; pos <= lastPos; pos++) {
        LRegion baseXRange = ui->seqArea->getBaseXRange(pos, false);
        QRect cr(baseXRange.startPos, yRange.startPos, baseXRange.len + 1, yRange.len);
        Q_ASSERT(cr.left() < w && cr.top() < h);
        char c = consensusLine[pos];
        p.drawText(cr, Qt::AlignVCenter | Qt::AlignHCenter, QString(c));
    }
}

void MSAEditorConsensusArea::drawRuler(QPainter& p) {
    //draw ruler
    p.setPen(Qt::darkGray);

    int w = width();
    int startPos = ui->seqArea->getFirstVisibleBase();
    int lastPos = ui->seqArea->getLastVisibleBase(true);

    QFontMetrics rfm(rulerFont);
    LRegion rr = getLineYRange(RULER_LINE);
    LRegion rrP = getLineYRange(CONS_LINE);
    int dy = rr.startPos - rrP.endPos();
    rr.len+=dy;
    rr.startPos-=dy;
    LRegion firstBaseXReg = ui->seqArea->getBaseXRange(startPos, false);
    LRegion lastBaseXReg = ui->seqArea->getBaseXRange(lastPos, false);
    int firstLastLen = lastBaseXReg.startPos - firstBaseXReg.startPos;
    int firstXCenter = firstBaseXReg.startPos + firstBaseXReg.len / 2;
    QPoint startPoint(firstXCenter, rr.startPos);
    
    GraphUtils::RulerConfig c;
    c.singleSideNotches = true;
    c.notchSize = 3;
    c.textOffset = (rr.len - rfm.ascent()) /2;
    c.extraAxisLenBefore = startPoint.x();
    c.extraAxisLenAfter = w - (startPoint.x() + firstLastLen);
    GraphUtils::drawRuler(p, startPoint, firstLastLen, startPos + 1, lastPos + 1, rulerFont, c);

    startPoint.setY(rr.endPos());
    c.drawNumbers = false;
    c.textPosition = GraphUtils::LEFT;
    GraphUtils::drawRuler(p, startPoint, firstLastLen, startPos, lastPos, rulerFont, c);
}

void MSAEditorConsensusArea::drawHistogram(QPainter& p) {
    QColor c("#255060");
    p.setPen(c);
    LRegion l1 = getLineYRange(0);
    LRegion l2 = getLineYRange(1);
    LRegion yr = LRegion::containingRegion(l1, l2);

    int nSeq = editor->getNumSequences();
    QBrush brush(c, Qt::Dense4Pattern);
    QVector<int> counts(256, 0);
    for (int pos = ui->seqArea->getFirstVisibleBase(), lastPos = ui->seqArea->getLastVisibleBase(true); pos <= lastPos; pos++) {
        LRegion xr = ui->seqArea->getBaseXRange(pos, true);
        int max = topSyms[pos].count;
        float k = max == 1 ? 0 : max / float(nSeq);
        assert(k >= 0 && k <= 1);
        int h =(int) ((1 - k) * yr.len);
        p.drawRect(xr.startPos+1, yr.startPos + h , xr.len-2, yr.len - h);
        p.fillRect(xr.startPos+1, yr.startPos + h , xr.len-2, yr.len - h, brush);
    }
}

LRegion MSAEditorConsensusArea::getLineYRange(int n) const {
#ifdef _DEBUG
    int consAreaY = geometry().y();
    int consListY = ui->consList->geometry().y();
    assert(consAreaY == consListY);
#endif
    int consListViewportOffset = ui->consList->viewport()->geometry().y();
    LRegion r;
    QListWidgetItem* item = ui->consList->item(n);
    QRect itemRect = ui->consList->visualItemRect(item);
    r.startPos = itemRect.top() + consListViewportOffset;
    r.len = itemRect.height();
    return r;
}

void MSAEditorConsensusArea::sl_startPosChanged(int pos) {
    Q_UNUSED(pos);
    update();
}

void MSAEditorConsensusArea::sl_alignmentModified() {
    updateStats();
    updateConsensus();
    update();
}

void MSAEditorConsensusArea::updateConsensus() {
    MSAUtils::updateConsensus(editor->getMSAObject()->getMAlignment(), consensusLine, MSAConsensusType_Default);
}

void MSAEditorConsensusArea::sl_cursorPosChanged(const QPoint& pos, const QPoint& prev) {
    if (prev.x() == pos.x()) {
        return;
    }
    update();
}

void MSAEditorConsensusArea::sl_buildStaticMenu(GObjectView* v, QMenu* m) {
    Q_UNUSED(v);
    buildMenu(m);
}

void MSAEditorConsensusArea::sl_buildContextMenu(GObjectView* v, QMenu* m) {
    Q_UNUSED(v);
    buildMenu(m);
}

void MSAEditorConsensusArea::buildMenu(QMenu* m) {
    QMenu* copyMenu = GUIUtils::findSubMenu(m, MSAE_MENU_COPY);
    assert(copyMenu!=NULL);
    copyMenu->addAction(copyConsensusAction);
}

void MSAEditorConsensusArea::sl_copyConsensusSequence() {
    QApplication::clipboard()->setText(QString(consensusLine));
}

void MSAEditorConsensusArea::updateStats() {
    
    const MAlignment& ma = editor->getMSAObject()->getMAlignment();
    int aliLen = editor->getAlignmentLen();
    int nSeq = ma.getNumSequences();
 
    topSyms.resize(aliLen);

    //row stat
    QVector<int> counts(256, 0);
    int* countsData = counts.data();
    
    //optimization -> zero only range from minChar to maxChar
    QByteArray syms = ma.alphabet->getAlphabetChars();
    int minS = uchar(syms[0]);
    int maxS = uchar(syms[syms.size()-1]);

    for (int pos = 0; pos < aliLen; pos++) {
        int max = 0;
        char maxC = MAlignment_GapChar;
        std::fill(countsData + minS, countsData + maxS + 1, 0);
        for (int seq = 0; seq < nSeq; seq++) {
            uchar c = (uchar)ma.getBase(seq, pos);
            if (c == MAlignment_GapChar) {
                continue;
            }
            assert(c <= maxS); //if not true -> the alignment alphabet is invalid
            int v = ++countsData[c];
            if (v > max) {
                max = v;
                maxC = c;
            }
            max = qMax(max, v);
            assert(max <= nSeq);
        }
        MSASymStatItem& item = topSyms[pos];
        item.count = max;
        item.c = maxC;
    }
}

}//namespace
