/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "PanViewRows.h"

#include <gobjects/AnnotationTableObject.h>

namespace GB2 {

PVRowsManager::~PVRowsManager() {
    clear();
}

bool PVRowsManager::isUpToDate() const {
	foreach(PVRowData* r, rows) {
        if (!r->uptodate) {
            return false;
        }
    }
    return true;
}

void PVRowsManager::clear() {
	foreach(PVRowData* r, rows) {
		delete r;
	}
    rows.clear();
	rowByAnnotaion.clear();
}

bool PVRowsManager::fitToRow(PVRowData* row, const LRegion& region) {
	foreach(const LRegion& r, row->ranges) {
		if (r.intersects(region)) {
            return false;
        }
	}
    return true;
}

bool PVRowsManager::addToRow(Annotation* a, PVRowData* row) {
    //every feature location must fit to row
	const QList<LRegion>& location = a->getLocation();
	foreach(const LRegion& r, location) {
        if (!fitToRow(row, r)) {
            return false;
        }
    }
    //ok this feature can be added to row;
	row->ranges << location;
    row->annotations.append(a);
    rowByAnnotaion[a] = row;
    return true;
}

int PVRowsManager::addAnnotation(Annotation* a, const QString& key) {
	foreach(PVRowData* row, rows) {
            if (row->key == key) {
            if (addToRow(a, row)) {
                return row->rowNum;
            }
        }
    }
    PVRowData* row = new PVRowData(key);
    row->rowNum = rows.size();
    rows.push_back(row);
    addToRow(a, row);
    row->uptodate = false;
    return row->rowNum;
}


void PVRowsManager::removeAnnotation(Annotation* a) {
	PVRowData* row = rowByAnnotaion.value(a, NULL);
	assert(row != NULL);
	rowByAnnotaion.remove(a);
	row->annotations.removeAll(a);
	row->uptodate = false;
    if (row->annotations.empty()) {
        resort();
    }
}

int PVRowsManager::getNumRows() const {
    return rows.size();
}

int PVRowsManager::getAnnotationRow(Annotation* a) {
	resort();
	PVRowData* row = rowByAnnotaion.value(a, NULL);
	assert(row != NULL);
	return row->rowNum;
}

inline bool compare_rows(PVRowData* x, PVRowData* y) { 
    return  x->key.compare(y->key) > 0;
}    


void PVRowsManager::resort() {
    if (isUpToDate()) {
        return;
    }
    //1 remove empty rows;
	QList<PVRowData*> toRemove;
	foreach(PVRowData* row, rows) {
        if (row->annotations.empty()) {
            toRemove.append(row);
        } else {
            row->uptodate = true;
        }
    }
	foreach(PVRowData* row, toRemove) {
		rows.removeAll(row);
		delete row;
	}
	
    //resort rows;
	qStableSort(rows.begin(), rows.end(), compare_rows);
	int i=0;
    foreach(PVRowData* row, rows) {
        row->rowNum = i;
		i++;
    }
}

const QString& PVRowsManager::getRowKey(int rowNum) const {
    assert(isUpToDate());
    assert(rowNum >= 0 && rowNum < (int)rows.size());
    PVRowData* r = rows[rowNum];
    return r->key;
}

int PVRowsManager::getNumAnnotationsInRow(int rowNum) const {
	assert(isUpToDate());
	assert(rowNum >= 0 && rowNum < (int)rows.size());
	PVRowData* r = rows[rowNum];
	return r->annotations.size();
}


bool PVRowsManager::contains(const QString& key) const {
    foreach(PVRowData* r, rows) {
        if (r->key == key) {
            return true;
        }
    }
    return false;
}
} // namespace
