/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "GSequenceLineViewAnnotated.h"
#include "ADVSequenceObjectContext.h"

#include <core_api/AppContext.h>
#include <gobjects/AnnotationTableObject.h>
#include <gobjects/AnnotationSettings.h>
#include <selection/DNASequenceSelection.h>
#include <util_gui/GUIUtils.h>
#include <datatype/AnnotationData.h>

#include <QtGui/QMenu>
#include <QtGui/QToolTip>

namespace GB2 {

#define MIN_ANNOTATION_WIDTH 3
#define MIN_SELECTED_ANNOTATION_WIDTH 4

GSequenceLineViewAnnotated::GSequenceLineViewAnnotated(QWidget* p, ADVSequenceObjectContext* ctx) 
: GSequenceLineView(p, ctx)
{
    foreach(AnnotationTableObject* ao, ctx->getAnnotationObjects()) {
		connectAnnotationObject(ao);
	}
	connect(ctx->getAnnotationsSelection(), 
		SIGNAL(si_selectionChanged(AnnotationSelection*, const QList<Annotation*>&, const QList<Annotation*>& )), 
		SLOT(sl_onAnnotationSelectionChanged(AnnotationSelection*, const QList<Annotation*>&, const QList<Annotation*>&)));

	connect(ctx, SIGNAL(si_annotationObjectAdded(AnnotationTableObject*)), SLOT(sl_onAnnotationObjectAdded(AnnotationTableObject*)));
	connect(ctx, SIGNAL(si_annotationObjectRemoved(AnnotationTableObject*)), SLOT(sl_onAnnotationObjectRemoved(AnnotationTableObject*)));
    
    connect(AppContext::getAnnotationsSettingsRegistry(),
        SIGNAL(si_annotationSettingsChanged(const QStringList&)),
        SLOT(sl_onAnnotationSettingsChanged(const QStringList&)));

}

void GSequenceLineViewAnnotated::connectAnnotationObject(AnnotationTableObject* ao) {
	connect(ao, SIGNAL(si_onAnnotationsAdded(const QList<Annotation*>&)), SLOT(sl_onAnnotationsAdded(const QList<Annotation*>&)));
	connect(ao, SIGNAL(si_onAnnotationsRemoved(const QList<Annotation*>&)), SLOT(sl_onAnnotationsRemoved(const QList<Annotation*>&)));
}

void GSequenceLineViewAnnotated::sl_onAnnotationSettingsChanged(const QStringList&) {
    addUpdateFlags(GSLV_UF_AnnotationsChanged);
    update();
}

void GSequenceLineViewAnnotated::sl_onAnnotationObjectAdded(AnnotationTableObject* o) {
	connectAnnotationObject(o);
	sl_onAnnotationsAdded(o->getAnnotations());
}

void GSequenceLineViewAnnotated::sl_onAnnotationObjectRemoved(AnnotationTableObject* o) {
	o->disconnect(this);
	sl_onAnnotationsRemoved(o->getAnnotations());
}


void GSequenceLineViewAnnotated::sl_onAnnotationsAdded(const QList<Annotation*>& l) {
	registerAnnotations(l);
    addUpdateFlags(GSLV_UF_AnnotationsChanged);
	update();
}

void GSequenceLineViewAnnotated::sl_onAnnotationsRemoved(const QList<Annotation*>& l) {
	unregisterAnnotations(l);
    addUpdateFlags(GSLV_UF_AnnotationsChanged);
    update();
}

void GSequenceLineViewAnnotated::sl_onAnnotationSelectionChanged(AnnotationSelection* as, const QList<Annotation*>& _added, const QList<Annotation*>& _removed) {
    const QSet<AnnotationTableObject*>& aos = ctx->getAnnotationObjects();

    bool changed = false;
    QList<Annotation*> added = ctx->selectRelatedAnnotations(_added);
    QList<Annotation*> removed = ctx->selectRelatedAnnotations(_removed);

	if (added.size() == 1) {
		Annotation* a = added.first();
        if (aos.contains(a->getGObject())) {
            const AnnotationSelectionData* asd = as->getAnnotationData(a);
            ensureVisible(a, asd->locationIdx);
            changed = true;
        }
	} 
	
	if (!changed) {
		foreach(Annotation* a, added) {
			if (aos.contains(a->getGObject()) && isAnnotationVisible(a)) {
				changed = true;
				break;
			}
		}
		if (!changed) {
			foreach(Annotation* a, removed) {
				if (aos.contains(a->getGObject()) && isAnnotationVisible(a)) {
					changed = true;
					break;
				}
			}
		}
	}

    if (changed) {
        addUpdateFlags(GSLV_UF_SelectionChanged);
		update();
	}
}

bool GSequenceLineViewAnnotated::isAnnotationVisible(Annotation* a) const  {
	foreach(const LRegion& r, a->getLocation()) {
		if (visibleRange.intersects(r)) {
			return true;
		}
	}
	return false;
}

QList<AnnotationSelectionData> GSequenceLineViewAnnotated::selectAnnotationByCoord(const QPoint& p) const {
    QList<AnnotationSelectionData> res;
	GSequenceLineViewAnnotatedRenderArea* ra = (GSequenceLineViewAnnotatedRenderArea*)renderArea;
    AnnotationSettingsRegistry* asr = AppContext::getAnnotationsSettingsRegistry();
    int pos = ra->coordToPos(p.x());
    int dPos = 0;
    if (visibleRange.len > renderArea->width()) {
	    float scale = renderArea->getCurrentScale();
        dPos = (int) ( 1 / (scale) );
        assert(dPos < seqLen);
    }
    LRegion reg(pos-dPos, 1 + 2*dPos);
    foreach(AnnotationTableObject* ao, ctx->getAnnotationObjects()) {
		foreach(Annotation* a, ao->getAnnotations()) {
            const QList<LRegion>& location = a->getLocation();
			for (int i=0, n = location.size(); i<n; i++) {
				const LRegion& l = location[i];
                if (l.intersects(reg) || l.endPos() == reg.startPos) {
                    bool ok = true;
                    if (l.endPos() == pos || pos == l.startPos) {//now check pixel precise coords for boundaries
                        int x1 = ra->posToCoord(l.startPos, true);
                        int x2 = ra->posToCoord(l.endPos(), true);
                        ok = p.x() <= x2 && p.x() >= x1;
                    }
                    if (ok) {
                        const AnnotationSettings* as = asr->getSettings(a->getAnnotationName());
                        if (as->visible) {
                            LRegion ry = ra->getAnnotationYRange(a, l, as);
                            if (ry.contains(p.y())) {
                                res.append(AnnotationSelectionData(a, i));
                            }
                        }
                    }
				}
			}
		}
	}
    return res;
}

void GSequenceLineViewAnnotated::mousePressEvent(QMouseEvent * me) {
    setFocus();
    QPoint p = toRenderAreaPoint(me->pos());
    if (renderArea->rect().contains(p) && me->button()== Qt::LeftButton) {
        bool expandAnnotationSelectionToSequence = (me->modifiers() & Qt::ALT) != 0;
        if (!(me->modifiers() & Qt::SHIFT)) {
            ctx->getAnnotationsSelection()->clear();
        }
        QList<AnnotationSelectionData> selected = selectAnnotationByCoord(p);
        if (!selected.isEmpty()) {		
            AnnotationSelectionData* asd = &selected.first();
            if (selected.size() > 1) {
                AnnotationSettingsRegistry* asr = AppContext::getAnnotationsSettingsRegistry();
                QMenu popup;
                foreach(const AnnotationSelectionData& as, selected) {
                    const QList<LRegion>& location = as.annotation->getLocation();
                    const LRegion& r = location[qMax(0, as.locationIdx)];
                    QString text = as.annotation->getAnnotationName()
                                   + QString(" [%1, %2]").arg(QString::number(r.startPos+1)).arg(QString::number(r.endPos()));
                    const AnnotationSettings* asettings = asr->getSettings(as.annotation->getAnnotationName());
                    QIcon icon = GUIUtils::createSquareIcon(asettings->color, 10);
                    popup.addAction(icon, text);
                }
                QAction* a = popup.exec(QCursor::pos());
                if (a == NULL) {
                    asd = NULL;
                } else {
                    int idx = popup.actions().indexOf(a);
                    asd = &selected[idx];
                }
            }
            if (asd!=NULL) {
                //add to annotation selection
                ctx->getAnnotationsSelection()->addToSelection(asd->annotation, asd->locationIdx);

                //select region
                if (expandAnnotationSelectionToSequence) {
                    assert(asd->locationIdx >=0 && asd->locationIdx <= asd->annotation->getLocation().size());
                    const LRegion& expandRegion = asd->annotation->getLocation().at(asd->locationIdx);
                    ctx->getSequenceSelection()->clear();
                    ctx->getSequenceSelection()->addRegion(expandRegion);
                }
            }
        }
    }
	GSequenceLineView::mousePressEvent(me);
}

//////////////////////////////////////////////////////////////////////////
/// Renderer

GSequenceLineViewAnnotatedRenderArea::GSequenceLineViewAnnotatedRenderArea(GSequenceLineViewAnnotated* d, bool overlap) : GSequenceLineViewRenderArea(d)
{
    annotationsCanOverlap = overlap;
	afNormal = new QFont("Courier", 10);
	afSmall = new QFont("Arial", 8);
	
	afmNormal = new QFontMetrics(*afNormal);
	afmSmall = new QFontMetrics(*afSmall);
	
	afNormalCharHeight = afmNormal->width('w');
	afNormalCharWidth = afmNormal->height();

	afSmallCharHeight = afmSmall->width('w');
	afSmallCharWidth = afmSmall->height();
}

GSequenceLineViewAnnotatedRenderArea::~GSequenceLineViewAnnotatedRenderArea() {
	delete afmNormal;
	delete afNormal;
	delete afmSmall;
	delete afSmall;
}

void GSequenceLineViewAnnotatedRenderArea::drawAnnotations(QPainter& p) {
    drawAnnotations(p, DrawAnnotationPass_DrawFill); 
    drawAnnotations(p, DrawAnnotationPass_DrawBorder); 
}

void GSequenceLineViewAnnotatedRenderArea::drawAnnotation(QPainter& p, DrawAnnotationPass pass, Annotation* a, const QPen& borderPen, bool selected) {
    const GSequenceLineViewAnnotated::DrawSettings& drawSettings = getGSequenceLineViewAnnotated()->getDrawSettings();
    const LRegion& vr = view->getVisibleRange();
    AnnotationSettingsRegistry* asr = AppContext::getAnnotationsSettingsRegistry();

    const QString& aName = a->getAnnotationName();
    const AnnotationSettings* as = asr->getSettings(aName);

    if (!as->visible && (pass == DrawAnnotationPass_DrawFill || !selected)) {
        return;
    }

    foreach(const LRegion& r, a->getLocation()) {
        LRegion y = getAnnotationYRange(a, r, as);
        if (y.startPos <  0){
            continue;
        }
        LRegion visibleLocation = r.intersect(vr);
        if (visibleLocation.isEmpty()) {
            continue;
        }
        float x1f = posToCoordF(visibleLocation.startPos);
        float x2f = posToCoordF(visibleLocation.endPos());
        assert(x2f>=x1f);

        int rw = qMax(selected ? MIN_SELECTED_ANNOTATION_WIDTH : MIN_ANNOTATION_WIDTH, qRound(x2f-x1f));
        int x1 = qRound(x1f);
        if (pass == DrawAnnotationPass_DrawFill) {
            QColor fillColor = as->color;
            p.fillRect(x1, y.startPos, rw, y.len, fillColor);
        } else {
            assert(pass == DrawAnnotationPass_DrawBorder);
            p.setPen(borderPen);
            if (rw > MIN_ANNOTATION_WIDTH) {
                QRect annotationRect(x1, y.startPos, rw, y.len);
                p.drawRect(annotationRect);
                if (drawSettings.drawAnnotationArrows) {
                    drawAnnotationArrow(p, annotationRect, a->isOnComplementStrand());
                }

                if (drawSettings.drawAnnotationNames) {
                    drawBoundedText(p, annotationRect, aName);
                }
            }
            drawAnnotationConnections(p, a, as);
        } 
    } 
}


void GSequenceLineViewAnnotatedRenderArea::drawAnnotations(QPainter& p, DrawAnnotationPass pass) {
    ADVSequenceObjectContext* ctx = view->getSequenceContext();

    QPen pen1(Qt::SolidLine);
    pen1.setWidth(1);

    foreach(AnnotationTableObject* ao, ctx->getAnnotationObjects()) {
		foreach(Annotation* a, ao->getAnnotations()) {
            drawAnnotation(p, pass, a, pen1);
        }
	}
}

void GSequenceLineViewAnnotatedRenderArea::drawAnnotationsSelection(QPainter& p) {
    ADVSequenceObjectContext* ctx = view->getSequenceContext();

    QPen pen1(Qt::SolidLine);
    pen1.setWidth(1);

    QPen pen2(Qt::SolidLine);
    pen2.setWidth(2);

    foreach(const AnnotationSelectionData& asd, ctx->getAnnotationsSelection()->getSelection()) {
        AnnotationTableObject* o = asd.annotation->getGObject();
        if (ctx->getAnnotationObjects().contains(o)) {
            if (annotationsCanOverlap) {
                drawAnnotation(p, DrawAnnotationPass_DrawFill,   asd.annotation, pen1, true);
            }
            drawAnnotation(p, DrawAnnotationPass_DrawBorder, asd.annotation, pen2, true);
        }
    }
}


void GSequenceLineViewAnnotatedRenderArea::drawBoundedText(QPainter& p, const QRect& r, const QString& text) const {
    if (afSmallCharWidth > r.width()) {
		return;
	}
	QFont *font = afNormal;
	QFontMetrics* fm = afmNormal;
	int ch = afNormalCharHeight;
	if (fm->width(text) > r.width()) {
		font = afSmall;
		fm = afmSmall;
		ch = afSmallCharHeight;
	}
	int len = text.length();
	int textWidth = fm->width(text, len);
	while (textWidth > r.width()) {
		len--;
		textWidth = fm->width(text, len);
	}
	if (len == 0) {
		return;
	}

	p.setFont(*font);

	int dx = (r.width() - textWidth) / 2;
	int dy = (r.height() - ch) / 2;
	int x = r.left() + dx;
	int y = r.bottom() - dy;
	p.drawText(x, y, text.left(len));
}

#define FEATURE_ARROW_WIDTH 4
#define MIN_FEATURE_RECT_WIDTH 2
void GSequenceLineViewAnnotatedRenderArea::drawAnnotationArrow(QPainter& p, const QRect& rect, bool leftArrow) {
	if (rect.width() <= MIN_FEATURE_RECT_WIDTH) {
		return;
	}
	int y = rect.top();
	int x = leftArrow ? rect.left() - 1 : rect.right() + 1;
	int dx = leftArrow ? -FEATURE_ARROW_WIDTH : FEATURE_ARROW_WIDTH;
	p.drawLine(x, y, x+dx, y + (rect.height()+1)/2);
	p.drawLine(x, y + rect.height(), x+dx, y + (rect.height()+1)/2);
}

#define MAX_VIRTUAL_RANGE 10000

void GSequenceLineViewAnnotatedRenderArea::drawAnnotationConnections(QPainter& p, Annotation* a, const AnnotationSettings* as) {
	if (a->getLocation().size() < 2) {
		return;
	}
	const GSequenceLineViewAnnotated::DrawSettings& drawSettings = getGSequenceLineViewAnnotated()->getDrawSettings();
	const LRegion& visibleRange = view->getVisibleRange();
	const LRegion* prev = NULL;
	int dx1 = 0;
	int dx2 = 0;
	if (drawSettings.drawAnnotationArrows) {
		if (a->isOnComplementStrand()) {
			dx2 = - FEATURE_ARROW_WIDTH;
		} else {
			dx1 = FEATURE_ARROW_WIDTH;
		}
	}
	foreach(const LRegion& r, a->getLocation()) {
		LRegion y = getAnnotationYRange(a, r, as);
		if (prev!=NULL) {
			int prevPos = prev->endPos();
			int pos = r.startPos;
			int min = qMin(prevPos, pos);
			int max = qMax(prevPos, pos);
			if (visibleRange.intersects(LRegion(min, max-min))) {
				int x1 = posToCoord(prevPos, true) + dx1;
				int x2 = posToCoord(pos, true) + dx2;
				if (qAbs(x2-x1) > 1) {
                    x1 = qBound(-MAX_VIRTUAL_RANGE, x1, MAX_VIRTUAL_RANGE); //qt4.4 crashes in line clipping alg for extremely large X values
                    x2 = qBound(-MAX_VIRTUAL_RANGE, x2, MAX_VIRTUAL_RANGE);
					int midX = (x1 + x2) / 2;
					LRegion pyr = getAnnotationYRange(a, *prev, as);
					LRegion yr = getAnnotationYRange(a, r, as);
					int y1 = pyr.startPos;
					int dy1 = pyr.len/2;
					int y2 = yr.startPos;
					int dy2 = yr.len/2;
					int midY = qMin(y1, y2);
                    p.drawLine(x1, y1+dy1, midX, midY);
					p.drawLine(midX, midY, x2, y2+dy2);
				}
			}
		}
		prev = &r;
	}
}


void GSequenceLineViewAnnotated::ensureVisible(Annotation* a, int locationIdx) {
	const QList<LRegion>& location = a->getLocation();
	assert(locationIdx < location.size());
	if (locationIdx == -1) {
		foreach(const LRegion& r, location) {
			if (visibleRange.intersects(r)) {
				return;
			}
		}
	}
	const LRegion& region = location[qMax(0, locationIdx)];
	if (!visibleRange.intersects(region)) {
        int pos = a->isOnComplementStrand() ? region.endPos() : region.startPos;
        setCenterPos(qBound(0, pos, seqLen-1));
	}
}

bool GSequenceLineViewAnnotated::event(QEvent* e) {
	if (e->type() == QEvent::ToolTip) {
		QHelpEvent* he = static_cast<QHelpEvent*>(e);
		QString tip = createToolTip(he);
		if (!tip.isEmpty()) {
			QToolTip::showText(he->globalPos(), tip);
		}
		return true;
	}
	return GSequenceLineView::event(e);
}

QString GSequenceLineViewAnnotated::createToolTip(QHelpEvent* e) {
	const int ROWS_LIMIT = 25;
	QList<AnnotationSelectionData> la = selectAnnotationByCoord(e->pos());
	if (la.isEmpty()) return QString();
	QString tip = "<table>";
	int rows = 0;
	if (la.size() > 1) {
		foreach(AnnotationSelectionData ad, la) {
			rows += ad.annotation->getQualifiers().size() + 1;
		}
	}
	bool skipDetails = rows > ROWS_LIMIT;
	rows = 0;
	foreach(AnnotationSelectionData ad, la) {
		if (++rows > ROWS_LIMIT) break;
		QString aname = ad.annotation->getAnnotationName();
		QColor acl = AppContext::getAnnotationsSettingsRegistry()->getSettings(aname)->color;
		tip += "<tr><td bgcolor="+acl.name()+" bordercolor=black width=15></td><td><big>"+aname+"</big></td></tr>";
		if (ad.annotation->getQualifiers().size() !=0) {
			if (skipDetails) {
				tip += "<tr><td/><td>...</td>";
				rows++;
			} else {
				tip += "<tr><td/><td>"; 
				tip += ad.annotation->getQualifiersTip(ROWS_LIMIT - rows);
				tip += "</td></tr>";
				rows += ad.annotation->getQualifiers().size();
			}
		}
	}
	tip += "</table>";
    if (rows > ROWS_LIMIT) {
        tip += "<hr> <div align=center>"+tr("etc ...")+"</div>"; 
    }
	return tip;
}

bool GSequenceLineViewAnnotatedRenderArea::isAnnotationSelectionInVisibleRange() const {
    const QSet<AnnotationTableObject*>& aos = view->getSequenceContext()->getAnnotationObjects();
    AnnotationSelection* as = view->getSequenceContext()->getAnnotationsSelection();
    foreach(const AnnotationSelectionData& asd, as->getSelection()) {
        if (!aos.contains(asd.annotation->getGObject())) {
            continue;
        }
        if (getGSequenceLineViewAnnotated()->isAnnotationVisible(asd.annotation)) {
            return true;
        }
    }
    return false;
}


} // namespace
