/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "AnnotatedDNAViewTasks.h"
#include "AnnotatedDNAView.h"
#include "AnnotatedDNAViewFactory.h"
#include "AnnotatedDNAViewState.h"

#include "GSequenceLineView.h"

#include <gobjects/DNASequenceObject.h>
#include <gobjects/AnnotationTableObject.h>
#include <gobjects/GObjectTypes.h>
#include <gobjects/GObjectRelationRoles.h>
#include <gobjects/GObjectUtils.h>

#include <core_api/Log.h>
#include <core_api/AppContext.h>
#include <core_api/ProjectModel.h>
#include <core_api/DocumentModel.h>

#include <QtCore/QSet>

namespace GB2 {

/* TRANSLATOR GB2::AnnotatedDNAView */
/* TRANSLATOR GB2::ObjectViewTask */

//////////////////////////////////////////////////////////////////////////
/// open new view

//opens a single view for all sequence object in the list of sequence objects related to the objects in the list
OpenAnnotatedDNAViewTask::OpenAnnotatedDNAViewTask(QList<GObject*> objects) 
: ObjectViewTask(AnnotatedDNAViewFactory::ID)
{
    foreach(GObject* obj, objects) {
        DNASequenceObject* dnaObj = qobject_cast<DNASequenceObject*>(obj);
        if (dnaObj != NULL) {
            dnaObjects.append(dnaObj);
            continue;
        }
        //look for related object
        QSet<GObject*> relatedDnaObjs = GObjectUtils::selectRelations(obj, GObjectType::null, 
            GObjectRelationRole::SEQUENCE, GObjectUtils::findAllObjects(GObjectTypes::DNA_SEQUENCE));

        foreach(GObject* robj, relatedDnaObjs) {
            dnaObj = qobject_cast<DNASequenceObject*>(robj);
            if (dnaObj!=NULL && !dnaObjects.contains(dnaObj)) {
                dnaObjects.append(dnaObj);
            }
        }
    }
}


OpenAnnotatedDNAViewTask::OpenAnnotatedDNAViewTask(DNASequenceObject* _obj) 
: ObjectViewTask(AnnotatedDNAViewFactory::ID)
{
    assert(_obj!=NULL);
    dnaObjects.append(_obj);
}

OpenAnnotatedDNAViewTask::OpenAnnotatedDNAViewTask(Document* doc) 
: ObjectViewTask(AnnotatedDNAViewFactory::ID)
{
	if (!doc->isLoaded()) {
		documentsToLoad.append(doc);
	} else {
		assert(0);
	}

}

#define MAX_SEQ_OBJS_PER_VIEW 20

void OpenAnnotatedDNAViewTask::open() {
	if (stateInfo.hasErrors() || (dnaObjects.isEmpty() && documentsToLoad.isEmpty())) {
		return;
	}
	if (dnaObjects.isEmpty()) {
		Document* doc = documentsToLoad.first();
		QList<GObject*> objects = doc->findGObjectByType(GObjectTypes::DNA_SEQUENCE);
		if (objects.isEmpty()) {
			return;
		}
        DNASequenceObject* dnaObject = qobject_cast<DNASequenceObject*>(objects.first());
        assert(dnaObject!=NULL);
        dnaObjects.append(dnaObject);
	}
    QList<DNASequenceObject*> finalObjs;
    foreach(DNASequenceObject* obj, dnaObjects) {
        if (obj!=NULL) {
            finalObjs.append(obj);
            if (finalObjs.size() == MAX_SEQ_OBJS_PER_VIEW) { 
                break; //TODO: notify user about this limit!
            }
        }
    }
    if (finalObjs.isEmpty()) { //object was removed asynchronously with the task
        return;
    }
    DNASequenceObject* refObj = finalObjs.first();
	viewName = GObjectViewUtils::genUniqueViewName(refObj->getDocument(), refObj);

    AnnotatedDNAViewFactory* f = (AnnotatedDNAViewFactory*)AppContext::getObjectViewFactoryRegistry()->getFactoryById(AnnotatedDNAViewFactory::ID);
	assert(f!=NULL);
	AnnotatedDNAView* v = new AnnotatedDNAView(viewName, finalObjs);
	GObjectViewWindow* w = new GObjectViewWindow(v, viewName, false);
	MWMDIManager* mdiManager = 	AppContext::getMainWindow()->getMDIManager();
	mdiManager->addMDIWindow(w);

}

//////////////////////////////////////////////////////////////////////////
// open view from state
static QSet<Document*> selectDocuments(Project* p, const QList<GObjectReference>& refs) {
	QSet<Document*> res;
	foreach(const GObjectReference& r, refs) {
		Document* doc = p->findDocumentByURL(r.docUrl);
		if (doc!=NULL) {
			res.insert(doc);
		}
	}
	return res;
}

/*
TODO: remove?
static GObject* findObject(Project* p, const GObjectReference& r) {
	Document* doc = p->findDocumentByURL(r.docUrl);
	if (doc == NULL) {
		return NULL;
	}
	GObject* obj = doc->findGObjectByName(r.objName);
	return obj;
}*/


OpenSavedAnnotatedDNAViewTask::OpenSavedAnnotatedDNAViewTask(const QString& viewName, const QVariantMap& stateData) 
: ObjectViewTask(AnnotatedDNAViewFactory::ID, viewName, stateData)
{
	AnnotatedDNAViewState state(stateData);
    QList<GObjectReference> refs = state.getSequenceObjects();
    if (refs.isEmpty()) {
        stateIsIllegal = true;
        stateInfo.error = ObjectViewTask::tr("no_sequence_info");
        return;
    }
 	foreach(const GObjectReference& ref, refs) {
        Document* doc = AppContext::getProject()->findDocumentByURL(ref.docUrl);
	    if (doc == NULL) {
		    stateIsIllegal = true;
		    stateInfo.error = ObjectViewTask::tr("document_not_found_%1").arg(ref.docUrl);
		    return;
	    }
	    if (!doc->isLoaded()) {
		    documentsToLoad.append(doc);
	    }
    }
	
	QSet<Document*> adocs = selectDocuments(AppContext::getProject(), state.getAnnotationObjects());
	foreach(Document* adoc, adocs) {
		if (!adoc->isLoaded()) {
			documentsToLoad.append(adoc);
		}
	}
}

void OpenSavedAnnotatedDNAViewTask::open() {
	if (stateInfo.hasErrors()) {
		return;
	}
	AnnotatedDNAViewState state(stateData);
    QList<DNASequenceObject*> sequenceObjects;
    foreach(const GObjectReference& ref, state.getSequenceObjects()) {
        Document* doc = AppContext::getProject()->findDocumentByURL(ref.docUrl);
    	if (doc == NULL) {
	    	stateIsIllegal = true;
		    stateInfo.error = ObjectViewTask::tr("document_not_found_%1").arg(ref.docUrl);
		    return;
	    }
	    GObject* obj = doc->findGObjectByName(ref.objName);
	    if (obj == NULL || obj->getGObjectType() != GObjectTypes::DNA_SEQUENCE) {
		    stateIsIllegal = true;
		    stateInfo.error = tr("dna_object_not_found_%1").arg(ref.objName);
		    return;
	    }
        DNASequenceObject* dnaObj= qobject_cast<DNASequenceObject*>(obj);
        sequenceObjects.append(dnaObj);
    }
    AnnotatedDNAViewFactory* f = (AnnotatedDNAViewFactory*)AppContext::getObjectViewFactoryRegistry()->getFactoryById(AnnotatedDNAViewFactory::ID);
	assert(f!=NULL);
	AnnotatedDNAView* v = new AnnotatedDNAView(viewName, sequenceObjects);
	GObjectViewWindow* w = new GObjectViewWindow(v, viewName, true);
	MWMDIManager* mdiManager = 	AppContext::getMainWindow()->getMDIManager();
	mdiManager->addMDIWindow(w);
    v->updateState(state);
}


//////////////////////////////////////////////////////////////////////////
// update
UpdateAnnotatedDNAViewTask::UpdateAnnotatedDNAViewTask(AnnotatedDNAView* v, const QString& stateName, const QVariantMap& stateData) 
: ObjectViewTask(v, stateName, stateData)
{
}

void UpdateAnnotatedDNAViewTask::update() {
	if (view.isNull()) {
		return; //view was closed;
	}

	AnnotatedDNAView* aview = qobject_cast<AnnotatedDNAView*>(view.data());
    assert(aview!=NULL);
	
    AnnotatedDNAViewState state(stateData);
    aview->updateState(state);
}



} // namespace
