/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "ADVSingleSequenceWidget.h"
#include "ADVSequenceObjectContext.h"
#include "AnnotatedDNAView.h"

#include <gobjects/DNASequenceObject.h>
#include <selection/DNASequenceSelection.h>

#include <util_gui/RangeSelector.h>
#include <util_gui/PositionSelector.h>
#include <util_gui/GUIUtils.h>


#include "ADVConstants.h"
#include "DetView.h"
#include "PanView.h"

#include <QtGui/QApplication>
#include <QtGui/QToolButton>
#include <QtGui/QWidgetAction>
#include <QtGui/QDialog>

namespace GB2 {
#define ADV_HEADER_HEIGHT 24

ADVSingleSequenceWidget::ADVSingleSequenceWidget(ADVSequenceObjectContext* seqCtx, AnnotatedDNAView* ctx) : ADVSequenceWidget(ctx) {
    seqContexts.append(seqCtx);
    linesLayout = new QVBoxLayout();
    linesLayout->setMargin(0);
    linesLayout->setSpacing(0);
    setLayout(linesLayout);
    headerWidget = new ADVSingleSequenceHeaderWidget(this);
    headerWidget->installEventFilter(this);
    linesLayout->addWidget(headerWidget);


    toggleViewAction = new QAction(tr("show_view"), this);
    toggleViewAction->setCheckable(true);
    toggleViewAction->setChecked(true);
    connect(toggleViewAction, SIGNAL(triggered(bool)), SLOT(sl_toggleView(bool)));

    togglePanViewAction = new QAction(tr("show_panview"), this);
    togglePanViewAction->setCheckable(true);
    togglePanViewAction->setChecked(true);
    connect(togglePanViewAction, SIGNAL(triggered(bool)), SLOT(sl_togglePanView(bool)));

    toggleDetViewAction = new QAction(tr("show_detview"), this);
    toggleDetViewAction->setCheckable(true);
    toggleDetViewAction->setChecked(true);
    connect(toggleDetViewAction, SIGNAL(triggered(bool)), SLOT(sl_toggleDetView(bool)));

    selectRangeAction = new QAction(QIcon(":/core/images/select_region.png"), tr("range_selection"), this);
    connect(selectRangeAction, SIGNAL(triggered()), SLOT(sl_onSelectRange()));

    zoomToRangeAction = new QAction(QIcon(":/core/images/zoom_reg.png"), tr("range_zoom"), this);
    connect(zoomToRangeAction, SIGNAL(triggered()), SLOT(sl_zoomToRange()));

    init();
}

void ADVSingleSequenceWidget::init() {
    ADVSequenceObjectContext* seqCtx = getSequenceContext();
    detView = new DetView(this, seqCtx);
    addSequenceView(detView);

    panView = new PanView(this, seqCtx);
    connect(panView, SIGNAL(si_centerPosition(int)), SLOT(sl_onLocalCenteringRequest(int)));

    addSequenceView(panView);

    panView->setFrameView(detView);
    
    setFixedHeight(linesLayout->minimumSize().height());

    QToolBar* hBar = headerWidget->getToolBar();

    if (seqCtx->getAminoTT()!=NULL) {
        QToolButton* tt = new QToolButton(hBar);
        QMenu* ttMenu = seqCtx->createTranslationsMenu();
        tt->setDefaultAction(ttMenu->menuAction());
        tt->setPopupMode(QToolButton::InstantPopup);
        hBar->addWidget(tt);
        tbMenues.append(ttMenu);
    }
    hBar->addAction(selectRangeAction);
    hBar->addSeparator();

    QAction* showComplementAction = detView->getShowComplementAction();
    if (showComplementAction->isEnabled()) {
        hBar->addAction(showComplementAction);
    }
    QAction* showTranslationAction = detView->getShowTranslationAction();
    if (showTranslationAction->isEnabled()) {
        hBar->addAction(showTranslationAction);
        hBar->addSeparator();
    }

    hBar->addAction(panView->getZoomInAction());
    hBar->addAction(panView->getZoomOutAction());
    hBar->addAction(panView->getZoomToSelectionAction());
    hBar->addAction(zoomToRangeAction);
    hBar->addAction(panView->getZoomToSequenceAction());
}


ADVSingleSequenceWidget::~ADVSingleSequenceWidget() {
    foreach(QMenu* m, tbMenues) {
        delete m;
    }
}

void ADVSingleSequenceWidget::setViewCollapsed(bool v) {
    toggleViewAction->disconnect(this);
    toggleViewAction->setChecked(!v);
    connect(toggleViewAction, SIGNAL(triggered(bool)), SLOT(sl_toggleView(bool)));

    foreach(GSequenceLineView* seqView, lineViews) {
        bool visible = !isViewCollapsed();
        if (seqView == panView) {
            visible = visible && !isPanViewCollapsed();
        } else if (seqView == detView) {
            visible = visible && !isDetViewCollapsed();
        }
        seqView->setVisible(visible);
    }
    updateMinMaxHeight();
}

void ADVSingleSequenceWidget::setPanViewCollapsed(bool v) {
    togglePanViewAction->disconnect(this);
    togglePanViewAction->setChecked(!v);
    connect(togglePanViewAction, SIGNAL(triggered(bool)), SLOT(sl_togglePanView(bool)));

    bool visible = !isViewCollapsed() && !isPanViewCollapsed();
    panView->setVisible(visible);
    updateMinMaxHeight();
}

void ADVSingleSequenceWidget::setDetViewCollapsed(bool v) {
    toggleDetViewAction->disconnect(this);
    toggleDetViewAction->setChecked(!v);
    connect(toggleDetViewAction, SIGNAL(triggered(bool)), SLOT(sl_toggleDetView(bool)));

    bool visible = !isViewCollapsed() && !isDetViewCollapsed();
    detView->setVisible(visible);
    updateMinMaxHeight();
}

void ADVSingleSequenceWidget::addSequenceView(GSequenceLineView* v) {
    assert(!lineViews.contains(v));
    lineViews.append(v);
    linesLayout->insertWidget(1, v);
    v->setVisible(true);
    v->installEventFilter(this);
    updateMinMaxHeight();
    connect(v, SIGNAL(destroyed(QObject*)), SLOT(sl_onViewDestroyed(QObject*)));
}

void ADVSingleSequenceWidget::removeSequenceView(GSequenceLineView* v, bool deleteView) {
    assert(lineViews.contains(v));
    lineViews.removeOne(v);
    linesLayout->removeWidget(v);
    v->setVisible(false);
    v->disconnect(this);
    v->removeEventFilter(this);
    if (deleteView) {
        delete v;
    } 
    updateMinMaxHeight();
}

void ADVSingleSequenceWidget::sl_onViewDestroyed(QObject* o) {
    GSequenceLineView* v = (GSequenceLineView*)o;
    bool r = lineViews.removeOne(v);
    linesLayout->removeWidget(v);//need here for updateMinMaxHeight
    assert(r);
    Q_UNUSED(r);
    updateMinMaxHeight();
}

void ADVSingleSequenceWidget::centerPosition(int pos, QWidget* skipView) {
    foreach(GSequenceLineView* v, lineViews) {
        if (v == skipView) {
            continue;
        }
        v->setCenterPos(pos);
    }
}

void ADVSingleSequenceWidget::updateMinMaxHeight() {
    int height = linesLayout->minimumSize().height();
    setFixedHeight(height);
}


void ADVSingleSequenceWidget::addZoomMenu(const QPoint& globalPos, QMenu* m) {
    GSequenceLineView* lineView = findSequenceViewByPos(globalPos);
    if (lineView == NULL) {
        return;
    }

    QAction* first = m->actions().isEmpty() ? NULL : m->actions().first();

    QAction * zoomInAction = lineView->getZoomInAction();
    QAction * zoomOutAction = lineView->getZoomOutAction();
    QAction * zoomToSelection = lineView->getZoomToSelectionAction();
    QAction * zoomToSequence = lineView->getZoomToSequenceAction();

    if (zoomInAction == NULL && zoomOutAction == NULL && zoomToSelection == NULL && zoomToSequence == NULL) {
        return;
    }

    QMenu* zm = m->addMenu(tr("zoom_menu"));

    if (zoomInAction!=NULL) {
        zm->insertAction(first, zoomInAction);
    }
    if (zoomOutAction!=NULL) {
        zm->insertAction(first, zoomOutAction);
    }
    if (zoomToSelection!=NULL) {
        zm->insertAction(first, zoomToSelection);
    }
    if (lineView == panView || lineView->getConherentRangeView() == panView) {
        zm->insertAction(first, zoomToRangeAction);
    }
    if (zoomToSequence!=NULL) {
        zm->insertAction(first, zoomToSequence);
    }
    zm->menuAction()->setObjectName(ADV_MENU_ZOOM);
    m->addSeparator();
}

GSequenceLineView* ADVSingleSequenceWidget::findSequenceViewByPos(const QPoint& globalPos) const {
    Q_UNUSED(globalPos);
    assert(0);
    return NULL;
}

int ADVSingleSequenceWidget::getSequenceLen() const {
    return getSequenceContext()->getSequenceLen();
}


DNATranslation* ADVSingleSequenceWidget::getComplementTT() const {
    ADVSequenceObjectContext* seqCtx = getSequenceContext();
    return seqCtx->getComplementTT();
}

DNATranslation* ADVSingleSequenceWidget::getAminoTT() const {
    ADVSequenceObjectContext* seqCtx = getSequenceContext();
    return seqCtx->getAminoTT();
}

DNASequenceSelection* ADVSingleSequenceWidget::getSequenceSelection() const {
    ADVSequenceObjectContext* seqCtx = getSequenceContext();
    return seqCtx->getSequenceSelection();
}

DNASequenceObject* ADVSingleSequenceWidget::getSequenceObject() const {
    ADVSequenceObjectContext* seqCtx = getSequenceContext();
    return seqCtx->getSequenceObject();
}

GSequenceLineView* ADVSingleSequenceWidget::getPanGSLView() const {
    return panView;
}

GSequenceLineView* ADVSingleSequenceWidget::getDetGSLView() const {
    return detView;
}

void ADVSingleSequenceWidget::buildPopupMenu(QMenu& m) {
    m.insertAction(GUIUtils::findActionAfter(m.actions(), ADV_GOTO_ACTION), getAnnotatedDNAView()->getCreateAnnotationAction());
    m.insertAction(GUIUtils::findActionAfter(m.actions(), ADV_GOTO_ACTION), selectRangeAction);
    ADVSequenceWidget::buildPopupMenu(m);
    foreach(GSequenceLineView* v, lineViews) {
        v->buildPopupMenu(m);
    }
}

bool ADVSingleSequenceWidget::isWidgetOnlyObject(GObject* o) const {
    foreach(GSequenceLineView* v, lineViews) {
        bool ok = v->isWidgetOnlyObject(o);
        if (ok) {
            return true;
        }
    }
    return false;
}


bool ADVSingleSequenceWidget::eventFilter(QObject* o, QEvent* e) {
    QEvent::Type t = e->type();
    if (t == QEvent::Resize) {
        GSequenceLineView* v = qobject_cast<GSequenceLineView*>(o);
        if (lineViews.contains(v)) {
            updateMinMaxHeight();
        }
    } else if (t == QEvent::FocusIn || t == QEvent::MouseButtonPress || t == QEvent::MouseButtonRelease) {
        ctx->setFocusedSequenceWidget(this);
    } 
    return false;
}

void ADVSingleSequenceWidget::sl_onLocalCenteringRequest(int pos) {
    assert(panView == sender());
    detView->setCenterPos(pos);
}

void ADVSingleSequenceWidget::addADVSequenceWidgetAction(ADVSequenceWidgetAction* a) {
    ADVSequenceWidget::addADVSequenceWidgetAction(a);
    if (a->addToBar) {
        QToolBar* tb = headerWidget->getToolBar();
        if (a->menu()!=NULL) {
            QToolButton* tt = new QToolButton(tb);
            tt->setDefaultAction(a);
            tt->setPopupMode(QToolButton::InstantPopup);
            tb->insertWidget(tb->actions().first(), tt);
        } else {
            tb->insertAction(tb->actions().first(), a);
        }
        
    }
}

void ADVSingleSequenceWidget::sl_onSelectRange() {
    QDialog dlg;
    dlg.setModal(true);
    dlg.setWindowTitle(tr("selection_range_title"));
    ADVSequenceObjectContext* ctx = getSequenceContext();
    RangeSelector* rs = new RangeSelector(&dlg, 1, ctx->getSequenceLen(), true);
    int rc = dlg.exec();
    if (rc == QDialog::Accepted) {
        LRegion r(rs->getStart() - 1, rs->getEnd() - rs->getStart() + 1);
        ctx->getSequenceSelection()->clear();
        getSequenceSelection()->addRegion(r);
        if (!detView->getVisibleRange().intersects(r)) {
            detView->setCenterPos(r.startPos);
        }
    }
    delete rs;
}


void ADVSingleSequenceWidget::sl_zoomToRange() {
    QDialog dlg;
    dlg.setModal(true);
    dlg.setWindowTitle(tr("visible_range_title"));
    RangeSelector* rs = new RangeSelector(&dlg, 1, getSequenceLen(), true);

    int rc = dlg.exec();
    if (rc == QDialog::Accepted) {
        LRegion r(rs->getStart() - 1, rs->getEnd() - rs->getStart() + 1);
        panView->setVisibleRange(r);
        detView->setStartPos(r.startPos);
    }
    delete rs;
}

#define SPLITTER_STATE_MAP_NAME  "ADVSI_MAP"
#define PAN_REG_NAME    "PAN_REG"
#define DET_POS_NAME    "DET_POS"

void ADVSingleSequenceWidget::updateState(const QVariantMap& m) {
    QVariantMap map = m.value(SPLITTER_STATE_MAP_NAME).toMap();
    QVariantMap myData = map.value(getActiveSequenceContext()->getSequenceObject()->getGObjectName()).toMap();
    LRegion panReg = myData.value(PAN_REG_NAME).value<LRegion>();
    int detPos = myData.value(DET_POS_NAME).toInt();
    
    LRegion seqRange(0, getActiveSequenceContext()->getSequenceLen());
    if (seqRange.contains(detPos)) {
        detView->setStartPos(detPos);
    }
    if (!panReg.isEmpty() && seqRange.contains(panReg)) {
        panView->setVisibleRange(panReg);
    }
}

void ADVSingleSequenceWidget::saveState(QVariantMap& m) {
    QVariantMap map = m.value(SPLITTER_STATE_MAP_NAME).toMap();
    
    QVariantMap myData;
    myData[PAN_REG_NAME] = QVariant::fromValue<LRegion>(panView->getVisibleRange());
    myData[DET_POS_NAME] = QVariant::fromValue<int>(detView->getVisibleRange().startPos);
    map[getActiveSequenceContext()->getSequenceObject()->getGObjectName()] = myData;
    m[SPLITTER_STATE_MAP_NAME] = map;
}


void ADVSingleSequenceWidget::closeView() {
    DNASequenceObject* dnaObj = getSequenceObject();
    AnnotatedDNAView* v = getAnnotatedDNAView();
    v->removeObject(dnaObj);
}

//////////////////////////////////////////////////////////////////////////
// header

class HBar : public QToolBar {
public:
    HBar(QWidget* w) : QToolBar(w){}
protected:
    void paintEvent(QPaintEvent* pe) {
        Q_UNUSED(pe);
        //do not draw any special toolbar control -> make is merged with parent widget
    }
};

ADVSingleSequenceHeaderWidget::ADVSingleSequenceHeaderWidget(ADVSingleSequenceWidget* p) : QWidget(p), ctx(p) {
    setFixedHeight(ADV_HEADER_HEIGHT);
    setBackgroundRole(QPalette::Window);
    setAutoFillBackground(true);

    connect(ctx->getAnnotatedDNAView(), SIGNAL(si_focusChanged(ADVSequenceWidget*, ADVSequenceWidget*)), 
                                        SLOT(sl_advFocusChanged(ADVSequenceWidget*, ADVSequenceWidget*)));

    //TODO: track focus events (mouse clicks) on toolbar in disabled state and on disabled buttons !!!

    QHBoxLayout* l = new QHBoxLayout();
    l->setSpacing(4);
    l->setContentsMargins(5, 1, 0, 2);

    QString objName = ctx->getSequenceObject()->getGObjectName();
    pixLabel= new QLabel(this);
    QFont f = pixLabel->font();
    if (f.pixelSize() > ADV_HEADER_HEIGHT) {
        f.setPixelSize(ADV_HEADER_HEIGHT-8);
    }
    QIcon objIcon(":/core/images/gobject.png");
    QPixmap pix = objIcon.pixmap(QSize(16, 16), QIcon::Active);
    pixLabel->setPixmap(pix);
    pixLabel->setFont(f);
    pixLabel->setToolTip(objName);
    pixLabel->installEventFilter(this);
    
    int labelWidth = 200;
    QString labelStr = objName;
    QFontMetrics fm(f);
    while(fm.width(labelStr) > labelWidth && labelStr.length() > 3) {
        labelStr = labelStr.left(labelStr.length()-3) + "..";
    }
    nameLabel = new QLabel(labelStr, this);
    nameLabel->setFixedWidth(qMin(labelWidth, fm.width(labelStr)));
    nameLabel->setFont(f);
    nameLabel->setToolTip(objName);

    toolBar = new HBar(this);
    toolBar->layout()->setSpacing(0);
    toolBar->layout()->setMargin(0);

    /// close bar
    closeBar = new HBar(this);
    closeBar->layout()->setSpacing(0);
    closeBar->layout()->setMargin(0);

        
    setLayout(l);

    l->addWidget(pixLabel);
    l->addWidget(nameLabel);
    l->addStretch();
    l->addWidget(toolBar);
    l->addWidget(closeBar);


    connect(toolBar, SIGNAL(actionTriggered(QAction*)), SLOT(sl_actionTriggered(QAction*)));
    connect(closeBar, SIGNAL(actionTriggered(QAction*)), SLOT(sl_actionTriggered(QAction*)));

    populateToolBars();
    updateActiveState();

}

void ADVSingleSequenceHeaderWidget::populateToolBars() {
    // close bar
    widgetStateMenuButton = new QToolButton(this);
    widgetStateMenuButton->setIcon(QIcon(":core/images/adv_widget_menu.png"));
    widgetStateMenuButton->setFixedWidth(20);
    connect(widgetStateMenuButton, SIGNAL(pressed()), SLOT(sl_showStateMenu()));


    closeViewAction = new QAction(tr("Remove sequence"), ctx);
    connect(closeViewAction, SIGNAL(triggered()), SLOT(sl_closeView()));

    closeBar->addWidget(widgetStateMenuButton);
}

bool ADVSingleSequenceHeaderWidget::eventFilter(QObject *o, QEvent *e) {
    if (o == pixLabel && e->type() == QEvent::MouseButtonPress) {
        sl_showStateMenu();
        return true;
    }
    return false;
}

void ADVSingleSequenceHeaderWidget::sl_actionTriggered(QAction* a){
    Q_UNUSED(a);
    ctx->getAnnotatedDNAView()->setFocusedSequenceWidget(ctx);
}

void ADVSingleSequenceHeaderWidget::sl_advFocusChanged(ADVSequenceWidget* prevFocus, ADVSequenceWidget* newFocus) {
    if (prevFocus == ctx || newFocus == ctx) {
        update();
        updateActiveState();
    } 
}
void ADVSingleSequenceHeaderWidget::updateActiveState() {
    bool focused = ctx->getAnnotatedDNAView()->getSequenceWidgetInFocus() == ctx;
    nameLabel->setEnabled(focused);
    pixLabel->setEnabled(focused);
    //toolBar->setEnabled(focused); TODO: click on disabled buttons does not switch focus!
}

void ADVSingleSequenceHeaderWidget::sl_showStateMenu() {
    QPointer<QToolButton> widgetStateMenuButtonPtr(widgetStateMenuButton);

    QMenu m;
    m.addAction(ctx->getToggleViewAction());
    m.addAction(ctx->getTogglePanViewAction());
    m.addAction(ctx->getToggleDetViewAction());
    m.addAction(closeViewAction);
    m.exec(QCursor::pos());

    if (!widgetStateMenuButtonPtr.isNull()) { //if not self closed
        widgetStateMenuButtonPtr->setDown(false);
    }
}


void ADVSingleSequenceHeaderWidget::sl_closeView() {
    ctx->closeView();
}

void ADVSingleSequenceHeaderWidget::mouseDoubleClickEvent(QMouseEvent *e) {
    ctx->getToggleViewAction()->trigger();
    QWidget::mouseDoubleClickEvent(e);
}

void ADVSingleSequenceHeaderWidget::paintEvent(QPaintEvent *e) {
    QWidget::paintEvent(e);

    QPainter p(this);
    p.setPen(QApplication::palette().color(QPalette::Dark));
    p.drawLine(0, height()-1, width(), height()-1);
}

}//namespace

