/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "ADVAlignmentSupport.h"

#include "AnnotatedDNAView.h"
#include "ADVSequenceObjectContext.h"
#include "ADVConstants.h"

#include <core_api/AppContext.h>
#include <core_api/DocumentFormats.h>
#include <core_api/IOAdapter.h>
#include <core_api/ProjectModel.h>
#include <core_api/Task.h>
#include <core_api/DNAAlphabet.h>
#include <core_api/DNATranslation.h>

#include <gobjects/AnnotationTableObject.h>
#include <gobjects/GObjectTypes.h>
#include <gobjects/MAlignmentObject.h>

#include <selection/AnnotationSelection.h>
#include <selection/DNASequenceSelection.h>
#include <selection/GObjectSelection.h>

#include <util_gui/GUIUtils.h>
#include <util_gui/AddNewDocumentDialogImpl.h>

#include <util_ov_msaedit/MSAEditorFactory.h>

#include <QtCore/QSet>
#include <QtGui/QMessageBox>

namespace GB2 {

//TODO: move to DNAExport plugin!    
//TODO: add wizard-like dialog to fit all possible options (translations..)

ADVAlignmentSupport::ADVAlignmentSupport(AnnotatedDNAView* v) : QObject(v){
    ctx = v;

    connect(ctx->getAnnotationsSelection(), 
        SIGNAL(si_selectionChanged(AnnotationSelection*, const QList<Annotation*>&, const QList<Annotation*>& )), 
        SLOT(sl_onAnnotationSelectionChanged(AnnotationSelection*, const QList<Annotation*>&, const QList<Annotation*>&)));

    connect(ctx, SIGNAL(si_sequenceAdded(ADVSequenceObjectContext*)), SLOT(sl_onSequenceContextAdded(ADVSequenceObjectContext*)));
    connect(ctx, SIGNAL(si_sequenceRemoved(ADVSequenceObjectContext*)), SLOT(sl_onSequenceContextRemoved(ADVSequenceObjectContext*)));

    connect(ctx, SIGNAL(si_buildPopupMenu(GObjectView*, QMenu*)), SLOT(sl_buildPopupMenu(GObjectView*, QMenu*)));
    connect(ctx, SIGNAL(si_buildStaticMenu(GObjectView*, QMenu*)), SLOT(sl_buildStaticMenu(GObjectView*, QMenu*)));
    
    annotationsToAlignmentAction = new QAction(QIcon(":core/images/msa.png"), tr("annotation_selection_to_alignment"), this);
    connect(annotationsToAlignmentAction, SIGNAL(triggered()), SLOT(sl_annotationsToAlignment()));

    seqSelectionToAliAction = new QAction(QIcon(":core/images/msa.png"), tr("sequence_selection_to_alignment"), this);
    connect(seqSelectionToAliAction, SIGNAL(triggered()), SLOT(sl_sequenceToAlignment()));

    seqSelectionToAliWithTranslationAction = new QAction(QIcon(":core/images/msa.png"), tr("sequence_selection_to_alignment_with_transl"), this);
    connect(seqSelectionToAliWithTranslationAction, SIGNAL(triggered()), SLOT(sl_sequenceToAlignmentWithTranslation()));

    foreach(ADVSequenceObjectContext* sCtx, ctx->getSequenceContexts()) {
        sl_onSequenceContextAdded(sCtx);
    }
    updateActions();
}

void ADVAlignmentSupport::sl_onSequenceContextAdded(ADVSequenceObjectContext* c) {
    connect(c->getSequenceSelection(), 
        SIGNAL(si_selectionChanged(LRegionsSelection*, const QList<LRegion>&, const QList<LRegion>&)), 
        SLOT(sl_onSequenceSelectionChanged(LRegionsSelection*, const QList<LRegion>&, const QList<LRegion>&)));

    updateActions();
}

void ADVAlignmentSupport::sl_onSequenceContextRemoved(ADVSequenceObjectContext* c) {
    c->disconnect(this);
    updateActions();
}


void ADVAlignmentSupport::sl_onAnnotationSelectionChanged(AnnotationSelection* as, const QList<Annotation*>& added, const QList<Annotation*>& removed) {
    Q_UNUSED(as); Q_UNUSED(added); Q_UNUSED(removed);
    updateActions();
}

void ADVAlignmentSupport::sl_onSequenceSelectionChanged(LRegionsSelection* ss, const QList<LRegion>& added, const QList<LRegion>& removed) {
    Q_UNUSED(ss); Q_UNUSED(added); Q_UNUSED(removed);
    updateActions();
}

static bool allNucleic(const QList<ADVSequenceObjectContext*>& seqs) {
    foreach(const ADVSequenceObjectContext* s, seqs) {
        if (!s->getAlphabet()->isNucleic()) {
            return false;
        }
    }
    return true;
}

void ADVAlignmentSupport::updateActions() {
    bool hasMultipleAnnotationsSelected = ctx->getAnnotationsSelection()->getSelection().size() > 1;
    annotationsToAlignmentAction->setEnabled(hasMultipleAnnotationsSelected);

    int nSelections = 0;
    foreach(ADVSequenceObjectContext* c, ctx->getSequenceContexts()) {
        nSelections += c->getSequenceSelection()->isEmpty() ? 0 : 1;
    }
    bool hasMultiSequenceSelection = nSelections > 1;
    seqSelectionToAliAction->setEnabled(hasMultiSequenceSelection);

    bool allN = allNucleic(ctx->getSequenceContexts());
    seqSelectionToAliWithTranslationAction->setEnabled(hasMultiSequenceSelection && allN);
}


void ADVAlignmentSupport::sl_buildPopupMenu(GObjectView* v, QMenu* m) {
    Q_UNUSED(v);
    QMenu* analyseMenu = GUIUtils::findSubMenu(m, ADV_MENU_EXPORT);
    assert(analyseMenu!=NULL);
    if (ctx->getSequenceContexts().size() > 1) {
        analyseMenu->addAction(seqSelectionToAliAction);
    }
    if (seqSelectionToAliWithTranslationAction->isEnabled()) {
        analyseMenu->addAction(seqSelectionToAliWithTranslationAction);
    }
    analyseMenu->addAction(annotationsToAlignmentAction);
}

void ADVAlignmentSupport::sl_buildStaticMenu(GObjectView* v, QMenu* m) {
    Q_UNUSED(v);
    QMenu* exportMenu = GUIUtils::findSubMenu(m, ADV_MENU_EXPORT);
    assert(exportMenu!=NULL);
    exportMenu->addAction(seqSelectionToAliAction);
    exportMenu->addAction(annotationsToAlignmentAction);
}


static QString genUniqueName(const QSet<QString>& names, QString prefix) {
    if (!names.contains(prefix)) {
        return prefix;
    }
    QString name = prefix;
    int i=0;
    do {
        if (!names.contains(name)) {
            break;
        }
        name = prefix + "_" + QString::number(++i);
    } while(true);
    return name;
}

#define MAX_ALI_MODEL (10*1000*1000)

QString ADVAlignmentSupport::prepareMAFromAnnotations(MAlignment& ma) {
    assert(ma.isEmpty());
    const QList<AnnotationSelectionData>& selection = ctx->getAnnotationsSelection()->getSelection();
    if (selection.size() < 2) {
        return tr("at_least_2_sequences_required");        
    }
    // check that all sequences are present and have the same alphabets
    DNAAlphabet* al = NULL;
    DNATranslation* complTT = NULL;
    foreach(const AnnotationSelectionData& a, selection) {
        AnnotationTableObject* ao = a.annotation->getGObject();
        ADVSequenceObjectContext* seqCtx = ctx->getSequenceContext(ao);
        if (seqCtx == NULL) {
            return tr("no_sequence_object_found");
        }
        if (al == NULL ) {
            al = seqCtx->getAlphabet();
            complTT = seqCtx->getComplementTT();
        } else {
            DNAAlphabet* al2 = seqCtx->getAlphabet();
            //BUG524: support alphabet reduction
            if (al->getType() != al2->getType()) {
                return tr("different_alphabets");                
            } else if (al != al2) {
                al = al->getMap().count(true) >= al2->getMap().count(true) ? al : al2;
            }
        }
    }
  
    int maxLen = 0;
    ma.alphabet = al;
    QSet<QString> names;
    //TODO: check if amino translation is needed
    foreach(const AnnotationSelectionData& a, selection) {
        MAlignmentItem item;
        item.name = genUniqueName(names, a.annotation->getAnnotationName());
        AnnotationTableObject* ao = a.annotation->getGObject();
        ADVSequenceObjectContext* seqCtx = ctx->getSequenceContext(ao);
        const QByteArray& sequence = seqCtx->getSequenceData();
        
        maxLen = qMax(maxLen, a.getSelectedRegionsLen());
        if (maxLen * ma.alignedSeqs.size() > MAX_ALI_MODEL) {
            return tr("alignment_is_too_large");
        }

        AnnotationSelection::getAnnotationSequence(item.sequence, a, MAlignment_GapChar, sequence, complTT, NULL);
        ma.alignedSeqs.append(item);
        names.insert(item.name);
    }

    ma.normalizeModel();
    return "";
}

QString ADVAlignmentSupport::prepareMAFromSequences(MAlignment& ma, bool translate) {
    assert(ma.isEmpty());

    DNAAlphabet* al = translate ? AppContext::getDNAAlphabetRegistry()->findById(BaseDNAAlphabetIds::AMINO_DEFAULT) : NULL;
    
    //derive alphabet
    int nItems = 0;
    foreach(ADVSequenceObjectContext* c, ctx->getSequenceContexts()) {
        if (c->getSequenceSelection()->isEmpty()) {
            continue;
        }
        nItems++;
        DNAAlphabet* seqAl = c->getAlphabet();
        if (al == NULL) {
            al = seqAl;
        } else if (al != seqAl) {
            if (al->isNucleic() && seqAl->isAmino()) {
                translate = true;
                al = seqAl;
            } else if (al->isAmino() && seqAl->isNucleic()) {
                translate = true;
            } else {
                return tr("can't_derive_alignment_alphabet");
            }
        }
    }
    
    if (nItems < 2) { 
        return tr("at_least_2_sequences_required");        
    }

    //cache sequences
    QSet<QString> names;
    QList<MAlignmentItem> sequences;
    int maxLen = 0;
    foreach(ADVSequenceObjectContext* c, ctx->getSequenceContexts()) {
        if (c->getSequenceSelection()->isEmpty()) {
            continue;
        }
        DNAAlphabet* seqAl = c->getAlphabet();
        LRegion r = c->getSequenceSelection()->getSelectedRegions().first();
        const QByteArray& seq = c->getSequenceData();
        maxLen = qMax(maxLen, r.len);
        if (maxLen * sequences.size() > MAX_ALI_MODEL) {
            return tr("alignment_is_too_large");
        }
        QByteArray mid = seq.mid(r.startPos, r.len);
        if (translate && seqAl->isNucleic()) {
            DNATranslation* aminoTT = c->getAminoTT();
            if (aminoTT!=NULL) {
                int len = aminoTT->translate(mid.data(), mid.size());
                mid.resize(len);
            }
        }
        MAlignmentItem mai(genUniqueName(names, c->getSequenceGObject()->getGObjectName()), mid);
        names.insert(mai.name);
        sequences.append(mai);
    }

    ma.alphabet = al;
    ma.alignedSeqs = sequences;
    
    ma.normalizeModel();
    return "";
}


void ADVAlignmentSupport::sl_annotationsToAlignment() {
    selectionToAlignment(true, false);   
}

void ADVAlignmentSupport::sl_sequenceToAlignment() {
    selectionToAlignment(false, false);   
}

void ADVAlignmentSupport::sl_sequenceToAlignmentWithTranslation() {
    selectionToAlignment(false, true);   
}


void ADVAlignmentSupport::selectionToAlignment(bool annotations, bool translate) {
    MAlignment ma;
    QString err = annotations ? prepareMAFromAnnotations(ma) : prepareMAFromSequences(ma, translate);
    if (!err.isEmpty()) {
        QMessageBox::critical(NULL, tr("error"), err);
        return;
    }

    DocumentFormatConstraints c;
    c.mustSupportWrite = true;
    c.supportedObjectTypes.append(GObjectTypes::MULTIPLE_ALIGNMENT);
    AddNewDocumentDialogModel m;
    m.format = BaseDocumentFormats::CLUSTAL_ALN;
    AddNewDocumentDialogImpl::run(NULL, m, c);
    if (!m.successful) {
        return;
    }
    DocumentFormat* format = AppContext::getDocumentFormatRegistry()->getFormatById(m.format);
    IOAdapterFactory* iof = AppContext::getIOAdapterRegistry()->getIOAdapterFactoryById(m.io);
    Project* p = AppContext::getProject();
    Document* doc = format->createNewDocument(iof, m.url);

    MAlignmentObject* obj = new MAlignmentObject(ma, MA_OBJECT_NAME);
    doc->addObject(obj);
    p->addDocument(doc);

    // open MSA editor 
    //BUG525: create registry: default view by object type!!
    GObjectViewFactory* f = AppContext::getObjectViewFactoryRegistry()->getFactoryById(MSAEditorFactory::ID);
    assert(f!=NULL);
    MultiGSelection ms;
    GObjectSelection os; os.addToSelection(obj);
    ms.addSelection(&os);
    AppContext::getTaskScheduler()->registerTopLevelTask(f->createViewTask(ms));

}


} //namespace
