/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "CreateAnnotationWidgetController.h"
#include "GObjectComboBoxController.h"
#include "DialogUtils.h"

#include <core_api/AppContext.h>
#include <core_api/DocumentFormats.h>
#include <core_api/IOAdapter.h>
#include <core_api/ProjectModel.h>
#include <core_api/Task.h>
#include <core_api/GObjectReference.h>
#include <gobjects/GObjectRelationRoles.h>
#include <core_api/Settings.h>
#include <gobjects/GObjectTypes.h>


#include <document_format/GenbankLocationParser.h>
#include <util_gui/AddNewDocumentDialogImpl.h>
#include <util_gui/ProjectTreeController.h>
#include <util_gui/ProjectTreeItemSelectorDialog.h>
#include <util_gui/GUIUtils.h>
#include <util_text/TextUtils.h>

#include <QtGui/QFileDialog>

#define SETTINGS_LAST_USED_ANNOTATION_NAME "create_annotation/last_name"
#define SETTINGS_LASTDIR "create_annotation/last_dir"

namespace GB2 {
/* TRANSLATOR GB2::CreateAnnotationWidgetController */    

CreateAnnotationModel::CreateAnnotationModel() : defaultIsNewDoc(false), hideLocation(false), hideAnnotationName(false) {
	data = new AnnotationData();	
    annotationObject = NULL;

}

CreateAnnotationWidgetController::CreateAnnotationWidgetController(CreateAnnotationModel* m, QObject* p) 
: QObject(p), model(m)
{
    assert(AppContext::getProject()!=NULL);
    assert(m->sequenceObjectRef.isValid());
    w = new QWidget();
    setupUi(w);
    if (m->hideLocation) {
        locationLabel->hide();
        locationEdit->hide();
    }
    if( m->hideAnnotationName ) {
        annotationNameEdit->hide();
        annotationNameLabel->hide();
    }

    QString dir = AppContext::getSettings()->getValue(SETTINGS_LASTDIR).toString();
    if (dir.isEmpty() || !QDir(dir).exists()) {
        dir = QDir::homePath();
    }
    dir+="/";
    QString baseName = "MyDocument";
    QString ext = ".gb";
    QString url = dir + baseName + ext;
    for (int i=1; QFileInfo(url).exists() || AppContext::getProject()->findDocumentByURL(url)!= NULL; i++) {
        url = dir + baseName +"_"+QString::number(i) + ext;
    }
    newFileEdit->setText(url);

    GROUP_NAME_AUTO = CreateAnnotationWidgetController::tr("group_name_auto");

    GObjectComboBoxControllerConstraints occc;
    occc.relationFilter.ref = m->sequenceObjectRef;
    occc.relationFilter.role = GObjectRelationRole::SEQUENCE;
    occc.typeFilter = GObjectTypes::ANNOTATION_TABLE;
    occc.onlyWritable = true;
    occ = new GObjectComboBoxController(this, occc, existingObjectCombo);

    //default field values
    if (!m->data->name.isEmpty()) {
        annotationNameEdit->setText(m->data->name);
    } else if (!m->hideAnnotationName) {
        QString name = AppContext::getSettings()->getValue(SETTINGS_LAST_USED_ANNOTATION_NAME, QString("misc_feature")).toString();
        annotationNameEdit->setText(name);
    }
    annotationNameEdit->selectAll();
    
    groupNameEdit->setText(m->groupName.isEmpty() ? GROUP_NAME_AUTO : m->groupName);

    if (!m->data->location.isEmpty()) {
        QString locationString = Genbank::LocationParser::buildLocationString(m->data);
        locationEdit->setText(locationString);
    }

    if (m->defaultIsNewDoc || existingObjectCombo->count() == 0) {
        newFileRB->setChecked(true);
    }

    connect(newFileButton, SIGNAL(clicked()), SLOT(sl_onNewDocClicked()));
    connect(existingObjectButton, SIGNAL(clicked()), SLOT(sl_onLoadObjectsClicked()));
}

void CreateAnnotationWidgetController::sl_onNewDocClicked() {
    QString openUrl = QFileInfo(newFileEdit->text()).absoluteDir().absolutePath();
    QString filter = DialogUtils::prepareDocumentsFileFilter(BaseDocumentFormats::PLAIN_GENBANK, false);
    QString name = QFileDialog::getSaveFileName(NULL, tr("save_file_dialog_caption"), openUrl, filter);
    if (!name.isEmpty()) {
        newFileEdit->setText(name);	
        AppContext::getSettings()->setValue(SETTINGS_LASTDIR, QFileInfo(name).absoluteDir().absolutePath());
    }
}

void CreateAnnotationWidgetController::sl_onLoadObjectsClicked() {
    ProjectTreeControllerModeSettings s;
    s.allowMultipleSelection = false;
    s.objectTypesToShow.append(GObjectTypes::ANNOTATION_TABLE);
    s.objectRelation.ref = model->sequenceObjectRef;
    s.objectRelation.role = GObjectRelationRole::SEQUENCE;
    s.sorted = true;
    QList<GObject*> objs = ProjectTreeItemSelectorDialog::selectObjects(s);
    if (objs.isEmpty()) {
        return;
    }
    assert(objs.size() == 1);
    GObject* obj = objs.first();
    occ->setSelectedObject(obj);
}

QString CreateAnnotationWidgetController::validate() {
	updateModel();

	if (model->annotationObject == NULL ) {
        if (model->newDocUrl.isEmpty()) {
            return tr("no_document_or_annotation_object_selected");
        }
        if (AppContext::getProject()->findDocumentByURL(model->newDocUrl)!=NULL) {
            return tr("document_exists_%1").arg(model->newDocUrl);
        }
        QFileInfo fi(model->newDocUrl);
        QString dirUrl = QFileInfo(newFileEdit->text()).absoluteDir().absolutePath();
        QDir dir(dirUrl);
        if (!dir.exists()) {
            return tr("dir_not_exists_%1").arg(dirUrl);
        }
    }

	if (model->data->name.isEmpty()) {
        annotationNameEdit->setFocus();
		return tr("no_annotation_name");
	}

	if (model->data->name.length() > 15) {
        annotationNameEdit->setFocus();
		return tr("annotation_name_is_too_long");
	}

    QBitArray validAnnotationChars = TextUtils::ALPHA_NUMS;
    validAnnotationChars['_'] = true;
    QByteArray aName = model->data->name.toLocal8Bit();;
    if (!TextUtils::fits(validAnnotationChars, aName.constData(), aName.size())) {
        annotationNameEdit->setFocus();
        return tr("illegal_chars_in_annotation_name");
    }
    
	if (model->groupName.isEmpty()) {
        groupNameEdit->setFocus();
		return tr("no_group_name");
	}
    if (model->groupName.trimmed()!= model->groupName) {
        groupNameEdit->setFocus();
        return tr("trailing spaces are not allowed in group name");
    }

    QBitArray validGroupChars = TextUtils::ALPHA_NUMS;
    validGroupChars['_'] = true;
    validGroupChars['-'] = true;
    validGroupChars[' '] = true;
    QByteArray groupName = model->groupName.toLocal8Bit();
    if (!TextUtils::fits(validGroupChars, groupName.constData(), groupName.size())) {
        groupNameEdit->setFocus();
        return tr("illegal_chars_in_group_name");
    }

	if (!model->hideLocation && model->data->location.isEmpty()) {
        locationEdit->setFocus();
		return tr("no_location");
	}

    AppContext::getSettings()->setValue(SETTINGS_LAST_USED_ANNOTATION_NAME, model->data->name);

	return QString::null;
}


void CreateAnnotationWidgetController::updateModel() {
	model->data->name = annotationNameEdit->text();

	model->groupName = groupNameEdit->text();
	if (model->groupName == GROUP_NAME_AUTO) {
		model->groupName = model->data->name;
	}

	model->data->location.clear();
	model->data->complement = false;
    
    if (!model->hideLocation) {
        QByteArray locEditText = locationEdit->text().toAscii();
        Genbank::LocationParser::parseLocation(	locEditText.constData(), 
											locationEdit->text().length(), 
											model->data->complement, model->data->location);
    }

    if (existingObjectRB->isChecked()) {
        model->annotationObject = qobject_cast<AnnotationTableObject*>(occ->getSelectedObject());
        model->newDocUrl = "";
    } else {
        model->annotationObject = NULL;
        model->newDocUrl = newFileEdit->text();
    }
}

void CreateAnnotationWidgetController::prepareAnnotationObject() {
    QString v = validate();
    assert(v.isEmpty());
    if (model->annotationObject == NULL) {
        assert(!model->newDocUrl.isEmpty());
        assert(AppContext::getProject()->findDocumentByURL(model->newDocUrl)==NULL);
        IOAdapterFactory* iof = AppContext::getIOAdapterRegistry()->getIOAdapterFactoryById(BaseIOAdapters::LOCAL_FILE);
        Document* d = AppContext::getDocumentFormatRegistry()->getFormatById(BaseDocumentFormats::PLAIN_GENBANK)->createNewDocument(iof, model->newDocUrl);
        model->annotationObject = new AnnotationTableObject("Annotations");
        model->annotationObject->addObjectRelation(GObjectRelation(model->sequenceObjectRef, GObjectRelationRole::SEQUENCE));
        d->addObject(model->annotationObject);
        AppContext::getProject()->addDocument(d);
    }
}

} // namespace
