/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#ifndef _GB2_ORFFINDALGORITHM_H_
#define _GB2_ORFFINDALGORITHM_H_

#include <core_api/LRegion.h>
#include <datatype/AnnotationData.h>

#include <QtCore/QList>

namespace GB2 {

class DNATranslation;

class GB2_COREAPI_EXPORT ORFFindResult {
public:
    ORFFindResult () : region(0, 0), frame(0){}
    ORFFindResult (const LRegion& _r, int frame) 
        : region(_r), frame(frame){}
    
    void clear() {region.startPos = 0; region.len = 0; frame = 0;}
    
    bool isEmpty() const {return region.startPos == 0 && region.len == 0;}

    bool operator ==(const ORFFindResult& o) const {
        return region == o.region && frame == o.frame;
    }

    SharedAnnotationData toAnnotation(const QString& name) const {
        SharedAnnotationData data;
        data = new AnnotationData;
        data->name = name;
        data->location.append(region);
        data->complement = (frame < 0);
        data->aminoStrand = TriState_Yes;
        //data->qualifiers.append(Qualifier("frame", QString::number(frame)));
        data->qualifiers.append(Qualifier("dna_len", QString::number(region.len)));
        data->qualifiers.append(Qualifier("protein_len", QString::number(region.len/3)));
        return data;
    }

    LRegion region;
    int frame;

    static QList<SharedAnnotationData> toTable(const QList<ORFFindResult>& res, const QString& name)
    {
        QList<SharedAnnotationData> list;
        foreach (const ORFFindResult& f, res) {
            list.append(f.toAnnotation(name));
        }
        return list;
    }
};

class GB2_COREAPI_EXPORT ORFFindResultsListener {
public:
    virtual ~ORFFindResultsListener(){}
    virtual void onResult(const ORFFindResult& r) = 0;
};

enum ORFAlgorithmStrand {
    ORFAlgorithmStrand_Both,
    ORFAlgorithmStrand_Direct,
    ORFAlgorithmStrand_Complement
};


class GB2_COREAPI_EXPORT ORFAlgorithmSettings {
public:
    ORFAlgorithmSettings(ORFAlgorithmStrand strand = ORFAlgorithmStrand_Both,
        DNATranslation* complementTT = NULL,
        DNATranslation* proteinTT = NULL,
        const LRegion& searchRegion = LRegion(),
        int minLen = 0,
        bool mustFit = false,
        bool mustInit = true,
        bool allowAltStart = false
        ) : strand(strand), complementTT(complementTT), proteinTT(proteinTT),
        searchRegion(searchRegion), minLen(minLen), mustFit(mustFit), 
        mustInit(mustInit), allowAltStart(allowAltStart) {}

    ORFAlgorithmStrand strand;
    DNATranslation*     complementTT;
    DNATranslation*     proteinTT;
    LRegion             searchRegion;
    int                 minLen;
    bool                mustFit;
    bool                mustInit;
    bool                allowAltStart;
};


class GB2_COREAPI_EXPORT ORFFindAlgorithm {
public:

    static void find(
        ORFFindResultsListener* rl,
        const ORFAlgorithmSettings& config,
        const char* sequence, 
        int seqLen, 
        bool& stopFlag, 
        int& percentsCompleted);
};

}//namespace

#endif
