/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "GObjectUtils.h"

#include "DNASequenceObject.h"
#include "AnnotationTableObject.h"

#include <core_api/AppContext.h>
#include <core_api/ProjectModel.h>
#include <core_api/DocumentModel.h>
#include <core_api/DNATranslation.h>
#include <core_api/GHints.h>
#include <gobjects/GObjectRelationRoles.h>

namespace GB2 {

QList<GObject*> GObjectUtils::findAllObjects(GObjectType t) {
    assert(AppContext::getProject()!=NULL);
    QList<GObject*> res;
    foreach(Document* doc, AppContext::getProject()->getDocuments()) {
        if (t.isEmpty()) {
            res+=doc->getObjects();
        } else {
            res+=doc->findGObjectByType(t);
        }
    }
    return res;
}


QSet<GObject*> GObjectUtils::selectRelations(GObject* obj, GObjectType type, const QString& relationRole, const QList<GObject*>& fromObjects) {
    QSet<GObject*> res;
    QList<GObjectRelation> relations = obj->getObjectRelations();
    foreach(const GObjectRelation& r, relations) {
        if (r.role != relationRole || (!type.isEmpty() && r.ref.objType!=type)) {
            continue;
        }
        GObject* obj = selectObjectByReference(r.ref, fromObjects);
        if (obj!=NULL) {
            res.insert(obj);
        }
    }
    return res;
}

QList<GObject*> GObjectUtils::findObjectsRelatedToObjectByRole(const GObject* obj, GObjectType resultObjType, 
                                                               const QString& role, const QList<GObject*>& fromObjects) 
{
    QList<GObject*> res;
    QList<GObject*>	objects = resultObjType.isEmpty() ? fromObjects : select(fromObjects, resultObjType);
    foreach(GObject* o, objects) {
        if (o->hasObjectRelation(obj, role)) {
            res.append(o);
        }
    }
    return res;
}

QList<GObject*> GObjectUtils::selectObjectsWithRelation(const QList<GObject*>& objs, GObjectType type, 
                                                        const QString& relationRole, bool allowUnloadedRelations) 
{
    QList<GObject*> res;
    QList<GObject*> allObjects = findAllObjects(type);
    foreach(GObject* obj, objs) {
        QList<GObjectRelation> relations = obj->getObjectRelations();
        foreach(const GObjectRelation& r, relations) {
            if (r.role != relationRole || (!type.isEmpty() && r.ref.objType!=type)) {
                continue;
            }
            if (!allowUnloadedRelations && selectObjectByReference(r.ref, allObjects) == NULL) {
                continue;
            }
            res.append(obj);
        }
    }
    return res;
}

GObject* GObjectUtils::selectObjectByReference(const GObjectReference& r, const QList<GObject*>& fromObjects) {
    foreach(GObject* o, fromObjects) {
        if (o->getGObjectName() != r.objName) {
            continue;
        }
        if (o->getDocument() == NULL) {
            if (r.docUrl.isEmpty()) {
                return o;
            }
            continue;
        }
        if (o->getDocument()->getURL() == r.docUrl) {
            return o;
        }
    }
    return NULL;
}

DNATranslation* GObjectUtils::findComplementTT(DNASequenceObject* so) {
    if (!so->getAlphabet()->isNucleic()) {
        return NULL;
    }
    DNATranslation* res = NULL;
    DNATranslationRegistry* tr = AppContext::getDNATranslationRegistry();
    QList<DNATranslation*> complTs = tr->lookupTranslation(so->getAlphabet(), DNATranslationType_NUCL_2_COMPLNUCL);
    if (!complTs.empty()) {
        res = complTs.first();
    }

    return res;
}

DNATranslation* GObjectUtils::findAminoTT(DNASequenceObject* so, bool fromHintsOnly) {
    if (!so->getAlphabet()->isNucleic()) {
        return NULL;
    }
    DNATranslationRegistry* tr = AppContext::getDNATranslationRegistry();
    QString tid = so->getGHints()->get(AMINO_TT_GOBJECT_HINT).toString();
    DNATranslation* res = tr->lookupTranslation(so->getAlphabet(), DNATranslationType_NUCL_2_AMINO, tid);
    if (res != NULL || fromHintsOnly) {
        return res;
    }
    QList<DNATranslation*> aminoTs = tr->lookupTranslation(so->getAlphabet(), DNATranslationType_NUCL_2_AMINO);
    if (!aminoTs.empty()) {
        res = aminoTs.first();
    }
    return res;
}


} //namespace

