/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "DNAAlphabetRegistryImpl.h"

#include <core_api/Log.h>
#include <core_api/DNAAlphabet.h>
#include <util_text/TextUtils.h>

namespace GB2 {

/* TRANSLATOR GB2::DNAAlphabetRegistryImpl */    

static LogCategory log(ULOG_CAT_CORE_SERVICES);

DNAAlphabetRegistryImpl::DNAAlphabetRegistryImpl(DNATranslationRegistry* _tr) {
	treg = _tr;
	initBaseAlphabets();
	initBaseTranslations();
}

DNAAlphabetRegistryImpl::~DNAAlphabetRegistryImpl() {
	foreach(DNAAlphabet* a, alphabets) {
		delete a;
	}
}

static bool alphabetComplexityComparator(const DNAAlphabet* a1, const DNAAlphabet* a2) {
    int a1Size = a1->getMap().count(true); //TODO: cache this val
    int a2Size = a2->getMap().count(true);
    return a1Size < a2Size;
}


bool DNAAlphabetRegistryImpl::registerAlphabet(DNAAlphabet* a) {
	if (findById(a->getId())!=NULL) {
		return false;
	}	
	alphabets.push_back(a);
    //WARN: original order for equal alphabets must not be changed (DNA must be before RNA)
    qStableSort(alphabets.begin(), alphabets.end(), alphabetComplexityComparator); 
	return true;
}

void DNAAlphabetRegistryImpl::unregisterAlphabet(DNAAlphabet* a) {
	int n = alphabets.removeAll(a);
    assert(n==1); Q_UNUSED(n);
}

DNAAlphabet* DNAAlphabetRegistryImpl::findById(const QString id) const {
	foreach(DNAAlphabet* al, alphabets) {
		if (al->getId() == id) {
			return al;
		}
	}
	return NULL;
}

static bool matches(DNAAlphabet* al, const QByteArray& ba) {
    bool rc = false;
    if (al->getType() == DNAAlphabet_RAW) {
        rc = true;
    } else {
	    rc = TextUtils::fits(al->getMap(), ba.constData(), ba.size());
    }
    if (rc == true) {
        log.trace("alphabet matches: " + al->getName());
    }
    return rc;
}

static bool matches(DNAAlphabet* al, const QByteArray& ba, const LRegion& r) {
    bool rc = false;
    if (al->getType() == DNAAlphabet_RAW) {
        rc = true;
    } else {
        rc = TextUtils::fits(al->getMap(), ba.constData() + r.startPos, r.len);
    }
    if (rc) {
        log.trace("alphabet matches (by region): " + al->getName());
    }
    return rc;
}


QList<DNAAlphabet*> DNAAlphabetRegistryImpl::findAlphabets(const QByteArray& seq, bool onlyOne) const {
	QList<DNAAlphabet*> res;
	foreach(DNAAlphabet* al, alphabets) {
		if(matches(al, seq)) {
			res.push_back(al);
            if (onlyOne) {
                break;
            }
		}
	}
    return res;
}

QList<DNAAlphabet*> DNAAlphabetRegistryImpl::findAlphabets(const QByteArray& seq, const QList<LRegion>& regionsToProcess, bool onlyOne) const {
    QList<DNAAlphabet*> res;
    foreach(DNAAlphabet* al, alphabets) {
        bool err = false;
        foreach (const LRegion& r, regionsToProcess) {
            if (!matches(al, seq, r)) {
                err = true;
                break;
            }
        }
        if (!err) {
            res.push_back(al);
            if (onlyOne) {
                break;
            }
        }
    }
    return res;
}

}//namespace
