/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "PhyTree.h"

namespace GB2 {

PhyTreeData::PhyTreeData() : rootNode(NULL)  {
}

PhyTreeData::PhyTreeData(const PhyTreeData& other) : QSharedData(other) {
    rootNode = other.rootNode == NULL ? NULL : other.rootNode->clone();
}

PhyNode::PhyNode() {
}

PhyBranch::PhyBranch() : node1(NULL), node2(NULL), distance(0) {
}


void PhyTreeData::validate() const {
#ifdef _DEBUG
    if (rootNode != NULL) {
        QList<const PhyNode*> track;
        rootNode->validate(track);
    }
#endif
}

void PhyNode::validate(QList<const PhyNode*>& track) const {
    assert(!track.contains(this));
    track.append(this);
    foreach(PhyBranch* b, branches) {
        assert(b->node1 != NULL && b->node2!=NULL);
        if (b->node1 != this) {
            b->node1->validate(track);
        } else if (b->node2 != this) {
            b->node2->validate(track);
        }
    }    
}

bool PhyNode::isConnected(const PhyNode* node) const {
    foreach(PhyBranch* b, branches) {
        if (b->node1 == node || b->node2 == node) {
            return true;
        }
    }
    return false;
}

PhyBranch* PhyNode::addBranch(PhyNode* node1, PhyNode* node2, double distance) {
    assert(!node1->isConnected(node2));

    PhyBranch* b = new PhyBranch();
    b->distance = distance;
    b->node1 = node1;
    b->node2 = node2;
    
    node1->branches.append(b);
    node2->branches.append(b);
    
    return b;
}

void PhyNode::removeBranch(PhyNode* node1, PhyNode* node2) {
    foreach(PhyBranch* b, node1->branches) {
        if (b->node1 == node2 && b->node2 == node2) {
            node1->branches.removeOne(b);
            return;
        }
    }
    assert(0);
}

PhyNode* PhyNode::clone() const {
    QSet<const PhyNode*> track;
    addToTrack(track);

    QSet<PhyBranch*> allBranches;
    QMap<const PhyNode*, PhyNode*> nodeTable;
    foreach(const PhyNode* n, track) {
        PhyNode* n2 = new PhyNode();
        n2->name = n->name;
        nodeTable[n] = n2;
        foreach(PhyBranch* b, n->branches) {
            allBranches.insert(b);
        }
    }
    foreach(PhyBranch* b, allBranches) {
        PhyNode* node1 = nodeTable[b->node1];
        PhyNode* node2 = nodeTable[b->node2];
        assert(node1!=NULL && node2!=NULL);
        PhyNode::addBranch(node1, node2, b->distance);
    }
    PhyNode* myClone = nodeTable.value(this);
    assert(myClone!=NULL);
    return myClone;
}

void PhyNode::addToTrack(QSet<const PhyNode*>& track) const {
    if (track.contains(this)) {
        return;
    }
    track.insert(this);
    foreach(PhyBranch* b, branches) {
        b->node1->addToTrack(track);
        b->node1->addToTrack(track);
    }
}

}//namespace
