/* ==================================================== ======== ======= *
 *
 *  uwin.hpp
 *  Ubit Project  [Elc][2003]
 *  Author: Eric Lecolinet
 *
 *  Part of the Ubit Toolkit: A Brick Construction Game Model for Creating GUIs
 *
 *  (C) 1999-2003 Eric Lecolinet @ ENST Paris
 *  WWW: http://www.enst.fr/~elc/ubit   Email: elc@enst.fr (subject: ubit)
 *
 * ***********************************************************************
 * COPYRIGHT NOTICE : 
 * THIS PROGRAM IS DISTRIBUTED WITHOUT ANY WARRANTY AND WITHOUT EVEN THE 
 * IMPLIED WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. 
 * YOU CAN REDISTRIBUTE IT AND/OR MODIFY IT UNDER THE TERMS OF THE GNU 
 * GENERAL PUBLIC LICENSE AS PUBLISHED BY THE FREE SOFTWARE FOUNDATION; 
 * EITHER VERSION 2 OF THE LICENSE, OR (AT YOUR OPTION) ANY LATER VERSION.
 * SEE FILES 'COPYRIGHT' AND 'COPYING' FOR MORE DETAILS.
 * ***********************************************************************
 *
 * ==================================================== [Elc:03] ======= *
 * ==================================================== ======== ======= */

#ifndef _uwin_hpp_
#define	_uwin_hpp_
//pragma ident	"@(#)uwin.hpp	ubit:03.06.04"
#include <ubit/ubox.hpp>
#include <ubit/uview.hpp>

/** Generic Window: base class for windows and menus.
 *
 *  UWin is an abstract class:
 *  - see subclasses UDialog and UFrame for making dialog boxes,
 *    main windows and the "main frame"
 *  - see subclasses UMenubar, UMenu and UPopmenu for making menu systems
 *
 *  Notes:
 *  - as other Ubit UBricks, windows MUST HAVE at least one parent
 *    (they won't show up otherwise).
 *
 * - as other UBricks, windows can have several parents 
 *
 * - if a parent of a window is a UButton (or a subclass), clicking on
 *   this button <b>opens this window</b> automatically
 *
 *  - the subcomponents of windows must be added BEFORE showing windows
 *    (see: UWin::show()). Window size won't change after the show()
 *    method has been called (more precisely, it is not not changed when
 *    new children are added but can still be changed by user interaction
 *    or by calling the UWin::resize() method).
 */
class UWin : public UBox {
public:
  static UStyle *style;

  UWin(const UArgs&);
  ///< constructor; see also class UGroup and destructor UGroup::~UGroup

  virtual ~UWin();
  ///< <b>important note</b> on <em>recursive</em> deletion: see UGroup:~UGroup

  virtual class UWin* winCast() {return this;}

  virtual const UStyle& getStyle(UContext*) const {return makeStyle();}
  static  const UStyle& makeStyle();

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  virtual void move(UWin*, u_pos x_in_win, u_pos y_in_win, UDisp* = null);
  ///< changes window location relatively to a (x,y) position in a Window.
  
  virtual void move(class UView*, u_pos x_in_view, u_pos y_in_view);
  ///< changes window location relatively to a (x,y) position in a View.
  
  virtual void move(class UView*, class UPlacement&);
  ///< changes window location relatively to a View using placement constraints (@refer UPlacement).
  
  virtual void move(class UEvent*, u_pos x_offset, u_pos y_offset);
  virtual void move(class UEvent&, u_pos x_offset, u_pos y_offset);
  ///< changes window location relatively to an Event position.
  
  virtual void moveOnScreen(u_pos x_in_screen, u_pos y_in_screen, UDisp* = null);
  ///< changes window location on Screen.
  
  virtual void moveAndCenter(UDisp* = null);
  ///< centers the window on the screen.
  
  virtual bool where(UWin*, u_pos &x, u_pos &y, UDisp* = null) const;
  ///< gets window location relatively to the origin of another Window.
  
  virtual bool where(class UView*, u_pos &x, u_pos &y) const;
  ///< gets window location relatively to the origin of a View.
  
  virtual bool whereOnScreen(u_pos &x, u_pos &y, UDisp* = null) const;
  ///< gets window location on Screen.
  
  virtual void resize(u_dim width, u_dim height, UDisp* = null);
  ///< changes current window size.
  
  virtual bool getSize(u_dim& width, u_dim& height, UDisp* = null) const;
  virtual u_dim getWidth( UDisp* = null) const;
  virtual u_dim getHeight( UDisp* = null) const;  
  ///< returns current size.

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  virtual void highlight(bool state);

  virtual void close(int status = 0);
  /**< 
   * closes this window.
   * Default behavior: hides the object (same as show(false)).
   * 
   * This function is called when the "Close" button of the window manager 
   * is clicked. It can be redefined by subclasses for appropriate behaviors
   * (typically for saving or freeing data)
   *
   * 'status' should be >= 0 (0 being the default). see: UWin::lock()
   * for details. close() is also called by UBox::closeWin() and
   * by the ucloseWin() callback
   */

  virtual int lock(UDisp* = null);
  /**< 
   * opens the window in modal mode and locks the execution thread.
   * This function returns when the window is closed. 
   *
   * Returned value: 
   * - 0 when the window is closed by clicking on the "Close" button 
   *   of the window manager
   * - the value given as an argument to close() 
   * - this value should be >= 0 (typically the "Ok" button
   *   should call close(1) and the "Cancel" button close(0))
   */

  virtual void show(bool state, UDisp*);
  virtual void show(bool state = true);  //polymorphism
  /**< 
   * shows the window to if argument is true, hides if it is false.
   * show(true) also and brings the window to the front.
   * - Note: by default, the window size won't change once the show() 
   *   method has been called.
   * - see also: class UWin, closeWin() and lock().
   */

  virtual bool isShown() const;
  ///< is the window currently shown ?

  virtual void setModal(bool) ;
  ///< sets this dialog in modal or non modal mode.

  virtual bool isModal() const;
  ///< is this dialog modal?.

  virtual void toBack(); 
  ///< sends this window to the back. 
 
  virtual void toFront();
  ///< brings this window to the front. 

  virtual void update();
  virtual void update(class UUpdate upmode);   //polymorphism
  virtual void update(class UUpdate upmode, UDisp*); 
  /**< 
   *updates this object's paint and/or layout.
   * Arguments:
   * - no argument: updates layout then repaints
   * - upmode argument : updates according to this argument. 
   *   see class UUpdate for details
   * Note:
   * - update() must be called after add() or remove() if their last argument
   *   was set to 'false'
   */

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  class UView* getWinViews() const {return winviews;}
  class UView* getWinView(class UDisp*) const;
  ///< returns the window View (UWins have only one (shared) view).

  class UAppli* getAppli() const;
  ///< returns the Application Context of this window.

  //class UDisp* getDisp()  const {return winview ? winview->getDisp() : null;}
  ///< returns the Display Context of this window.

  virtual void setSoftwinMode(bool = true);
  UWin& softwin() {setSoftwinMode(true); return *this;}
  /**<
   * sets the "soft window" mode if arg is true. 
   * see: isSoftwin() for details.
   * - Warning: this mode can only be set *before* window initialization
   *   (ie. before the window is added to a visible object) */

  bool isHardwin() const {return is_hardwin;}
  bool isSoftwin() const {return !is_hardwin;}
  /**<
   * is this window a "hard window" or a "soft window" ?.
   * Notes:
   * - a "hard window" has its own graphics context and (UNatWin) native 
   *   window  (= a native X Window counterpart)
   * - a "soft window" simulates an actual window and uses the graphics
   *   context and native window of another hard window (typically the
   *   main frame of the application)
   * - any window can either be a hard or soft window (except the main frame
   *   which must always be a hard window).*/

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  // implementation

  virtual bool realize() = 0;
  /**< [impl.] initializes the window and its children. 
   * creates the associated graphics context and native window if this is 
   * a "hard window".
   *
   * this fct. is automatically called when the window is shown for the first time
   * (so that unused windows won't consume unnecessary resources) */

  class UWinGraph* getWinGraph(class UDisp* = null) const;
  ///< [impl] returns the Graphics context that is used for drawing this window

  class UNatWin* getNatWin(class UDisp* = null) const;
  ///< [impl] returns the Native Window that is used for drawing this window

#ifndef NO_DOC

  friend class UBox;
  friend class UAppli;
  friend class UBoxLink;
  friend class UHardwinImpl;
  friend class USoftwinImpl;
  friend class UNatWin;
  friend class UNatDisp;
  
  class ULink* getSubSoftwinLink() const;
  void addWinView(class UView*);
  
  virtual void initView(ULink* selflink, UView* ancestor_view);
  virtual UView* _initView(ULink* selflink, UView* ancestor_view, UDisp*);
  virtual void deleteRelatedViews(class UView* parview, class ULink*);
  //NB: parview and link can be null for UWin::deleteRelatedViews
  
protected:
  bool  is_hardwin;
  class UView* winviews;
  union {
    void* initialized;          // null if not initialized
    class USoftwinImpl* soft;
    class UHardwinImpl* hard;
  } impl;

  virtual UView* realizeSoftwin(UWin* hardwin, UView* hardwin_view);
  virtual bool realizeHardwin(bool (*xrealize)(class UNatDisp*, 
					       class UNatWin*, UWin*));
  ///< [impl] internal realization function

  virtual void realizeChildren(class UView* winview);
  ///< [impl] called by realize() to create the views of the children 

#endif
};

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

/** UDialog: dialog box.
 *  see class UWin for important info. (most methods being inherited 
 *  from this class).
 *
 * Iconification and window stack
 * - Dialogs can't be iconified and are always located on the top
 *   of the main frame (NB: this standard behavior is not enforced
 *   by certain window managers)
 * - dialog boxes are hidden when the Main Frame is iconified (see UFrame)
 *
 * Implicit hehaviors:
 * - if a parent of a dialog is a UButton (or a subclass), clicking on
 *   this button <b>opens the dialog</b> automatically
 */
class UDialog : public UWin {
public:
  static UStyle *style;

  UDialog(const UArgs& = UArgs::none);
  ///< constructor; see also ~UGroup and the <em>creator shortcut</em> <b>udialog()</b>

  friend UDialog& udialog(const UArgs& a = UArgs::none) {return *new UDialog(a);}
  ///< creator shortcut that is equivalent to: *new UDialog()

  virtual ~UDialog() {destructs();}

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  virtual void show(bool state, UDisp*);
  virtual void show(bool state = true);
  virtual bool realize();

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  virtual const UStyle& getStyle(UContext*) const {return makeStyle();}
  static  const UStyle& makeStyle();

#ifndef NO_DOC
  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  // implementation

  virtual void addingTo(class ULink *selflink, UGroup *parent);
  virtual void removingFrom(class ULink *selflink, UGroup *parent);
  ///< NOTE that this function requires a specific destructor.

private:
  UCall* open_call;
#endif
};

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

/** UFrame: main frame and primary main window.
 *  see class UWin for important info. (most methods being inherited 
 *  from this class).
 *
 *  the "Main Frame" is the first UFrame that is added to the UAppli
 *  other UFrame(s) are considered as "primary windows"
 *
 *  Iconification and window stack
 *  - In contrast with Dialogs, Frames can be iconified. 
 *  - dialogs are hidden when the Main Frame is iconified, frames are not
 *
 * Implicit hehaviors:
 * - if a parent of a frame is a UButton (or a subclass), clicking on
 *   this button <b>opens the frame</b> automatically
 */
class UFrame : public UDialog {
  friend class UAppli;
  bool is_main_frame;
public:
  static UStyle *style;
 
  UFrame(const UArgs& a = UArgs::none);
  ///< constructor; see also ~UGroup and the <em>creator shortcut</em> <b>uframe()</b>

  friend UFrame& uframe(const UArgs& a = UArgs::none) {return *new UFrame(a);}
  ///< creator shortcut that is equivalent to: *new UFrame()

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  virtual const UStyle& getStyle(UContext*) const {return makeStyle();}
  static  const UStyle& makeStyle();

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  virtual void close(int status);
  /**< closes this window and quits the application if this window is the main frame.
   * the main frame is the first UFrame that was added to the UAppli.
   * <p>see: UWin::close(), UAppli::quit(), UFrame::isMainFrame()
   */

  bool isMainFrame() const {return is_main_frame;}
  ///< is this frame the "main frame"? (see: UAppli)

  virtual bool realize();
};

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

/** Constrainst for automatic Window Placement.
 */
class UPlacement {
public:
  // Horizontal placement. 
  // -- Explicit if &uleft() or &uright()
  // -- Default placement if null or other UHalign values.
  class UHalign *halign;

  // vertical placement. Explicit if &utop() or &ubottom()
  class UValign *valign;

  // Relative location from reference object:
  //
  // if oppositeBorder is true:
  //    the specified border (eg. uleft()) of the reference object is 
  //    aligned with the OPPOSITE border of 'this' object (eg. right)
  //    --> this implies that 'this' object will be located OUTSIDE
  //    the ref. object (exemple: a cascaded submenu located on the
  //    right side of its parent menu)
  //
  // if oppositeBorder is false:
  //    same borders are used for both objects (eg. both left
  //    sides are aligned together)
  //    --> this implies that 'this' object will be located INSIDE
  //    the ref. object (in the same way as for usual UH/Valign rules)

  bool hoppositeBorder, voppositeBorder;

  // relative 'distance' between reference and 'this' object borders.
  // Usually 1 or 0. Can be < 0 as it is a relative 'distance'.
  u_dim hdist, vdist;

  UPlacement();
};

// Examples (R = reference object, T = this object)
// --  RR TT   hplace = uright(),  hoppositeBorder = true,  hdist = 9
//     RR      vplace = utop(),    hoppositeBorder = false, vdist = 0
//
// --  RR      hplace = uleft(),   hoppositeBorder  = false, hdist = 0
//     RR      vplace = ubottom(), hoppositeBorder  = true,  vdist = 1
//     TT
//
// --  TT      hplace = uleft(),   hoppositeBorder = true,  hdist = 0
//     TTRR    vplace = ubottom(), hoppositeBorder = false, vdist = 0


/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

/** Incrusted Window. 
 * a "hard window" (= having a native X Window counterpart)
 * that is tiled inside another ubit window. this is useful for inserting
 * a true X window that was created by another application or another part
 * of the same program into the Ubit GUI. a typical use is for drawing 
 * OpenGL graphics. Note that incrusted windows can only have a single
 * (shared) view (as other Ubit windows).
 */
class UIncrust : public UWin {
public:
  static UStyle *style;

  UIncrust(const UArgs& a = UArgs::none);
  virtual ~UIncrust();

  friend UIncrust& uincrust(const UArgs& a = UArgs::none);
  ///< creator shortcut that is equivalent to: *new UIncrust().

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  virtual const UStyle& getStyle(UContext*) const {return makeStyle();}
  static  const UStyle& makeStyle();

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  virtual unsigned long getXWin() const {return xwin;}
  ///< returns the associated (internal or external) X Window

  virtual void setExternalXWin(unsigned long external_xwin);
  /**< specifies the external X Window to be incrusted in the Ubit GUI.
   * If this fct. is not called, the X Window is automatically created 
   * when the object is shown for the first time
   */

  virtual bool isExternalXWin() const {return is_external_win;}
  /** is this window an externa X Window ?
   * returns true if this is an external X Window,
   * and false if the window was created by the toolkit
   */

  virtual bool realize();

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  // Impl.

  virtual void initView(ULink* selflink, UView* ancestor_view);
  // [impl: must be redefined to init parent view]

private:
  friend class UNatWin;
  bool is_external_win;
  unsigned long xwin;
};

#endif
/* ==================================================== [TheEnd] ======= */
/* ==================================================== [Elc:03] ======= */


