/* ==================================================== ======== ======= *
 *
 *  uuappli.cpp
 *  Ubit Project  [Elc][2003]
 *  Author: Eric Lecolinet
 *
 *  Part of the Ubit Toolkit: A Brick Construction Game Model for Creating GUIs
 *
 *  (C) 1999-2003 Eric Lecolinet @ ENST Paris
 *  WWW: http://www.enst.fr/~elc/ubit   Email: elc@enst.fr (subject: ubit)
 *
 * ***********************************************************************
 * COPYRIGHT NOTICE : 
 * THIS PROGRAM IS DISTRIBUTED WITHOUT ANY WARRANTY AND WITHOUT EVEN THE 
 * IMPLIED WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. 
 * YOU CAN REDISTRIBUTE IT AND/OR MODIFY IT UNDER THE TERMS OF THE GNU 
 * GENERAL PUBLIC LICENSE AS PUBLISHED BY THE FREE SOFTWARE FOUNDATION; 
 * EITHER VERSION 2 OF THE LICENSE, OR (AT YOUR OPTION) ANY LATER VERSION.
 * SEE FILES 'COPYRIGHT' AND 'COPYING' FOR MORE DETAILS.
 * ***********************************************************************
 *
 * ==================================================== [Elc:03] ======= *
 * ==================================================== ======== ======= */

//pragma ident	"@(#)uuappli.cpp	ubit:03.06.04"
#include <iostream>
#include <algorithm>
#include <locale.h>
#include <stdlib.h>
#include <sys/time.h>
#include <udefs.hpp>
#include <ubrick.hpp>
#include <ucond.hpp>
#include <ucall.hpp>
#include <uerror.hpp>
#include <ubox.hpp>
#include <uwin.hpp>
#include <umenu.hpp>
#include <umenuImpl.hpp>
#include <uappli.hpp>
#include <uflow.hpp>
#include <unatappli.hpp>
#include <update.hpp>
#include <utextsel.hpp>
#include <umsclient.hpp>
using namespace std;

UAppli* UAppli::default_appli;
UDispList UAppli::displist;
std::vector<void*> UAppli::safe_delete_list;
UGroup *UAppli::inputs, *UAppli::timers;
const char *UAppli::pixmapPath, *UAppli::bitmapPath;
struct UAPP_VAR* UAppli::varDB;
int UAppli::varDB_count;

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

UAppli::UAppli(int* argc, char** argv) : 
  UDisp(0, *this, *new UConf(*argc, argv)) {
  constructs();
}

UAppli::UAppli(int& argc, char** argv) : 
  UDisp(0, *this, *new UConf(argc, argv)) {
  constructs();
}

UAppli::UAppli(UConf& _conf) : 
  UDisp(0, *this, _conf) {
  constructs();
}

void UAppli::constructs() {
  if (getStatus() < UDisp::Opened) { // throws an exception
    UError::error("fatal@UAppli::UAppli","can't create application context");
  }

  if (!default_appli) default_appli = this;
  varDB         = null;
  varDB_count   = 0;
  main_frame    = null;
  modalwin      = null;
  main_status   = LOOP_STOP;
  modal_status  = LOOP_STOP;
  pixmapPath    = null;
  bitmapPath    = null;
  inputs        = null;
  timers        = null;

  // add 'this' to the disp list
  displist.push_back(this);

  // create the standard event flow
  //openFlow(0);

  natappli = new UNatAppli(*this, *natdisp);
}

UAppli::~UAppli() {
  default_appli = null;
}

UAppli* UAppli::getApp() {
  return default_appli;
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

void UAppli::realize() {
  UDisp::realize();
  //if (conf.locale) setlocale(LC_CTYPE, conf.locale);
  if (conf.locale) setlocale(LC_ALL, conf.locale);
  natappli->setProperties(conf.app_argc, conf.app_argv);
} 

bool UAppli::isRealized() const {
  return natdisp->isRealized();
}

const char* UAppli::getCommandName() const {
  return conf.app_name;
}

const char* UAppli::getCommandPath() const {
  return (conf.app_argc > 0 ? conf.app_argv[0] : null);
}

UFrame* UAppli::getMainFrame() const {
  return main_frame;
}

//bool UAppli::setConf(const char* profile) {
//  return conf.setProfile(profile);
//}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

void UAppli::addModalwin(class UWin* win) {
  // add at the beginning of the list
  modalwin_list.insertAfter(win->makeLink(), null);
  modalwin = win;
}

void UAppli::removeModalwin(class UWin* win) {
  ULink *l = modalwin_list.search(win);
  if (l) {
    modalwin_list.remove(l);
    if (win == modalwin) // pop : get the previous one
    modalwin = (UWin*)
      (modalwin_list.first() ? modalwin_list.first()->getChild() : null);
  }
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */
// En arg: n'importe quelle Window; mais seul le MainFrame (= le 1er Frame 
// ajoute a UAppli) est automatiquement affichee par defaut

void UAppli::add(UWin& win) {
  UDisp::add(win);
  UFrame* frame = dynamic_cast<UFrame*>(&win);
  if (frame) {
    if (!main_frame) {    // makes the first UFrame be the 'Main Frame'
      main_frame = frame; 
      main_frame->is_main_frame = true;
    }
  }
}

void UAppli::add(UWin* win) {
  if (!win) UError::error("UAppli::add", UError::Null_argument);
  else add(*win);
}

/* ==================================================== ======== ======= */

void UAppli::remove(UWin& win, int remove_mode) {
  if (&win == main_frame) main_frame = null;
  /*return*/ UDisp::remove(win, remove_mode);
}

void UAppli::remove(UWin* win, int remove_mode) {
  if (!win) {
    UError::error("UAppli::remove", UError::Null_argument);
    //return null;
  }
  else /*return*/ remove(*win, remove_mode);
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

// sert a quoi ??? utitle() devrait suffire
void UAppli::setTitle(const UStr &title) {
  if (main_frame) {
    UUpdate upd(UUpdate::TITLE);
    upd.paintTitle(&title);
    main_frame->update(upd);
  }
}

static void updateAll2(UGroup *grp, UUpdate mode) {
  // NB: du fait des parents multipls, il est possible de remettre
  // a jour plusieurs fois la meme window
  // ==> il faudrait tenir compte du flag must-update

  if (grp->winCast()) grp->winCast()->update(mode);

  for (ULink *l = grp->getChildLinks(); l != null; l = l->getNext()) {
    UGroup *chg = l->getChild()->groupCast();
    if (chg) {
      //printf("updateAll2\n");
      updateAll2(chg, mode);    // in any case
    }
  }
}

//  updates all visible windows   !! A REVOIR !!
void UAppli::updateAll(UUpdate mode) {

  for (ULink *l = winlist.getChildLinks(); l != null; l = l->getNext()) {
    UGroup *chg = l->getChild()->groupCast();
    if (chg) {
      //printf("updateAll\n");
      updateAll2(chg, mode);
    //EX: if (ww && ww->isShowable() && (/*ww->mustUpdate() ||*/ force_update))
    }
  }
}

void UAppli::updateAll() {
  updateAll(UUpdate::all);
}

// updates all visible windows
/* le pbm c'est que ce n'est pas la liste de TOUTES les windos
   mais seulement de celles qui sont filles de UAppli

void UAppli::updateAll(UUpdate mode) {
  for (ULink *l = winlist.getChildLinks(); l != null; l = l->getNext()) {
    printf("update : obj %d %s\n", l->brick(), l->brick()->cname());
    //UWin *win = dynamic_cast<UWin*>(l->brick());
    UWin *win = l->getChild()->winCast();
    if (win) {
      printf("update win %d\n", win);
      win->update(mode);
    }
  }
}
*/
/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

UMSclient* UAppli::openUMS(const UStr& hostname, int port) {
  UMSclient* cl = new UMSclient(*getApp());
  if (cl->open(hostname, port) > 0) return cl;
  else {
    delete cl;
    return null;
  }
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

UDisp* UAppli::openDisp(const UStr& _display_name) {
  UDisp* d = new UDisp(displist.size(), *getApp(), _display_name);

  if (d->getStatus() > 0) {
    displist.push_back(d);
    return d;
  }
  else {
    delete d;
    return null;
  }
}

unsigned int UAppli::getDispCount()  {
  return displist.size();
}

UDisp* UAppli::getDisp(int _id)  {
  for (unsigned int k = 0; k < displist.size(); k++) {
    if (displist[k]->getID() == _id) return displist[k];
  }
  return false;
}

void UAppli::closeDisp(class UDisp* d) {
  if (!d) return;
  deleteNotifyAll(d);
  delete d;
}


/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

void UAppli::quitLoop(int status) {
  if (status < 0) status = 0;
  main_status = modal_status = status;
}

void UAppli::quit(int status) {
  if (default_appli) default_appli->quitLoop(status);
}

int UAppli::mainLoop() {
  main_status = LOOP_RUN;
  natappli->eventLoop(main_status);  // mainloop
  UAppli::processSafeDeleteRequests();
  return main_status;
}

int UAppli::subLoop() {
  modal_status = LOOP_RUN;

  natappli->eventLoop(modal_status);  // subloop

  int res = modal_status;
  // rearm so that we won't exit enclosing subloops
  modal_status = LOOP_STOP;
  UAppli::processSafeDeleteRequests();
  return res;
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

void UAppli::safeDeleteRequest(void* p) {
  safe_delete_list.push_back(p);
}

void UAppli::processSafeDeleteRequests() {

  if (safe_delete_list.size() > 0)
    // @ cerr << " ??? processSafeDeleteRequests # " <<safe_delete_list.size() << endl;
      
  for (unsigned int k = 0; k < safe_delete_list.size(); k++) {
    ::operator delete(safe_delete_list[k]);
  }
  safe_delete_list.clear();
}

/* ==================================================== ======== ======= */

void UAppli::deleteNotifyAll(class UDisp* d) {
  for (unsigned int k = 0; k < displist.size(); k++) {
    if (displist[k] == d) {
      // va tout decaler si on enleve de la liste !
      displist[k] = null;
      break;
    }
  }
}

void UAppli::deleteNotifyAll(UView* deleted_view) {
  for (unsigned int d = 0; d < displist.size(); d++)

    if (displist[d]) {
      const UFlowList& flist = displist[d]->getFlows();
      for (unsigned int f = 0; f < flist.size(); f++)
	if (flist[f]) flist[f]->deleteNotify(deleted_view);
    }
}

void UAppli::deleteNotifyAll(UGroup* deleted_group) {
   for (unsigned int d = 0; d < displist.size(); d++)

    if (displist[d]) {
      const UFlowList& flist = displist[d]->getFlows();
      for (unsigned int f = 0; f < flist.size(); f++)
	if (flist[f]) flist[f]->deleteNotify(deleted_group);
    }
}

/* ==================================================== ======== ======= */

UGroup* UAppli::getOpenedMenu() {
  //return openFlow(0)->menuCtrl.getActiveMenu();
  UFlow* fl = getFlow(0);               // DEFAULT IFLOW : A REVOIR
  return fl ? fl->menuCtrl.getActiveMenu() : null; 
}

void UAppli::closeRelatedMenus(class UMenu*) {
  // fermer tous les menus, sinon il y a des cas 
  // (en part. avec les filebox)
  //  ou certains super menus restent ouverts

  //openFlow(0)->menuCtrl.closeAllMenus(true);
  UFlow* fl = getFlow(0);               // DEFAULT IFLOW : A REVOIR
  if (fl) fl->menuCtrl.closeAllMenus(true);
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

UInput::UInput(UAppli& a, int _source) :
  appli(a),
  source(_source) {
}

UInput::~UInput() {}

void UInput::close()  {
  source = -1;
  appli.closeInput(this);
}

void UInput::onAction(UCall &c) {
  ULink *l = new ULink(&c);
  l->setCond(UOn::action);
  //setBmodes(UMode::INPUT_CB, true);
  addCall(*l);
}

UInput* UAppli::openInput(int _source) {
  if (!inputs) inputs = new UGroup();
  UInput* i = new UInput(*getApp(), _source);
  inputs->add(i);
  return i;
}

void UAppli::closeInput(UInput* i) {
  if (inputs) inputs->remove(i, true /*UGroup::QUIET_DEL*/);
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

UTimer::UTimer(UAppli& a, u_long _delay, int _ntimes) :
  appli(a),
  timeout(new timeval)
{
  reset(_delay,_ntimes);
}

UTimer::~UTimer() {
  delete timeout;
}

void UTimer::close()  {
  reset(0,0);
}

void UTimer::reset(u_long _delay, int _ntimes) {
  delay  = _delay;
  ntimes = _ntimes;
  must_rearm = true;
  timeout->tv_sec  = 0;
  timeout->tv_usec = 0;
  if (appli.timers){
    if (ntimes == 0) appli.timers->remove(this, true /*UGroup::QUIET_DEL*/);
    else appli.timers->addOnce(this);  // ne pas mettre 2 fois dans la liste!
  }
}

void UTimer::onAction(UCall &c) {
  ULink *l = new ULink(&c);
  l->setCond(UOn::action);
  //setBmodes(UMode::TIMEOUT_CB, true);
  addCall(*l);
}

UTimer* UAppli::openTimer(u_long _delay, int _ntimes) {
  if (!timers) timers = new UGroup();
  UTimer* t = new UTimer(*getApp(), _delay, _ntimes);
  return t;
}

void UAppli::closeTimer(UTimer* t) {
  if (timers) timers->remove(t, true /*UGroup::QUIET_DEL*/);
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */
// pbm: de tsearch() et tfind() : ces fct ne sont pas presentes sur certains
// Unixes (en part Mac OS X / BDS) => on utilise bsearch (in stdlib.h)

struct UAPP_VAR {
  const char *name;
  char *value;
};

extern "C" {
  static int varCompar(const void *n1, const void *n2) {
    return strcmp( ((UAPP_VAR*)n1)->name, ((UAPP_VAR*)n2)->name);
  }
}

const char* UAppli::getVar(const char *name, bool get_shell_env) {
  UAPP_VAR key;
  key.name = name;
  //void *p = tfind(&key, &varDB, varCompar);
  UAPP_VAR *item = null;
  if (varDB) item = (UAPP_VAR*)::bsearch(&key, varDB, varDB_count,
                                         sizeof(UAPP_VAR), varCompar);

  // if (p) {
  //   UAPP_VAR *item = *(UAPP_VAR**)p;
  //   if (item) return item->value;  //note that value can be null
  // }

  if (item) {        //found in local appli environment
    return item->value;  //note that value can be null
  }
  else if (get_shell_env) { //search in SHELL env (and add to local env)
    const char *value = ::getenv(name);
    // if value not found, a NULL value is stored so that we won't
    // search the SHELL env again and again
    setVar(name, value);
    if (value) return value; //found in shell env
  }

  return null; // definitively not found (all default cases)
}

const char *UAppli::setVar(const char *name, const char *value) {
  UAPP_VAR key;
  key.name = name;
  UAPP_VAR *item = null;
  if (varDB) item = (UAPP_VAR*)::bsearch(&key, varDB, varDB_count,
                                         sizeof(UAPP_VAR), varCompar);
  if (item) {
    if (item->value) free(item->value);
    item->value = value ? CStr::strdup(value) : null;
    return item->value;
  }
  else {
    varDB_count++;
    varDB = (UAPP_VAR*) realloc(varDB, varDB_count * sizeof(UAPP_VAR));
    item = &varDB[varDB_count - 1];
    item->name  = CStr::strdup(name);
    item->value = value ? CStr::strdup(value) : null;
    // reordonner (necessaire pour bsearch)
    ::qsort(varDB, varDB_count, sizeof(UAPP_VAR), varCompar);
    return item->value;
  }
}

const char* UAppli::getImaPath() {
  return getVar("UIMA_PATH");
}
const char* UAppli::setImaPath(const char* value) {
  return setVar("UIMA_PATH", value);
}
const char* UAppli::setImaPath(const UStr& value) {
  return setVar("UIMA_PATH", value.chars());
}

char *UAppli::makeImaPath(const char* name) {
  if (!name || name[0] == '\0')
    return null;

  else if (name[0] == '.' || name[0] == '/')
    return CStr::strdup(name);

  else if (name[0] == '$') {  //variable
    const char *val = null;
    const char *sep = strchr(name, '/');

    if (!sep)
      val = ::getenv(name+1);   //just a varname
    else {
      int varlen = sep - (name+1);   //skip the $
      char *varname = (char*)malloc((varlen+1) * sizeof(char));
      strncpy(varname, name+1, varlen);
      varname[varlen] = 0;
      val = getVar(varname);
      free(varname);
    }

    if (val) {
      if (sep) return CStr::strdupcat(val, sep); // 'sep' includes the slash
      else return CStr::strdup(val);  // just the varname's value
    }
    else return CStr::strdup(name);  //val not found => just return name 'as it'
  }

  else {   // no prefix ==> add default pixmap path
    // penser a liste de repertoires !!!!

    const char* ima_path = getImaPath();
    if (ima_path)
      return CStr::strdupcat(ima_path, '/', name);
    else 
      return CStr::strdup(name);
  }
}

/* ==================================================== [TheEnd] ======= */
/* ==================================================== [Elc:03] ======= */
