/* ==================================================== ======== ======= *
 *
 *  unumber.hh
 *  Ubit Project  [Elc][2003]
 *  Author: Eric Lecolinet
 *
 *  Part of the Ubit Toolkit: A Brick Construction Game Model for Creating GUIs
 *
 *  (C) 1999-2003 Eric Lecolinet @ ENST Paris
 *  WWW: http://www.enst.fr/~elc/ubit   Email: elc@enst.fr (subject: ubit)
 *
 * ***********************************************************************
 * COPYRIGHT NOTICE : 
 * THIS PROGRAM IS DISTRIBUTED WITHOUT ANY WARRANTY AND WITHOUT EVEN THE 
 * IMPLIED WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. 
 * YOU CAN REDISTRIBUTE IT AND/OR MODIFY IT UNDER THE TERMS OF THE GNU 
 * GENERAL PUBLIC LICENSE AS PUBLISHED BY THE FREE SOFTWARE FOUNDATION; 
 * EITHER VERSION 2 OF THE LICENSE, OR (AT YOUR OPTION) ANY LATER VERSION.
 * SEE FILES 'COPYRIGHT' AND 'COPYING' FOR MORE DETAILS.
 * ***********************************************************************
 *
 * ==================================================== [Elc:03] ======= *
 * ==================================================== ======== ======= */

#ifndef _unumber_hh
#define	_unumber_hh
//pragma ident	"@(#)unumber.hh	ubit:03.02.00"

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

/** Integer Base Class
 */
class UIntgBase {
private:
  int value;

public:
  UIntgBase(int v = 0)                 : value(v) {}
  UIntgBase(const UIntgBase& v)        : value(v.value) {}
  UIntgBase(const UFloatBase& v);
  UIntgBase(const UStr& v);

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  int getValue() const {return value;}

  virtual void changed(bool update_now) {}
  ///< called when object's content is changed.

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  void set(int);
  void set(const UIntgBase&);
  void set(const UFloatBase&);
  void set(const UStr&);

  UIntgBase& operator=(int v)               {set(v); return *this;}
  UIntgBase& operator=(const UIntgBase& v)  {set(v); return *this;}
  UIntgBase& operator=(const UFloatBase& v) {set(v); return *this;}
  
  UIntgBase& operator=(const UStr& v)       {set(v); return *this;}
  /**< converts and sets value.
   * the string is supposed to contain a valid integer
   */

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  void incr(int);
  void incr(const UIntgBase&);
  void decr(int);
  void decr(const UIntgBase&);

  UIntgBase& operator++()                {incr(1); return *this;}
  UIntgBase& operator++(int)             {incr(1); return *this;}
  UIntgBase& operator--()                {decr(1); return *this;}
  UIntgBase& operator--(int)             {decr(1); return *this;}

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  bool equals(int v)                     const {return value == v;}
  bool equals(const UIntgBase& v)        const {return value == v.value;}
  bool equals(const UFloatBase& v)       const;

  friend bool operator==(const UIntgBase& v1, int v2) 
  {return v1.equals(v2);}

  friend bool operator==(int v1, const UIntgBase& v2) 
  {return v2.equals(v1);}

  friend bool operator==(const UIntgBase& v1, const UIntgBase& v2) 
  {return v1.equals(v2);}

  friend bool operator==(const UIntgBase& v1, const UFloatBase& v2) 
  {return v1.equals(v2);}

  friend bool operator==(const UFloatBase& v1, const UIntgBase& v2) 
  {return v2.equals(v1);}
};

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

/** Active Integer
 */
class UIntg: public UIntgBase, public UBrick {
public:
  UIntg(int v = 0)                       : UIntgBase(v) {}
  UIntg(const UIntg& v)                  : UIntgBase(v) {}
  UIntg(const UIntgBase& v)              : UIntgBase(v) {}
  UIntg(const UFloatBase& v)             : UIntgBase(v) {}
  UIntg(const UStr& v)                   : UIntgBase(v) {}

  virtual void changed(bool update_now);
  ///< called when object's content is changed.

  virtual void onChange(UCall&);
  ///< adds callbacks that are activated when the objects' value changes.

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  UIntg& operator=(int v)                {set(v); return *this;}
  UIntg& operator=(const UIntg& v)       {set(v); return *this;}
  UIntg& operator=(const UIntgBase& v)   {set(v); return *this;}
  UIntg& operator=(const UFloatBase& v)  {set(v); return *this;}
  
  UIntg& operator=(const UStr& v)        {set(v); return *this;}
  /**< converts and sets value.
   * the string is supposed to contain a valid integer */

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  UIntg& operator++()                    {incr(1); return *this;}
  UIntg& operator++(int)                 {incr(1); return *this;}
  UIntg& operator--()                    {decr(1); return *this;}
  UIntg& operator--(int)                 {decr(1); return *this;}
};

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

/** Float Base Class
 */
class UFloatBase {
private:
  float value;

public:
  UFloatBase(float v = 0.0)                : value(v) {}
  UFloatBase(const UFloatBase& v)        : value(v.getValue()) {}
  UFloatBase(const UIntgBase& v)         : value(v.getValue()) {}
  UFloatBase(const UStr& v);

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  float getValue() const {return value;}

  virtual void changed(bool update_now) {}
  ///< called when object's content is changed.

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  void set(float);
  void set(const UIntgBase&);
  void set(const UFloatBase&);
  void set(const UStr&);

  UFloatBase& operator=(float v)             {set(v); return *this;}
  UFloatBase& operator=(const UFloatBase& v) {set(v); return *this;}
  UFloatBase& operator=(const UIntgBase& v)  {set(v); return *this;}  
  UFloatBase& operator=(const UStr& v)       {set(v); return *this;}
  /**< converts and sets value.
   * the string is supposed to contain a valid float */

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  void incr(float);
  void incr(const UFloatBase&);
  void decr(float);
  void decr(const UFloatBase&);

  UFloatBase& operator++()               {incr(1.); return *this;}
  UFloatBase& operator++(int)            {incr(1.); return *this;}
  UFloatBase& operator--()               {decr(1.); return *this;}
  UFloatBase& operator--(int)            {decr(1.); return *this;}

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  bool equals(float v)                   const {return value == v;}
  bool equals(const UIntgBase& v)        const {return value == v.getValue();}
  bool equals(const UFloatBase& v)       const {return value == v.getValue();}

  friend bool operator==(const UFloatBase& v1, float v2)
  {return v1.equals(v2);}

  friend bool operator==(float v1, const UFloatBase& v2)
  {return v2.equals(v1);}

  friend bool operator==(const UFloatBase& v1, const UFloatBase& v2)
  {return v1.equals(v2);}
};

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

/** Active Float
 */
class UFloat: public UFloatBase, public UBrick {
public:
  UFloat(float v = 0.0)                      : UFloatBase(v) {}
  UFloat(const UFloat& v)                : UFloatBase(v) {}
  UFloat(const UFloatBase& v)            : UFloatBase(v) {}
  UFloat(const UIntgBase& v)             : UFloatBase(v) {}
  UFloat(const UStr& v)                  : UFloatBase(v) {}

  virtual void changed(bool update_now);
  ///< called when object's content is changed.

  virtual void onChange(UCall&);
  ///< adds callbacks that are activated when the objects' value changes.

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  UFloat& operator=(float v)             {set(v); return *this;}
  UFloat& operator=(const UFloat& v)     {set(v); return *this;}
  UFloat& operator=(const UFloatBase& v) {set(v); return *this;}
  UFloat& operator=(const UIntgBase& v)  {set(v); return *this;}  
  UFloat& operator=(const UStr& v)       {set(v); return *this;}
  /**< converts and sets value.
   * the string is supposed to contain a valid float */

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  UFloat& operator++()                   {incr(1.); return *this;}
  UFloat& operator++(int)                {incr(1.); return *this;}
  UFloat& operator--()                   {decr(1.); return *this;}
  UFloat& operator--(int)                {decr(1.); return *this;}
};


#endif
/* ==================================================== [TheEnd] ======= */
/* ==================================================== [Elc:03] ======= */
