/* ==================================================== ======== ======= *
 *
 *  ugadgets.hpp
 *  Ubit Project [Elc::2003]
 *  Author: Eric Lecolinet
 *
 *  Part of the Ubit Toolkit: A Brick Construction Game Model for Creating GUIs
 *
 *  (C) 1999-2003 Eric Lecolinet @ ENST Paris
 *  WWW: http://www.enst.fr/~elc/ubit   Email: elc@enst.fr (subject: ubit)
 *
 * ***********************************************************************
 * COPYRIGHT NOTICE : 
 * THIS PROGRAM IS DISTRIBUTED WITHOUT ANY WARRANTY AND WITHOUT EVEN THE 
 * IMPLIED WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. 
 * YOU CAN REDISTRIBUTE IT AND/OR MODIFY IT UNDER THE TERMS OF THE GNU 
 * GENERAL PUBLIC LICENSE AS PUBLISHED BY THE FREE SOFTWARE FOUNDATION; 
 * EITHER VERSION 2 OF THE LICENSE, OR (AT YOUR OPTION) ANY LATER VERSION.
 * SEE FILES 'COPYRIGHT' AND 'COPYING' FOR MORE DETAILS.
 * ***********************************************************************
 *
 * ==================================================== [Elc:03] ======= *
 * ==================================================== ======== ======= */

#ifndef _ugadgets_hpp_
#define	_ugadgets_hpp_
//pragma ident	"@(#)ugadgets.hpp	ubit:03.06.04"
#include <ubit/ustr.hpp>
#include <ubit/ubox.hpp>
#include <ubit/uedit.hpp>


/** Separator gadget
 */
class USepar: public UBox {
  int orient;
public:
  static UStyle *hstyle, *vstyle;

  USepar(const class UOrient&, const UArgs& a = UArgs::none);
  USepar(const UArgs& a = UArgs::none);

  friend USepar& usepar(const UArgs& a = UArgs::none);
  friend USepar& usepar(const class UOrient&, const UArgs& a = UArgs::none);
  friend USepar& uhsepar(const UArgs& a = UArgs::none);
  friend USepar& uvsepar(const UArgs& a = UArgs::none);

  virtual const UStyle& getStyle(UContext*) const;
  static  const UStyle& makeVStyle();
  static  const UStyle& makeHStyle();
};

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

/** Label gadget.
 *  Geometry:
 *  - horizontal lay out by default (add: UOrient::vertical otherwise). 
 *  - this object is resized when its children are changed (or resized).
 *  - consider UTextbox an subclasses for displaying textual messages
 *    that are dynamically changed.
 *
 *  Default Properties: 
 *  - font and foreground color are inherited from parents.
 *  - background is transparent (= UBgcolor::none)
 *
 *  Example:
 *  <br>  ULabel& my_label = ulabel( uima("whatever.jpg")+ " My Label" )
 *  <p>Note: 
 *  <br>  " My Label"  is equivalent to:  ustr(" My Label")
 */
class ULabel: public UBox {
public:
  static UStyle *style;

  ULabel(const UArgs& a = UArgs::none);
  ///< constructor; see also ~UGroup() and the <em>creator shortcut</em> <b>ulabel()</b>

  friend ULabel& ulabel(const UArgs& a = UArgs::none) {return *new ULabel(a);}
  ///< <em>creator shortcut</em> that is equivalent to *new ULabel().

  virtual const UStyle& getStyle(UContext*) const {return makeStyle();}
  static  const UStyle& makeStyle();
};

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */
/** Active Item gadget.
 *  Active gadget that behaves as a UButton but has no decoration. 
 *  It is typically used as a child of a UListbox or of a UTrow in a UTable 
 *  (when active items a prefered to UTcells).
 *
 *  Geometry: same as ULabel
 *
 *  Default Properties: same as ULabel when object is idle.
*
*  Callbacks:
* see UButton.
 */
class UItem: public UBox {
public:
  static UStyle *style;

  UItem(const UArgs& = UArgs::none);
  friend UItem& uitem(const UArgs& a = UArgs::none) {return *new UItem(a);}

  virtual const UStyle& getStyle(UContext*) const {return makeStyle();}
  static  const UStyle& makeStyle();
};

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */
/** Button gadget.
 *  Similar to the UItem gadget but has decorations. These decorations are
 *  determined in a dynamical way depending on the button's parents.
 *
 *  Geometry: same as ULabel
 *
 *  Default Properties: 
 *  - font and foreground color are inherited from parents.
 *  - background color depends on parent. It is transparent if the button
 *    is idle and inside a UMenu, a UMenubar or a UBar, and grey otherwise.
 *
*  Callbacks:
*  as for other UBox subclasses generic callback conditions make it
*  possible to specify callback methods and functions: see class UCond.
*  For instance:
*  - UOn::action activates callbacks when the user clicks the button
*  - UOn::arm activates callbacks when the user arms the button
*  - UOn::disarm activates callbacks when the user disarm the button
*
* Exemple:
*  <pre>
      XXX* obj = ...;
      UButton& btn = ubutton( UPix::diskette + " Save..."
                              + UOn::action / ucall(obj, "save", &XXX::foo)
                              + UOn::arm / ucall(obj, "save", &XXX::foo)
                              );
* </pre>
*
* foo is a method of XXX (non member functions could also be specified: see class UCall).
* a method callback can have 0 to 2 arguments.
*
* <pre>
    void XXX::foo(UEvent& e, const char* msg) {
     if (e.getSource()) {          // returns the button
       if (e.getCond == &UOn::action)
         cout << "Action, arg= " << msg << endl;
       else if (e.getCond == &UOn::arm)
         cout << "Arm, arg= " << msg << endl;
     }
* </pre>
*/
class UButton: public UBox {
public:
  static UStyle *style, *menuStyle, *barStyle, *flatStyle, *linkStyle;
  ///< contextual styles depending on the Button's parent

  UButton(const UArgs& a = UArgs::none);
  ///< constructor; see also ~UGroup() and the <em>creator shortcut</em> <b>ubutton()</b>

  friend UButton& ubutton(const UArgs& a = UArgs::none) {return *new UButton(a);}
  ///< <em>creator shortcut</em> that is equivalent to *new UButton().

  virtual const UStyle& getStyle(UContext*) const;
  /**<
   * returns the contextual UStyle of this object.
   *  This virtual function calls one of the makeXxxStyle() functions that
   *  are defined for this specific class.
   * - see also: UStyle and makeXxxStyle() functions
    */

  static const UStyle& makeStyle();
  static const UStyle& makeMenuStyle();
  static const UStyle& makeBarStyle();
  static const UStyle& makeFlatStyle();
  static const UStyle& makeLinkStyle();
};

/* ==================================================== ======== ======= */
/** Flat Button gadget.
 * has a "flat" square border
 */
class UFlatbutton: public UButton {
public:
  UFlatbutton(const UArgs& a = UArgs::none);
  friend UFlatbutton& uflatbutton(const UArgs& a = UArgs::none) {return *new UFlatbutton(a);}

  virtual const UStyle& getStyle(UContext*) const {return UButton::makeFlatStyle();}
};

/* ==================================================== ======== ======= */
/** Link Button gadget.
 * resembles an HTML link (no border and specific properties)
 */
class ULinkbutton: public UButton {
public:
  ULinkbutton(const UArgs& a = UArgs::none);
  friend ULinkbutton& ulinkbutton(const UArgs& a = UArgs::none) {return *new ULinkbutton(a);}

  virtual const UStyle& getStyle(UContext*) const {return UButton::makeLinkStyle();}
};

/* ==================================================== ======== ======= */
/** Radio Button gadget.
 */
class URadiobutton: public UButton {
public:
  URadiobutton(const UArgs& a = UArgs::none);
  friend URadiobutton& uradiobutton(const UArgs& a = UArgs::none) {return *new URadiobutton(a);}

  virtual const UStyle& getStyle(UContext*) const;
};

/* ==================================================== ======== ======= */
/** Checkbox gadget
*
* Callbacks
* see UButton. in addition:
* - UOn::select activates callbacks when the the checkbox is selected
* - UOn::unselect activates callbacks when the the checkbox is deselected
* Note:
*  - UOn::action activates callbacks when the user clicks the checkbox.
*    => action is performed if the user selects or deselects the checkbox
*    by clicking on it (but it is not activated if the checkbox is selected
*    or deselected by a function call or a radio selection behavior)
*    
 */
class UCheckbox: public UBox {
public:
  /// three contextual styles depending on the Button's parent
  static UStyle *style, *menuStyle, *barStyle;

  UCheckbox(const UArgs& = UArgs::none);
  friend UCheckbox& ucheckbox(const UArgs& a = UArgs::none) {return *new UCheckbox(a);};

  virtual const UStyle& getStyle(UContext*) const;
};

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */
/** Textbox gadget: base class for handling text.
 *
 *  The main puropose of this class is to display text. 
 *  Use UTextfield to edit a single line of text and UFlowbox or UTextarea
 *  to edit multi-line text regions.
 *
 *  By opposition to UBox objects, UTextbox, UTexfield, UFlowbox... 
 *  keep their initial size. For this reason, they are well suited for
 *  displaying text that is dynamically changed (the layout of the GUI
 *  will be more stable and calculated faster).
 *
 *  Geometry (default behavior):
 *  - horizontal lay out (add: UOrient::vertical otherwise). 
 *  - keeps its initial size (calculated to make children that were added
 *    at creation time visible) except if it is included in a parent that
 *    enforces "flexible" lay out (see uvflex()/uhflex())
 *  - width and height can also be set by adding UWidth/UHeight children
 *
 *  Default Properties: 
 *  - the font and the foreground color are inherited from parents.
 *  - the background is transparent (= UBgcolor::none)
 *
 *  Examples:
 *  <pre>
 *      UTextbox& tb1 = utextbox( "This textbox will display this string" )
 *
 *      UStr& msg = ustr("error message");
 *      UTextbox& tb2 = utextbox( UPix::ray + "Warning: Error: " + msg );
 *
 *      msg = "another message";    // changes 'msg' and updates 'txb2'
 *
 *      // collates the text enclosed in the textbox and copies it to 's'
 *      // 's' will contain:  "Warning: Error: another message"
*      UStr s = tb2.copyText();
 *  </pre>
 *
 *  Callbacks and Editable text:
 *  The UTextfield subclass provides text edition:
 *  <pre>
 *      utextfield( "bla bla" ); 
 *  </pre>
 *  This is functionally equivalent to:
 *  <pre>
 *      utextbox( uedit() + "bla bla" );
 *  </pre>
*  See:  classes UTextfield and UEdit for callbacks and other functions.
*/
class UTextbox: public UBox {
public:
  static UStyle *style;

  UTextbox(const UArgs& = UArgs::none);
  ///< constructor; see also ~UGroup() and the <em>creator shortcut</em> <b>utextbox()</b>

  friend UTextbox& utextbox(const UArgs& a = UArgs::none) {return *new UTextbox(a);}
  ///< <em>creator shortcut</em> that is equivalent to *new UTextbox().

  virtual const UStyle& getStyle(UContext*) const {return makeStyle();}
  static  const UStyle& makeStyle();
};

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */
/** Text field gadget: single line editor.
 *
 * A UTextfield is a UTextbox that can edit text.
*  <pre>
*      utextfield( "bla bla" );  
*  </pre>
*  is functionally equivalent to:
*  <pre>
*      utextbox( uedit() + "bla bla" );
*  </pre>
*
*   A textbox can contain arbitrary children (for instance pixmap images,
*   other gadgets...) If a textbox contain several UStr(s) all of them
*   are editable.
*  <pre>
*      UStr& msg = ustr("my message");
*     utextfield( UPix::doc + "Error: " + UPix::ray + msg )
*  </pre>
*
*  Geometry:
*  Same Geometry and Properties as UTextbox except that the text is
 *  editable and the default background is white. This gadget keeps its 
 *  initial size (which is calculated to make children that were added
 *  at creation time visible). See UTextbox for details.
*
*  Callbacks:
*  as for other UBox subclasses generic callback conditions make it 
*  possible to specify callback methods and functions: see class UCond.
*  For instance:
*  - UOn::action activates callbacks when the user hits the Return key
*  - UOn::strChange activates callbacks when the text is changed
*
 * Exemple:
 *  <pre>
 *      XXX* obj = ...;
 *      UStr& msg = ustr("error message");
 *      utextfield( UPix::ray + msg
 *                + UOn::action / ucall(obj, &XXX::foo1)
 *                + UOn::strChange / ucall(obj, &msg, &XXX::foo2)
 *               )
 * </pre>
 * foo1 and foo2 are XXX methods (non member functions can also
 * be specified: see class UCall). There are several ways to retreive
 * the text in callback functions: foo1() retreives all the text 
 * included in the textfield while foo2() passes 'msg' as an argument.
 * The text can be changed by foo2() but not by foo1().
 *
 * <pre>
 *  void XXX::foo1(UEvent& e) {
 *    if (e.getSource()) {          // returns the textfield
 *      // collates the text enclosed in the textfield and copies it to 's'
 *      UStr s = e.getSource()->copyText();
 *      cout << s << endl;
 *    }
 *
 *  void XXX::foo2(UStr* s) {
 *    // 's' points to 'msg'. 'msg' is changed if 's' is changed.
 *    cout << *s << endl;
 *  }
 * </pre>
*/
class UTextfield: public UTextbox {
public:
  UTextfield(const UArgs& = UArgs::none);
  ///< constructor; see also ~UGroup() and the <em>creator shortcut</em> <b>utextfield()</b>

  friend UTextfield& utextfield(const UArgs& a = UArgs::none) 
  {return *new UTextfield(a);}
  ///< <em>creator shortcut</em> that is equivalent to *new UTextField().

  UEdit& edit() {return *pedit;}
  const UEdit& edit() const {return *pedit;}

  virtual void setEditable(bool state = true) {edit().setEditable(state);}
  virtual bool isEditable() const {return edit().isEditable();}

private:
  uptr<UEdit> pedit;
};

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */
/** List gadget.
 * Vertical (or horizontal) list of selectable gadgets.
 *
 * A listbox gadget can contain any brick but only ARMable gadgets (UItem,
 * UButton, UCheckbox...) will be *selectable*. Use UItem children to obtain
* the usual appearance of a list gadget. Strings that are directly added
* to the listbox are visible but cant be seletecd.
 *
 * Listbox children (for instance selectable UItem children) can contain 
 * an arbitrary combination of text, images, gadgets...
 *
 * Geometry:
 *  - vertical layout (add: UOrient::horizontal otherwise).
 *  - size is dynamically updated in order to make all children visible
 *  - width and height can also be set by adding UWidth/UHeight children
 *
 *  Default Properties: 
 *  - the font and the foreground color are inherited from parents.
 *  - the default background is white (add an UBgcolor object before
 *    gadget children otherwise)
 *
 *  Selection management:
 * - UListbox::choice() returns the UChoice object that controls the
 *   selection in the listbox.
 *
 * Exemple:
 * <pre>
 *    XXX* obj = ...;
 *    UListbox& list = ulistbox
 *    (
 *      ulabel("title")         // can't be selected
 *      + uitem("first item")
 *      + uitem("second item")
 *      + uitem(UPix::right + "third" + " item" + ubutton(...))
        + UOn::action / ucall(obj, &XXX::foo)
        + UOn::select / ucall(obj, &XXX::foo)
 *     );
 *
 *    list.add(uitem("4th item"));

 *    UGroup* item = list.choice().getItem());   // returns the selected item
 *    int index = list.choice().getIndex())      // returns the index of the selected item
 * </pre>
*
*  Callbacks:
*  as for other UBox subclasses generic callback conditions make it
*  possible to specify callback methods and functions: see class UCond.
*  For instance:
*  - UOn::select activates callbacks when a *selectable* item of the list
*    is selected.
*  - UOn::action activates callbacks when a *selectable* item of the list
*    is selected, reselected or deselected
*
* <pre>
* void XXX::foo(UEvent& e) {
*     // getSource() returns the listbox
*     UListbox* list = e.getSource() ? dynamic_cast<UListbox*>(e.getSource()) : null;
*
*    if (list) {
*      UGroup* item = list->choice().getItem());   // returns the selected item
*      int index = list->choice().getIndex())      // returns the index of the selected item
*    }
*
*    // getTarget() returns the selected item (same as list->choice().getItem())
*    if (e.getTarget()) {
*      // collates the text that is enclosed in the item and copies it to 's'
*      UStr s = e.getTarget()->copyText();
*    }
* }
* </pre>
 */
class UListbox: public UBox {
public:
  static UStyle *style;

  UListbox(const UArgs& a = UArgs::none);

  friend UListbox& ulistbox(const UArgs& a = UArgs::none)
  {return *new UListbox(a);}

  virtual const UStyle& getStyle(UContext*) const {return makeStyle();}
  static  const UStyle& makeStyle();

  class UChoice& choice() const {return *pchoice;}

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  // inherited: add(), addlist(), etc
  
  UListbox& addItem(UStr*);
  UListbox& addItems(const std::vector<UStr*>& str_list);  
  UListbox& addItems(const UArgs& item_prefix, const std::vector<UStr*>& str_list);
  ///< UStr strings are *NOT* duplicated and should not be destroyed.

  UListbox& addItem(const char*);
  UListbox& addItems(const char* string_list[]);
  UListbox& addItems(const UArgs& item_prefix, const char* string_list[]);
  ///< char* strings are duplicated and can be destroyed.

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  // implementation

  // modif: UGroup::getBrowsingGroup() fait le necessaire
  // virtual UGroup* getBrowsingGroup() {return this;}
  
protected:
  uptr<class UChoice> pchoice;
  uptr<UCall> callbacks;
  virtual void mouseCB(UEvent&);
};

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */
/** Combo box gadget.
*
*  Geometry:
*  Same Geometry and Properties as UTextfield. This gadget keeps its
*  initial size (which is calculated to make the first child
                 *  of the list entirely visible).
*
*  List and Selection management:
* - list() returns the list that was passed as an argument to the constructor
* - choice() returns the UChoice object that manages the selection in list()
*
*  Text mode means that the combox box only displays the text of the
*  selected item. Non-text mode (the default) means that the actual content
*  of the selected item is displayed inside the combo box.
*
* Exemple
* <pre>
 * XXX* obj = ...;
* 
* UCombobox& files =
   ucombobox( ulistbox(uitem(UPix::doc + "~/ubit/")
 *            + uitem(UPix::doc + "~/ubit/campus/")
 *            + uitem(UPix::doc + "~/ubit/campus/examples/"),   // dont forget the comma
 *            // this is a callback of the Combobox, not a callback of the Listbox
 *            UOn::action / ucall(rem, target, "file:", &XXX::foo)
            );
*
*  void ObjClass::foo(UEvent& e) {
  *    if (e.getSource()) {          // returns the UCombobox
    *      // collates the text enclosed in the UCombobox and copies it to 's'
    *      UStr s = e.getSource()->copyText();
    *      cout << s << endl;
    *    }  
</pre>
*/
class UCombobox : public UTextfield  {
public:
  UCombobox(class UListbox& list, const UArgs& = UArgs::none);

  friend UCombobox& ucombobox(class UListbox& list, const UArgs& a = UArgs::none)
  {return *new UCombobox(list, a);}

  class UListbox& list() {return *plist;}
  const class UListbox& list() const {return *plist;}

  class UChoice& choice();
  const class UChoice& choice() const;

  bool isTextMode() const {return text_only;};
  void setTextMode(bool = true);
  UCombobox& textMode(bool st = true) {setTextMode(st); return *this;}

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

private:
    uptr<class UListbox> plist;
  uptr<UGroup> pentry;
  uptr<class UMenu> pmenu;
  bool text_only;
  virtual void changed(bool upd, bool call_cb);
  virtual void openMenu(UEvent&);
};

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */
/** Spinbox gadget.
 */
class USpinbox : public UBox {
public:
  static UStyle *style;

  USpinbox(const UArgs& = UArgs::none);
  ///< standard constructor.

  USpinbox(UIntg& value, const UArgs& = UArgs::none);
  USpinbox(UIntg& value, UIntg& increment, const UArgs& = UArgs::none);
  /**< 
   * constructors for specifing a shared 'value' or 'increment'.
   * Arguments 'value' and 'increment':
   * - are NOT duplicated internally and CAN'T be destroyed by 'delete'
   *   (nor be allocated in the stack).
   * - can be shared by other objects and their value can be changed by clients
   * - are automatically deleted when the Spinbox is deleted if no uptr 
   *   points to them.
   *
   * UAction callbacks are fired when value changes
   */

  friend USpinbox& uspinbox(const UArgs& a = UArgs::none)
  {return *new USpinbox(a);}

  friend USpinbox& uspinbox(UIntg& value, const UArgs& a = UArgs::none)
  {return *new USpinbox(value, a);}
  ///< <em>creator shortcut</em> that is equivalent to *new USpinbox().

  virtual const UStyle& getStyle(UContext*) const {return makeStyle();}
  static  const UStyle& makeStyle();

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  virtual void setValue(const UIntg&);
  virtual void setIncrement(const UIntg&);
  virtual void setStr(const UStr&);

  UIntg& value()     const {return *pvalue;}
  UIntg& increment() const {return *pincrement;}
  UStr&  str()       const {return *pstr;}
 /**< 
  * direct access to internal fields.
  * these objects can be shared by other objects and their value can be changed 
  * by clients. They are automatically deleted when the Spinbox is deleted 
  * if no uptr points to them.
  */

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  // implementation

  virtual void constructs(const UArgs& a);
  virtual void valueChanged();
  virtual void updateValue(int dir);

private:
  uptr<UIntg> pvalue, pincrement;
  uptr<UStr>  pstr;
  UTextbox textbox;
};

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */
/** Alertbox gadget
 */
class UAlertbox : public UVbox {
  UGroup message;

public:
  UAlertbox(const UArgs& = UArgs::none);
  virtual void show(bool = true);
  virtual void show(const UStr& mesg);
};

#endif
/* ==================================================== [TheEnd] ======= */
/* ==================================================== [Elc:03] ======= */
