/* ==================================================== ======== ======= *
 *
 *  udisp.hpp
 *  Ubit Project  [Elc][2003]
 *  Author: Eric Lecolinet
 *
 *  Part of the Ubit Toolkit: A Brick Construction Game Model for Creating GUIs
 *
 *  (C) 1999-2003 Eric Lecolinet @ ENST Paris
 *  WWW: http://www.enst.fr/~elc/ubit   Email: elc@enst.fr (subject: ubit)
 *
 * ***********************************************************************
 * COPYRIGHT NOTICE : 
 * THIS PROGRAM IS DISTRIBUTED WITHOUT ANY WARRANTY AND WITHOUT EVEN THE 
 * IMPLIED WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. 
 * YOU CAN REDISTRIBUTE IT AND/OR MODIFY IT UNDER THE TERMS OF THE GNU 
 * GENERAL PUBLIC LICENSE AS PUBLISHED BY THE FREE SOFTWARE FOUNDATION; 
 * EITHER VERSION 2 OF THE LICENSE, OR (AT YOUR OPTION) ANY LATER VERSION.
 * SEE FILES 'COPYRIGHT' AND 'COPYING' FOR MORE DETAILS.
 * ***********************************************************************
 *
 * ==================================================== [Elc:03] ======= *
 * ==================================================== ======== ======= */

#ifndef _udisp_hpp_
#define	_udisp_hpp_
//pragma ident	"@(#)udisp.hpp	ubit:03.05.06"
#include <ubit/ustr.hpp>
#include <ubit/ugroup.hpp>

/**
 */
typedef std::vector<class UFlow*> UFlowList;
typedef std::vector<class UDisp*> UDispList;

/* ==================================================== ======== ======= */
/** Display Context.
 *  this class modelizes the physical X Display.
 *  it is the base class of UAppli, the Application Context.
 */
class UDisp {
public:
  enum Stat {
    CannotOpen    = -1,
    NotOpened     =  0,
    Opened        =  1,
    OpenGL        =  2
  };

  UDisp(int disp_id, class UAppli&, const UStr& display_name);
  UDisp(int disp_id, class UAppli&, class UConf&);
  /**<
   * creates a new logical Display.
   * Args:
    * - 'display_name' name of the X Window server:
    * - syntax: hostname[:screen_number]
    * - hostname:0 is the default if [:screen_number] is omitted
     * - depth and color mode can be specified in the UConf argument. 
   *   Default is 24 bits true colors.
   * - see also: UDisp::setTrueColors() and class UConf.
   */

  virtual ~UDisp();

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  int getID() const  {return id;}
  ///< returns the ID of this logical display.

  class UAppli& getAppli() const {return appli;}
  ///< returns the corresponding Application Context.

  const class UConf& getConf() const {return conf;}
  ///< returns the configuration of this Display.

  int getStatus() const;
  /**< returns connection status (see: enum UDisp::Stat).
   *  value is > 0 if connected to an X server (and == UDisp::OpenGL 
   *  if connected in OpenGL mode)
   */

  virtual bool isRealized() const;
  ///< is this Display initialized?.

  virtual void realize();
  /**<
   * realizes this display (allocates the X resources).
   * this function does not need to be explicitely called except if you 
   * need to draw Graphics before calling the mainLoop.
   * The UError exception is thrown if the UDisp can't be realized
   */

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  // Windows

  virtual void add(class UWin*);
  virtual void add(class UWin&);
  /**< 
   * add a window (UFrame, UDialog, etc) to this logical display.
   * notes:
   * - at least one window (typically an UFrame) should be added to
   *   the UAppli instance
   * - the "Main Frame" is the first UFrame that is added to the UAppli
   * - windows are initially hidden (including the Main Frame), their
   *   show() method must be called to make them visible
   * - window children must be added before calling the show() method
   *   (as window sizes won't change after this call)
   * - See also: classes UWin, UFrame, UDialog, UMenu.
   */ 

  virtual void remove(UWin& child, int remove_mode);
  virtual void remove(UWin* child, int remove_mode);
  ///< same as UGroup::remove().

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  // Multiple Event Flows.

  class UFlow* openFlow(int flow_id);
  /**< creates a new event flow (or returns an existing flow if ID already used).
   */

  void closeFlow(class UFlow*);
  ///< [unstable].

  class UFlow* getFlow(int flow_id);
  /**< returns the corresponding Event Flow (null otherwise).
   * NB: the ID of the standard X event flow is 0.
   */

  unsigned int getFlowCount() const;
  const UFlowList& getFlows() const {return flowlist;}

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  // Graphics.

  int setTrueColors(int depth_hint);
  int setPseudoColors(int depth_hint);
  /**< 
   * tries to set TrueColor or PseudoColor mode with this depth.
   * Arguments:
   * - depth and color mode can be specified in the UConf argument 
   *   of the UDisp constructor. Default is 24 bits true colors.
   * - the 'depth' argument is a hint: the visual with the closest depth 
   *   available will be used
   * - returns the actual depth that was set. 0 means that no compatible
   *   depth could be found for this color mode
   * Notes:
   * - TrueColor mode should be used whenever possible because it is
   *   requested for alpha blending (transparency)
   * - this fct. must be called before realize() or mainLoop()
   */

  int getScreenWidth() const;
  int getScreenHeight() const;
  ///< size of the Screen used by this UDisp.

  int getDepth() const;
  ///< depth of this specific UDisp.

  int getScreenDefaultDepth() const;
  /**< default depth of the Screen used by this UDisp.
   * Note: this value may differ from getDepth() */

  int getScreenNumber() const;
  ///< returns the number of the Screen of this UDisp.

  int getScreenCount() const;
  //< returns the total number of Screens on this X Server.

  const UStr& getDisplayName() const;
  ///< name of the X server (default = null).

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  // implementation

  class UNatDisp* getNatDisp() const {return natdisp;}
  ///< [impl] implementation dependant data.

  void deleteNotify(class UView* deleted_view);
  void deleteNotify(class UGroup* deleted_group);
  ///< [impl] notifies the UDisp that a view/a widget is being destroyed.

#ifndef NO_DOC

  // will manage the events of this window if it is realized and shown
  void addHardwin(class UWin*);
  void removeHardwin(class UWin*);

  bool realizeFont(const class UFont&);
  bool realizeColor(class UColor&);
  bool realizeCursor(class UCursor&);

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 
protected:
  friend class UConf;
  friend class UWin;
  friend class UGraph;
  friend class UWinGraph;
  friend class UNatDisp;

  const int id;
  class UAppli&   appli;         // corresponding UAppli
  class UConf&    conf;          // configuration data (may be shared with UAppli)
  UGroup          winlist;       // list of windows belonging to this display
  UChain          hardwin_list;	 // list of hardwins that are event managed
  UFlowList       flowlist;      // list of event flows (doit etre detruit en dernier!)
  UStr      display_name;  // name of the X Display
  class UNatDisp* natdisp;       // native Display
#endif
};


#endif
/* ==================================================== [TheEnd] ======= */
/* ==================================================== [Elc:03] ======= */
