//**********************************************************************
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <string>
#include <map>
#include <vector>
#include <qmessagebox.h>
#include <qpushbutton.h>
#include <qapplication.h>
#include <qcolordialog.h>
#include <qfiledialog.h>
#include <qinputdialog.h>
#include <qworkspace.h>
#include <limits.h>
#include <qmenubar.h>
#include <qtable.h>
#include <qvbox.h>
#include <qstatusbar.h>
#include <qpixmap.h>
#include <qstrlist.h>
#include <qimage.h>
#include <qpainter.h>
#include <qprogressdialog.h>
#include <qdockwindow.h>
#include <qlayout.h>
#include <qcombobox.h>
#include <qcursor.h>
#include <qaction.h>
#include <qradiobutton.h>
#include <qprinter.h>
#include <qregexp.h>
#include <qvalidator.h>
#include <qdatetime.h>
#include <fstream>
#include <iostream>
#include <unistd.h>

#include <tulip/TlpTools.h>
#include <tulip/Reflect.h>
#include <tulip/GlGraphWidget.h>
#include <tulip/TulipElementProperties.h>
#include <tulip/PropertyWidgets.h>
#include <tulip/ClusterTree.h>
#include <tulip/PropertyProxy.h>
#include <tulip/SelectionProxy.h>
#include <tulip/SizesProxy.h>
#include <tulip/ColorsProxy.h>
#include <tulip/MetaGraphProxy.h>
#include <tulip/TlpQtTools.h>
#include <tulip/StableIterator.h>

#include "FindSelectionData.h"
#include "NavigateGlGraph.h"
#include "PropertyDialog.h"
#include "viewGl.h"
#include "Application.h"
#include "QtProgress.h"
#include "ElementInfoToolTip.h"
#include "TabWidgetData.h"
#include "Overview.h"
#include "ToolBar.h"
#include "InfoDialog.h"

#define WITHQT3


#define	MORPHING_MAX_FPS		30
#define	MORPHING_MIN_DURATION	1.0f		// s
#define	MORPHING_MAX_DURATION	3.0f		// s
#define	MORPHING_MIN_FRAMES		8			// #



using namespace std;
using namespace tlp;

//**********************************************************************
///Constructor of ViewGl
viewGl::viewGl(QWidget* parent,	const char* name):TulipData( parent, name ) {
  //  cerr << __PRETTY_FUNCTION__ << endl;
  Observable::holdObservers();
  glWidget=0;
  aboutWidget=0;
  copyCutPasteGraph = 0;
  //=======================================
  buildMenus();
  //MDI
  workspace->setScrollBarsEnabled( TRUE );
  connect (workspace, SIGNAL(windowActivated(QWidget *)), this, SLOT(windowActivated(QWidget *)));
  //Create overview widget
  overviewDock = new QDockWindow(this,"Overview");
  overviewDock->setCloseMode(QDockWindow::Always);
  overviewDock->setResizeEnabled(true);
  overviewWidget = new Overview(overviewDock);
  overviewDock->boxLayout()->add(overviewWidget);
  this->addDockWindow(overviewDock,"Overview", Qt::DockLeft);
  overviewWidget->view->getGlGraph()->setBackgroundColor(Color(255,255,255));
  overviewWidget->show();
  overviewDock->show();
  //Create Data information editor (Hierarchy, Element info, Property Info)
  tabWidgetDock = new QDockWindow(this,"Data manipulation");
  tabWidgetDock->setCloseMode(QDockWindow::Always);
  tabWidgetDock->setResizeEnabled(true);
  TabWidgetData *tabWidget = new TabWidgetData(tabWidgetDock);
  tabWidgetDock->boxLayout()->add(tabWidget);
  this->addDockWindow(tabWidgetDock,"Data manipulation", Qt::DockLeft);
  tabWidget->show();
  tabWidgetDock->show();
  //Create toolbar widget
  mouseToolBarDock = new QDockWindow(this,"Tool Bar");
  mouseToolBarDock->setCloseMode(QDockWindow::Always);
  mouseToolBar = new ToolBar(mouseToolBarDock);
  mouseToolBarDock->boxLayout()->add(mouseToolBar);
  this->addDockWindow(mouseToolBarDock,"ToolBar", Qt::DockRight);
  mouseToolBar->show();
  mouseToolBarDock->show();
  //Init hierarchy visualization widget
  clusterTreeWidget=tabWidget->clusterTree;
  //Init Property Editor Widget
  propertiesWidget=tabWidget->propertyDialog;
  //Init Element info widget
  nodeProperties = tabWidget->elementInfo;
  ((Application*)qApp)->nodeProperties = nodeProperties;
  //connect signals related to supergraph replacement
  connect(clusterTreeWidget, SIGNAL(supergraphChanged(SuperGraph *)), 
	  this, SLOT(hierarchyChangeSuperGraph(SuperGraph *)));
  connect(clusterTreeWidget, SIGNAL(aboutToRemoveView(SuperGraph *)), this, SLOT(superGraphAboutToBeRemoved(SuperGraph *)));
  connect(clusterTreeWidget, SIGNAL(aboutToRemoveAllView(SuperGraph *)), this, SLOT(superGraphAboutToBeRemoved(SuperGraph *)));
  //+++++++++++++++++++++++++++
  //Connection of the menu
  connect(&stringMenu     , SIGNAL(activated(int)), SLOT(changeString(int)));
  connect(&metricMenu     , SIGNAL(activated(int)), SLOT(changeMetric(int)));
  connect(&layoutMenu     , SIGNAL(activated(int)), SLOT(changeLayout(int)));
  connect(&selectMenu     , SIGNAL(activated(int)), SLOT(changeSelection(int)));
  connect(clusteringMenu  , SIGNAL(activated(int)), SLOT(makeClustering(int)));
  connect(&sizesMenu      , SIGNAL(activated(int)), SLOT(changeSizes(int)));
  connect(&intMenu        , SIGNAL(activated(int)), SLOT(changeInt(int)));
  connect(&colorsMenu     , SIGNAL(activated(int)), SLOT(changeColors(int)));
  connect(&exportGraphMenu, SIGNAL(activated(int)), SLOT(exportGraph(int)));
  connect(&importGraphMenu, SIGNAL(activated(int)), SLOT(importGraph(int)));
  connect(&exportImageMenu, SIGNAL(activated(int)), SLOT(exportImage(int)));
  connect(&dialogMenu     , SIGNAL(activated(int)), SLOT(showDialog(int)));
  windowsMenu->setCheckable( TRUE );
  connect(windowsMenu, SIGNAL( aboutToShow() ), this, SLOT( windowsMenuAboutToShow() ) );
  Observable::unholdObservers();
  //  cerr << "Finished" << endl << flush;
}
//**********************************************************************
void viewGl::update ( ObserverIterator begin, ObserverIterator end) {
  Observable::holdObservers();
  clearObservers();
  //  cerr << __PRETTY_FUNCTION__ << endl;
  nodeProperties->updateTable();
  //  cin >> i;
  propertiesWidget->update();
  //  cin >> i;
  redrawView();
  //  cin >> i;
  Observable::unholdObservers();
  initObservers();
}
//**********************************************************************
static const unsigned int NB_VIEWED_PROPERTIES=8;
static const string viewed_properties[8]= {"viewLabel",
					   "viewColor",
					   "viewSelection",
					   "viewMetaGraph",
					   "viewShape",
					   "viewSize",
					   "viewTexture",
					   "viewLayout" };
void viewGl::initObservers() {
  //  cerr << __PRETTY_FUNCTION__ << endl;
  if (!glWidget) return;
  SuperGraph *graph=glWidget->getSuperGraph();
  if (graph==0) return;
  for (unsigned int i=0;i<NB_VIEWED_PROPERTIES;++i) {
    graph->getProperty(viewed_properties[i])->addObserver(this);
  }
}
void viewGl::clearObservers() {
  //  cerr << __PRETTY_FUNCTION__ << endl;
  if (glWidget==0) return;
  SuperGraph *graph=glWidget->getSuperGraph();
  if (graph==0) return;
  for (unsigned int i=0;i<NB_VIEWED_PROPERTIES;++i) {
    graph->getProperty(viewed_properties[i])->deleteObservers();
  }
}
//**********************************************************************
///Destructor of viewGl
viewGl::~viewGl() {
  //  cerr << __PRETTY_FUNCTION__ << endl;
}
//**********************************************************************
void viewGl::changeSuperGraph(SuperGraph *graph) {
  //  cerr << __PRETTY_FUNCTION__ << endl;
  clearObservers();
  clusterTreeWidget->setSuperGraph(graph);
  propertiesWidget->setSuperGraph(graph);
  nodeProperties->setSuperGraph(graph);
  propertiesWidget->setGlGraphWidget(glWidget->getGlGraph());
  overviewWidget->setObservedView(glWidget->getGlGraph());
  updateSatutBar();
  redrawView();
  initObservers();
}
//**********************************************************************
void viewGl::hierarchyChangeSuperGraph(SuperGraph *graph) {
  //  cerr << __PRETTY_FUNCTION__ << " (SuperGraph = " << graph->getId() << ")" << endl;
  if (glWidget->getSuperGraph()==graph) return;
  clearObservers();
  glWidget->setSuperGraph(graph);  
  changeSuperGraph(graph);
  initObservers();
}
//**********************************************************************
void viewGl::windowActivated(QWidget *w) {
  //  cerr << __PRETTY_FUNCTION__ << " (QWidget = " << (int)w << ")" << endl;
  if (w==0) return;
  if (typeid(*w)==typeid(NavigateGlGraph)) {
    glWidget=((NavigateGlGraph *)w)->getGlGraphWidget();
    changeSuperGraph(glWidget->getSuperGraph());
  }
}
//**********************************************************************
GlGraphWidget * viewGl::newOpenGlView(SuperGraph *graph,const QString &name) {
  //Create 3D graph view
  //  cerr << __PRETTY_FUNCTION__ << endl;
  NavigateGlGraph *glWidget1 = new NavigateGlGraph(workspace, "3D Graph View",graph);
  glWidget1->move(0,0);
  glWidget1->setCaption(name);
  glWidget1->show();
  GlGraphWidget *glWidget = glWidget1->getGlGraphWidget();
  glWidget1->setMinimumSize(0, 0);
  glWidget1->setMaximumSize(32767, 32767);
  glWidget1->setFocusPolicy(QWidget::NoFocus);
  glWidget1->setBackgroundMode(QWidget::PaletteBackground);  
  glWidget ->installEventFilter(this);
  glWidget ->setMouse(mouseToolBar->getCurrentMouse());
  connect(mouseToolBar,   SIGNAL(mouseChanged(MouseInterface *)), glWidget, SLOT(setMouse(MouseInterface *)));
  connect(glWidget,       SIGNAL(nodeClicked(SuperGraph *, const node &)), 
	  nodeProperties, SLOT(setCurrentNode(SuperGraph*, const node &)));
  connect(glWidget,       SIGNAL(edgeClicked(SuperGraph *, const edge &)), 
	  nodeProperties, SLOT(setCurrentEdge(SuperGraph*, const edge &)));

  new ElementInfoToolTip(glWidget,"toolTip",glWidget);
  QToolTip::setWakeUpDelay(2500);
  changeSuperGraph(graph);
  //  cerr << __PRETTY_FUNCTION__ << "...END" << endl;
  return glWidget;
}
//**********************************************************************
void viewGl::new3DView() {
  //  cerr << __PRETTY_FUNCTION__ << endl;
  DataSet param=glWidget->getGlGraph()->getParameters();
  QString name(glWidget->name());
  newOpenGlView(glWidget->getSuperGraph(),glWidget->parentWidget()->caption());
  glWidget->getGlGraph()->setParameters(param);
  //  cerr << __PRETTY_FUNCTION__ << "...END" << endl;
}
//**********************************************************************
void viewGl::fileNew() {
  Observable::holdObservers();
  SuperGraph *newSuperGraph=tlp::newSuperGraph();
  newSuperGraph->setAttribute("name",string("unamed"));
  newSuperGraph->getProperty<SizesProxy>("viewSize")->setAllNodeValue(Size(1,1,1));
  newSuperGraph->getProperty<SizesProxy>("viewSize")->setAllEdgeValue(Size(0.125,0.125,0.5));
  newSuperGraph->getProperty<ColorsProxy>("viewColor")->setAllNodeValue(Color(255,0,0));
  newSuperGraph->getProperty<ColorsProxy>("viewColor")->setAllEdgeValue(Color(0,0,0));
  newSuperGraph->getProperty<IntProxy>("viewShape")->setAllNodeValue(1);
  newSuperGraph->getProperty<IntProxy>("viewShape")->setAllEdgeValue(0);
  GlGraph *glW=newOpenGlView(newSuperGraph,"Unamed")->getGlGraph();
  glW->setViewArrow(true);
  glW->setDisplayEdges(true);
  glW->setFontsType(1);
  glW->setViewLabel(true);
  glW->setBackgroundColor(Color(255,255,255));
  glW->setViewOrtho(true);
  glW->setViewStrahler(false);
  glW->setEdgeColorInterpolate(false);
  overviewWidget->syncFromView();
  redrawView();
  Observable::unholdObservers();
}
//**********************************************************************
void viewGl::fileSave() {
  if (!glWidget) return;
  if (openFiles.find((unsigned int)glWidget)==openFiles.end()) {
    fileSaveAs();
    return;
  }
  fileSave("tlp", openFiles[(unsigned int)glWidget]);
}
//**********************************************************************
bool viewGl::fileSave(string plugin, string filename) {
  if (!glWidget) return false;
  ostream *os;
  if (filename.rfind(".gz") == (filename.length() - 3))
    os = tlp::getOgzstream(filename.c_str());
  else
    os = new ofstream(filename.c_str());
  DataSet dataSet;
  dataSet.set("displaying", glWidget->getGlGraph()->getParameters());
  bool result;
  if (!(result=tlp::exportGraph(glWidget->getSuperGraph(), *os, plugin, dataSet, NULL))) {
    QMessageBox::critical( 0, "Tulip export Failed",
			   "The file has not been saved"
			   );
  }
  delete os;
  return result;
}
//**********************************************************************
void viewGl::fileSaveAs() {
  if (!glWidget) return;
  if( !glWidget->getSuperGraph() )
  	return;
  QString name = QFileDialog::getSaveFileName( QString::null,
					       tr("Tulip graph (*.tlp *.tlp.gz)"),
					       this,
					       tr("open file dialog"),
					       tr("Choose a file to open" ));
  if (name == QString::null) return;
  string filename = name.latin1();
  if (fileSave("tlp",filename))
    openFiles[(unsigned int)glWidget]=filename;
}
//**********************************************************************
void viewGl::fileOpen() {
  QString s;
  fileOpen(0,s);
}
//**********************************************************************
void viewGl::fileOpen(string *plugin, QString &s) {
  Observable::holdObservers();
  DataSet dataSet;
  string tmpStr="tlp";
  bool cancel=false;
  if (s == QString::null) {
    if (plugin==0) {
      plugin = &tmpStr;
      s = QFileDialog::getOpenFileName( QString::null,
					tr("Tulip graph (*.tlp *.tlp.gz)"),
					this,
					tr("open file dialog"),
					tr("Choose a file to open" ));

      if (s == QString::null)
	cancel=true;
      else
	dataSet.set("filename", string(s.latin1()));
    }
    else {
      s = "import";
      StructDef parameter = tlp::importFactory.getParam(*plugin);
      parameter.buildDefaultDataSet( dataSet );
      cancel = !tlp::openDataSetDialog(dataSet, parameter, &dataSet, "Enter plugin parameter");
    }
  } else {
    plugin = &tmpStr;
    dataSet.set("filename", string(s.latin1()));
  }
  if (!cancel) {
    QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));
    SuperGraph *newGraph = tlp::newSuperGraph();
    bool result=true;
    MyProgress progressBar(this,string("Loading : ")+ s.section('/',-1).ascii() );
    result = tlp::importGraph(*plugin, dataSet, &progressBar ,newGraph);
    if (!progressBar.compute || !result ) {
      delete newGraph;
      QApplication::restoreOverrideCursor();
      qWarning("Canceled import/Open");
      Observable::unholdObservers();
      return;
    }
    QString cleanName=s.section('/',-1);
    cleanName=cleanName.section('.',0,0);
    newGraph->setAttribute("name",string(cleanName.latin1()));
    GlGraphWidget *glW=newOpenGlView(newGraph,s);
    DataSet glGraphData;
    if (dataSet.get<DataSet>("displaying", glGraphData))
      glW->getGlGraph()->setParameters(glGraphData);
    if (plugin==0)
      openFiles[((unsigned int)glW)] = s.latin1();
    QApplication::restoreOverrideCursor();
    restoreView();
  }
  else {
    qWarning("Canceled  Open/import");
  }
  Observable::unholdObservers();
}
//**********************************************************************
void viewGl::importGraph(int id) {
  string name(importGraphMenu.text(id).ascii());
  QString s;
  fileOpen(&name,s);
}

//==============================================================

namespace {

	typedef std::vector<node> NodeA;
	typedef std::vector<edge> EdgeA;

	void GetSelection(	NodeA & outNodeA,
						EdgeA & outEdgeA,
						SuperGraph *inG,
						SelectionProxy * inSel )
	{
		assert( inSel );
		assert( inG );
		outNodeA.clear();
		outEdgeA.clear();

		// Get edges
		Iterator<edge> * edgeIt = inG->getEdges();
		while( edgeIt->hasNext() ) {
			edge e = edgeIt->next();
			if( inSel->getEdgeValue(e) )
				outEdgeA.push_back( e );
		}
		delete edgeIt;
	
		// Get nodes
		Iterator<node> * nodeIt = inG->getNodes();
		while( nodeIt->hasNext() ) {
			node n = nodeIt->next();
			if( inSel->getNodeValue(n) )
				outNodeA.push_back( n );
		}
		delete nodeIt;
	}

	void SetSelection(	SelectionProxy * outSel,
						NodeA & inNodeA,
						EdgeA & inEdgeA,
						SuperGraph * inG	 )
	{
		assert( outSel );
		assert( inG );
		outSel->setAllNodeValue( false );
		outSel->setAllEdgeValue( false );

		// Set edges
		for( uint e = 0 ; e < inEdgeA.size() ; e++ )
			outSel->setEdgeValue( inEdgeA[e], true );
	
		// Set nodes
		for( uint n = 0 ; n < inNodeA.size() ; n++ )
			outSel->setNodeValue( inNodeA[n], true );
	}

}



void viewGl::editCut()
{
	if( !glWidget )
		return;
	SuperGraph * g = glWidget->getSuperGraph();
	if( !g )
		return;

	// free the previous ccpGraph
	if( copyCutPasteGraph ) {
		delete copyCutPasteGraph;
		copyCutPasteGraph = 0;
	}

	SelectionProxy * selP = g->getProperty<SelectionProxy>("viewSelection");
	if( !selP )
		return;

	// Save selection
	NodeA nodeA;
	EdgeA edgeA;
	GetSelection( nodeA, edgeA, g, selP );

	Observable::holdObservers();
	copyCutPasteGraph = tlp::newSuperGraph();
	tlp::copyToGraph( copyCutPasteGraph, g, selP );
	// Restore selection
	SetSelection( selP, nodeA, edgeA, g );
	tlp::removeFromGraph( g, selP );
	Observable::unholdObservers();
}

void viewGl::editCopy()
{
	if( !glWidget )
		return;
	SuperGraph * g = glWidget->getSuperGraph();
	if( !g )
		return;

	// free the previous ccpGraph
	if( copyCutPasteGraph ) {
		delete copyCutPasteGraph;
		copyCutPasteGraph = 0;
	}

	SelectionProxy * selP = g->getProperty<SelectionProxy>("viewSelection");
	if( !selP )
		return;

	Observable::holdObservers();
	copyCutPasteGraph = tlp::newSuperGraph();
	tlp::copyToGraph( copyCutPasteGraph, g, selP );
	Observable::unholdObservers();
}

void viewGl::editPaste()
{
	if( !glWidget )
		return;
	SuperGraph * g = glWidget->getSuperGraph();
	if( !g )
		return;


	if( !copyCutPasteGraph )
		return;

	Observable::holdObservers();
	SelectionProxy * selP = g->getProperty<SelectionProxy>("viewSelection");
	tlp::copyToGraph( g, copyCutPasteGraph, 0, selP );
	Observable::unholdObservers();
}


namespace
{
	
	bool IsEvaluableProxy(	PProxy *		p	)
	{
		return		dynamic_cast<MetricProxy*>(p)
				||	dynamic_cast<StringProxy*>(p)
				||	dynamic_cast<SelectionProxy*>(p);
	}

	struct GItem {
		GItem( const node & inN ) {
			isnode = true;
			n = inN;
		}
		GItem( const edge & inE ) {
			isnode = false;
			e = inE;
		}
		node n;
		edge e;
		bool isnode;
	};


	bool EvalProxy(		PProxy *		p,
						const GItem &	gi,
						std::string		value,
						int				mode	)
	{
		assert( IsEvaluableProxy(p) );
		MetricProxy *    metp = dynamic_cast<MetricProxy*>(p);
		StringProxy *    strp = dynamic_cast<StringProxy*>(p);
		SelectionProxy * selp = dynamic_cast<SelectionProxy*>(p);

		if( metp ) {
			// Metric Proxy
			DoubleType::RealType v = gi.isnode ? metp->getNodeValue( gi.n )
											   : metp->getEdgeValue( gi.e );
			DoubleType::RealType v0 = atof( value.c_str() );
			if( mode == 0 )		return v < v0;
			if( mode == 1 )		return v <= v0;
			if( mode == 2 )		return v == v0;
			if( mode == 3 )		return v >= v0;
			if( mode == 4 )		return v > v0;
			else				return v != v0;
		}

		if( strp ) {
			// String Proxy
			std::string v = gi.isnode ? strp->getNodeValue( gi.n )
									  : strp->getEdgeValue( gi.e );
			std::string v0 = value.c_str();
			QRegExp rexp( v0.c_str() );
			if( mode == 2 )		return rexp.exactMatch( v.c_str() );
			else				return ! rexp.exactMatch( v.c_str() );
		}

		if( selp ) {
			// Selection Proxy
			bool v  = gi.isnode ? selp->getNodeValue( gi.n )
								: selp->getEdgeValue( gi.e );
			bool v0 = !( value.size() == 0 || value == "False" || value == "false" || value == "0" );
			if( mode == 2 )		return v == v0;
			else				return v != v0;
		}
	}


	struct FindDialog : public FindSelection {

		SuperGraph * g;

	    virtual void propertyChanged( int no )
		{
			PProxy * p = getProperty();
			if( dynamic_cast<MetricProxy*>(p) ) {
				filterOp->clear();
				filterOp->insertItem( "<" );
				filterOp->insertItem( "<=" );
				filterOp->insertItem( "=" );
				filterOp->insertItem( ">=" );
				filterOp->insertItem( ">" );
				filterOp->insertItem( "!=" );
				filterValue->setText( QString() );
				filterValue->setValidator( new QDoubleValidator(this) );
				filterValue->show();
			}
			else if( dynamic_cast<StringProxy*>(p) ) {
				filterOp->clear();
				filterOp->insertItem( "=" );
				filterOp->insertItem( "!=" );
				filterValue->setValidator( 0 );
				filterValue->show();
			}
			else if( dynamic_cast<SelectionProxy*>(p) ) {
				filterOp->clear();
				filterOp->insertItem( "False" );
				filterOp->insertItem( "True" );
				filterValue->hide();
				filterValue->setValidator( 0 );
				filterValue->setText( QString() );
			}
		}

		int getMode() {
			if( filterOp->count() == 2 )
				// == -> 2 & != -> 5
				return filterOp->currentItem() == 0 ? 2 : 5;
			else
				return filterOp->currentItem();
		}

		std::string getFilterValue() {
			return filterValue->text();
		}

		PProxy * getProperty( ) {
			return g->getProperty( inputProp->currentText() );
		}

		int getOperation() {
			if( setToSelectionOpt->isOn() )		return 0;
			if( addToSelectionOpt->isOn() )		return 1;
			if( rmvFromSelectionOpt->isOn() )	return 2;
			return 3;
		}

		int getSource() {
			return srcOpt->currentItem();
		}

	};

}



void viewGl::editFind()
{
	if( !glWidget )
		return;
	SuperGraph * g = glWidget->getSuperGraph();
	if( !g )
		return;
	SelectionProxy * selP = g->getProperty<SelectionProxy>("viewSelection");
	if( !selP )
		return;

	FindDialog * sel = new FindDialog();
	sel->g = g;

	Iterator<std::string> * propIt = g->getProperties();
	while( propIt->hasNext() ) {
		std::string n = propIt->next();
		PProxy * p = g->getProperty( n );
		if(	IsEvaluableProxy(p) )
			sel->inputProp->insertItem( QString(n.c_str()) );
	}
	delete propIt;

	// Rejected ?
	sel->propertyChanged(-1);
	if( sel->exec() == QDialog::QDialog::Rejected ) {
		delete sel;
		return;
	}

	PProxy * p     = sel->getProperty();
	int      mode  = sel->getMode();
	std::string fv = sel->getFilterValue();
	int      op    = sel->getOperation();
	assert( g );

	// Hold obs
	Observable::holdObservers();

	// Eval nodes
	if( (sel->getSource()+1) & 1 )
	{
		Iterator<node> * nodeIt = g->getNodes();
		while( nodeIt->hasNext() ) {
			node n = nodeIt->next();
			bool v = EvalProxy( p, GItem(n), fv, mode );
			if( op == 0 ) {
				selP->setNodeValue( n, v );
			} else if( op == 1 ) {
				if( v )
					selP->setNodeValue( n, true );
			} else if( op == 2 )  {
				if( v )
					selP->setNodeValue( n, false );
			} else {
				if( !v )
					selP->setNodeValue( n, false );
			}
		}
		delete nodeIt;
	}

	// Eval edges
	if( (sel->getSource()+1) & 2 )
	{
		Iterator<edge> * edgeIt = g->getEdges();
		while( edgeIt->hasNext() ) {
			edge e = edgeIt->next();
			bool v = EvalProxy( p, GItem(e), fv, mode );
			if( op == 0 ) {
				selP->setEdgeValue( e, v );
			} else if( op == 1 ) {
				if( v )
					selP->setEdgeValue( e, true );
			} else if( op == 2 ) {
				if( v )
					selP->setEdgeValue( e, false );
			} else {
				if( !v )
					selP->setEdgeValue( e, false );
			}
		}
		delete edgeIt;
	}

	// Unhold obs
	Observable::unholdObservers();
	delete sel;
}

//**********************************************************************
void viewGl::setParameters(const DataSet data) {
  //  cerr << __PRETTY_FUNCTION__ << endl;
  glWidget->getGlGraph()->setParameters(data);
  clusterTreeWidget->setSuperGraph(glWidget->getSuperGraph());
  nodeProperties->setSuperGraph(glWidget->getSuperGraph());
  propertiesWidget->setSuperGraph(glWidget->getSuperGraph());
}
//**********************************************************************
void viewGl::updateSatutBar() {
  //  cerr << __PRETTY_FUNCTION__ << endl;
  SuperGraph *graph=clusterTreeWidget->getSuperGraph();
  if (graph==0) return;
  char tmp[255];
  sprintf(tmp,"Ready, Nodes:%d, Edges:%d",graph->numberOfNodes(),graph->numberOfEdges());
  statusBar()->message(tmp);
}
//**********************************************************************
template <typename TYPEN,typename TYPEE>
void buildPropertyMenu(QPopupMenu &menu) {
  typename TemplateFactory<PropertyFactory<Property<TYPEN,TYPEE> >,Property<TYPEN,TYPEE>,PropertyContext>::ObjectCreator::const_iterator it;
  it=PropertyProxy<TYPEN,TYPEE>::factory.objMap.begin();
  for (;it!=PropertyProxy<TYPEN,TYPEE>::factory.objMap.end();++it)  
    menu.insertItem( it->first.c_str() );
}
//**********************************************************************
void viewGl::buildMenus(){
   //Properties PopMenus
  buildPropertyMenu<IntType,IntType>(intMenu);
  buildPropertyMenu<StringType,StringType>(stringMenu);
  buildPropertyMenu<SizeType,SizeType>(sizesMenu);
  buildPropertyMenu<ColorType,ColorType>(colorsMenu);
  buildPropertyMenu<PointType,LineType>(layoutMenu);
  buildPropertyMenu<DoubleType,DoubleType>(metricMenu);
  buildPropertyMenu<BooleanType,BooleanType>(selectMenu);
  //Clustering PopMenu
  TemplateFactory<ClusteringFactory,Clustering,ClusterContext>::ObjectCreator::const_iterator it3;
  for (it3=tlp::clusteringFactory.objMap.begin();it3!=tlp::clusteringFactory.objMap.end();++it3)
    clusteringMenu->insertItem( it3->first.c_str() );
  //Export PopMenu
  TemplateFactory<ExportModuleFactory,ExportModule,ClusterContext>::ObjectCreator::const_iterator it9;
  for (it9=tlp::exportFactory.objMap.begin();it9!=tlp::exportFactory.objMap.end();++it9)
    exportGraphMenu.insertItem( it9->first.c_str() );
  //Import PopMenu
  TemplateFactory<ImportModuleFactory,ImportModule,ClusterContext>::ObjectCreator::const_iterator it4;
  for (it4=tlp::importFactory.objMap.begin();it4!=tlp::importFactory.objMap.end();++it4) {
    importGraphMenu.insertItem( it4->first.c_str() );
  }
  //Image PopuMenu
  QStrList listFormat=QImageIO::outputFormats();
  char *tmp=listFormat.first();
  while (tmp!=0) {
    exportImageMenu.insertItem(tmp);
    tmp=listFormat.next();
  }
  exportImageMenu.insertItem("EPS");
  //Windows
  dialogMenu.insertItem("Mouse Tool Bar");
  dialogMenu.insertItem("3D Overview");
  dialogMenu.insertItem("Info Editor");
  //==============================================================
  //File Menu 
  fileMenu->insertSeparator();
  fileMenu->insertItem("&Import", &importGraphMenu );
  fileMenu->insertItem("&Export", &exportGraphMenu );
  fileMenu->insertItem("&Save Picture as " , &exportImageMenu); //this , SLOT( outputImage() ));
  //View Menu
  viewMenu->insertItem( "Redraw View", this, SLOT( redrawView() ));
  viewMenu->insertItem( "Center View", this, SLOT( centerView() ));
  viewMenu->insertItem( "Restore View", this, SLOT( restoreView() ));
  viewMenu->insertItem( "Dialogs",  &dialogMenu);
  //EditMenu
  editMenu->insertSeparator();
  editMenu->insertItem( "Deselect All", this, SLOT( deselectALL() ));
  editMenu->insertItem( "Reverse Selection", this, SLOT( reverseSelection() ));
  editMenu->insertItem( "Delete Selection", this , SLOT(delSelection() ));
  editMenu->insertItem( "New subgraph", this, SLOT( newSubgraph() ));
  editMenu->insertItem( "Reverse selected edges direction", this, SLOT( reverseSelectedEdgeDirection() ));
  //Property Menu
  propertyMenu->insertItem("&Selection", &selectMenu );
  propertyMenu->insertItem("&Layout", &layoutMenu );
  propertyMenu->insertItem("&Metric", &metricMenu );
  propertyMenu->insertItem("&Colors", &colorsMenu );
  propertyMenu->insertItem("&Integer", &intMenu );
  propertyMenu->insertItem("&String", &stringMenu );
  propertyMenu->insertItem("&Sizes", &sizesMenu );
}
//**********************************************************************
void viewGl::outputEPS() {
  if (!glWidget) return;
  QString s( QFileDialog::getSaveFileName());
  if (!s.isNull()) {
    glWidget->getGlGraph()->outputEPS(64000000,true,s.ascii());
  }
}
//**********************************************************************
void viewGl::exportImage(int id) {
  if (!glWidget) return;
  string name(exportImageMenu.text(id).ascii());
  if (name=="EPS") {
    outputEPS();
    return;
  }
  QString s(QFileDialog::getSaveFileName());
  if (s.isNull()) return;    
  int width,height;
  unsigned char* image = glWidget->getGlGraph()->getImage(width,height);
  QPixmap pm(width,height);
  QPainter painter;
  painter.begin(&pm);
  for (int y=0; y<height; y++)
    for (int x=0; x<width; x++) {
      painter.setPen(QColor(image[(height-y-1)*width*3+(x)*3],
			    image[(height-y-1)*width*3+(x)*3+1],
			    image[(height-y-1)*width*3+(x)*3+2]));
      painter.drawPoint(x,y);
    }
  painter.end();
  delete image;
  pm.save( s, name.c_str());
}
//**********************************************************************
void viewGl::exportGraph(int id) {
  if (!glWidget) return;
  string filename( QFileDialog::getSaveFileName(this->caption().ascii()).ascii());
  if (filename == "") return;
  DataSet dataSet;
  string name(exportGraphMenu.text(id).ascii());
  fileSave(name,filename);
}
//**********************************************************************
void viewGl::windowsMenuActivated( int id ) {
  QWidget* w = workspace->windowList().at( id );
  if ( w ) {
    w->setFocus();
    w->show();
  }
}
//**********************************************************************
void viewGl::windowsMenuAboutToShow() {
  windowsMenu->clear();
  int cascadeId = windowsMenu->insertItem("&Cascade", workspace, SLOT(cascade() ) );
  int tileId = windowsMenu->insertItem("&Tile", workspace, SLOT(tile() ) );
  if ( workspace->windowList().isEmpty() ) {
    windowsMenu->setItemEnabled( cascadeId, FALSE );
    windowsMenu->setItemEnabled( tileId, FALSE );
  }
  windowsMenu->insertSeparator();
  QWidgetList windows = workspace->windowList();
  for ( int i = 0; i < int(windows.count()); ++i ) {
    int id = windowsMenu->insertItem(windows.at(i)->caption(),
				     this, SLOT( windowsMenuActivated( int ) ) );
    windowsMenu->setItemParameter( id, i );
    windowsMenu->setItemChecked( id, workspace->activeWindow() == windows.at(i) );
  }
}
//**********************************************************************
int viewGl::closeWin() {
  delete this;
  return true;
}
//**********************************************************************
void viewGl::goInside() {
  //  cerr << __PRETTY_FUNCTION__ << endl;
  node tmpNode;
  edge tmpEdge;
  tlp::ElementType type;
  if (glWidget->getGlGraph()->doSelect(mouseClicX, mouseClicY, type, tmpNode,tmpEdge)) {
    if (type==NODE) {
      SuperGraph *supergraph=glWidget->getSuperGraph();
      MetaGraphProxy *meta=supergraph->getProperty<MetaGraphProxy>("viewMetaGraph");
      if (meta->getNodeValue(tmpNode)!=0) {
	changeSuperGraph(meta->getNodeValue(tmpNode));
      }
    }
  }
}
//**********************************************************************
bool viewGl::eventFilter(QObject *obj, QEvent *e) {
  if ((typeid(*obj) == typeid(GlGraphWidget)) &&
      (e->type() == QEvent::MouseButtonRelease)) {
    QMouseEvent *me = (QMouseEvent *) e;
    GlGraphWidget *glw = (GlGraphWidget *) obj;
    if (me->button()==RightButton) {
      glw->setContextCoord(me->x(), me->y());
      mouseClicX = me->x();
      mouseClicY = me->y();
      QPopupMenu *contextMenu=new QPopupMenu(this,"dd");
      contextMenu->insertItem(tr("Go inside"), this, SLOT(goInside()));
      contextMenu->insertItem(tr("New 3D View"), this, SLOT(new3DView()));
      contextMenu->insertItem(tr("Delete"), glw, SLOT(contextDel()));
      contextMenu->insertItem(tr("Select"), glw, SLOT(contextSelect()));
      contextMenu->insertItem(tr("Add/Remove selection"), glw, SLOT(contextAddRemoveSelection()));
      contextMenu->exec(me->globalPos());
      delete contextMenu;
      return true;
    }
    else {
      return false;
    }
  }
  return false;
}
//**********************************************************************
void viewGl::focusInEvent ( QFocusEvent * ) {
}

//Fonction de param�trage de la fen�tre Couleurs ect ...
//Change the base node color of the node in the graph view
void viewGl::showDialog(int id){
  string name(dialogMenu.text(id).ascii());
  if (name=="Mouse Tool Bar") {
    mouseToolBarDock->show();
    clusterTreeWidget->raise();
  }
  if (name=="Info Editor") {
    tabWidgetDock->show();
    tabWidgetDock->raise();
  }
  if (name=="3D Overview") {
    overviewDock->show();
    overviewDock->raise();
  }
}
//======================================================================
//Fonction du Menu de vue
//======================================================================
///Redraw the view of the graph
void  viewGl::redrawView() {
  if (!glWidget) return;
  glWidget->UpdateGL();
}
//**********************************************************************
///Reccenter the layout of the graph
void viewGl::centerView() {
  if (!glWidget) return;
  SuperGraph *graph=glWidget->getSuperGraph();
  if (graph==0) return;
  Observable::holdObservers();
  glWidget->getGlGraph()->centerScene();
  redrawView();
  Observable::unholdObservers();
}
//**********************************************************************
///Restore the view of the graph
void viewGl::restoreView() {
  if (!glWidget) return;
  SuperGraph *graph=glWidget->getSuperGraph();
  if (graph==0) return;
  Observable::holdObservers();
  glWidget->getGlGraph()->centerScene();
  redrawView();
  overviewWidget->setObservedView(glWidget->getGlGraph());
  updateSatutBar();
  Observable::unholdObservers();
}
//===========================================================
//Menu Edit : functions
//===========================================================
///Deselect all entries in the glGraph current selection Proxy
void viewGl::deselectALL() {
  if (!glWidget) return;
  SuperGraph *graph=glWidget->getSuperGraph();
  if (graph==0) return;
  Observable::holdObservers();
  graph->getProperty<SelectionProxy>("viewSelection")->setAllNodeValue(false);
  graph->getProperty<SelectionProxy>("viewSelection")->setAllEdgeValue(false);
  Observable::unholdObservers();
}
void viewGl::delSelection() {
  SuperGraph *graph=glWidget->getSuperGraph();
  if (graph==0) return;
  Observable::holdObservers();
  SelectionProxy *elementSelected=graph->getProperty<SelectionProxy>("viewSelection");
  StableIterator<node> itN(graph->getNodes());
  while(itN.hasNext()) {
    node itv = itN.next();
    if (elementSelected->getNodeValue(itv)==true)
      graph->delNode(itv);
  }
  StableIterator<edge> itE(graph->getEdges());
  while(itE.hasNext()) {
    edge ite=itE.next();
    if (elementSelected->getEdgeValue(ite)==true)   
      graph->delEdge(ite);
  }
  Observable::unholdObservers();
}
//==============================================================
///Reverse all entries in the glGraph current selection Proxy
void viewGl::reverseSelection() {
  SuperGraph *graph=glWidget->getSuperGraph();
  if (graph==0) return;
  Observable::holdObservers();
  graph->getProperty<SelectionProxy>("viewSelection")->reverse();
  Observable::unholdObservers();
}
//==============================================================
void viewGl::newSubgraph() {
  if (!glWidget) return;
  SuperGraph *graph=glWidget->getSuperGraph();
  if (graph==0) return;

  bool ok = FALSE;
  string tmp;
  bool verifGraph = true;
  SelectionProxy *sel1 = graph->getProperty<SelectionProxy>("viewSelection");  
  Observable::holdObservers();
  Iterator<edge>*itE = graph->getEdges();
  while (itE->hasNext()) {
    edge ite= itE->next();
    if (sel1->getEdgeValue(ite)) {
      if (!sel1->getNodeValue(graph->source(ite))) {sel1->setNodeValue(graph->source(ite),true); verifGraph=false;}
      if (!sel1->getNodeValue(graph->target(ite))) {sel1->setNodeValue(graph->target(ite),true); verifGraph=false;}
    }
  } delete itE;
  Observable::unholdObservers();
  
  if(!verifGraph) 
    QMessageBox::critical( 0, "Tulip Warning" ,"The selection wasn't a graph, missing nodes have been added");
  QString text = QInputDialog::getText( "View building" ,  "Please enter view name" , QLineEdit::Normal,QString::null, &ok, this );
  if (ok && !text.isEmpty()) {
    sel1 = graph->getProperty<SelectionProxy>("viewSelection");
    SuperGraph *tmp = graph->addSubGraph(sel1);
    tmp->setAttribute("name",string(text.latin1()));
    clusterTreeWidget->update();
  }
  else if (ok) {
    sel1 = graph->getProperty<SelectionProxy>("viewSelection");
    SuperGraph *tmp=graph->addSubGraph(sel1);
    tmp->setAttribute("name",string("unnamed"));
    clusterTreeWidget->update();
  }
}
//==============================================================
void viewGl::reverseSelectedEdgeDirection() {
  if (!glWidget) return;
  SuperGraph *graph=glWidget->getSuperGraph();
  if (graph==0) return;
  Observable::holdObservers();
  graph->getProperty<SelectionProxy>("viewSelection")->reverseEdgeDirection();  
  Observable::unholdObservers();
}
//==============================================================
void viewGl::superGraphAboutToBeRemoved(SuperGraph *sg) {
  cerr << __PRETTY_FUNCTION__ <<  "Possible bug" << endl;
  glWidget->setSuperGraph(0);
}
//==============================================================
void viewGl::helpAbout() {
  if (aboutWidget==0)
    aboutWidget = new InfoDialog(this);
  aboutWidget->show();
}
//==============================================================
void viewGl::fileExit(){
  closeWin();
}
//==============================================================
void viewGl::filePrint() {
  if (!glWidget) return;
  SuperGraph *graph=glWidget->getSuperGraph();
  if (graph==0) return;
  
  QPrinter printer;
  if (!printer.setup(this)) 
    return;
  int width,height;
  unsigned char* image = glWidget->getGlGraph()->getImage(width,height);
  QPainter painter(&printer);
  for (int y=0; y<height; y++)
    for (int x=0; x<width; x++) {
      painter.setPen(QColor(image[(height-y-1)*width*3+(x)*3],
			    image[(height-y-1)*width*3+(x)*3+1],
			    image[(height-y-1)*width*3+(x)*3+2]));
      painter.drawPoint(x,y);
    }
  painter.end();
  delete image;
}



//==============================================================
//
// Morphing process
//

namespace
{

	bool HaveSameValues( SuperGraph * inG, PProxy * inP0, PProxy * inP1 )
	{
		assert( inP0 );
		assert( inP1 );
		assert( inG );

		// Nodes interpolation
		Iterator<node> * nodeIt = inG->getNodes();
		while( nodeIt->hasNext() ) {
			node n = nodeIt->next();

			if( inP0->getNodeStringValue(n) != inP1->getNodeStringValue(n) ) {
				delete nodeIt;
				return false;
			}
		}
		delete nodeIt;

		// Edges interpolation
		Iterator<edge> * edgeIt = inG->getEdges();
		while( edgeIt->hasNext() ) {
			edge e = edgeIt->next();

			if( inP0->getEdgeStringValue(e) != inP1->getEdgeStringValue(e) ) {
				delete edgeIt;
				return false;
			}
		}
		delete edgeIt;

		return true;
	}


	struct GraphState
	{
		SuperGraph   * g;
		LayoutProxy  * layout;
		SizesProxy   * size;
		ColorsProxy  * color;
		Camera		   camera;
		Coord		   sceneT;
		Coord		   sceneR;

		GraphState( GlGraphWidget * glgw )
		{
			assert( glgw );
			g = glgw->getSuperGraph();
			layout = new LayoutProxy( g );
			*layout = *( g->getProperty<LayoutProxy>("viewLayout") );
			size = new SizesProxy( g );
			*size = *( g->getProperty<SizesProxy>("viewSize") );
			color = new ColorsProxy( g );
			*color = *( g->getProperty<ColorsProxy>("viewColor") );
			
			GlGraph * glg = glgw->getGlGraph();
			assert( glg );
			camera = glg->getCamera();
			sceneT = glg->getSceneTranslation();
			sceneR = glg->getSceneRotation();
		}

		~ GraphState()
		{
			if( layout )	delete layout;
			if( size )		delete size;
			if( color )		delete color;
		}

		static bool setupDiff( SuperGraph * inG, GraphState * inGS0, GraphState * inGS1 )
		{
			int remain = 3;
			if( HaveSameValues(inG,inGS0->layout,inGS1->layout) ) {
				delete inGS0->layout;
				delete inGS1->layout;
				inGS0->layout = inGS1->layout = 0;
				remain--;
			}
			if( HaveSameValues(inG,inGS0->size,inGS1->size) ) {
				delete inGS0->size;
				delete inGS1->size;
				inGS0->size = inGS1->size = 0;
				remain--;
			}
			if( HaveSameValues(inG,inGS0->color,inGS1->color) ) {
				delete inGS0->color;
				delete inGS1->color;
				inGS0->color = inGS1->color = 0;
				remain--;
			}
			return ( remain > 0 );
		}
	};


	void InterpolateColors( Color & outc, const Color & c0, const Color & c1, float inT ) {
		for( int i = 0 ; i < 4 ; i++ ) {
			float f0 = float( c0.array[i] ),
				  f1 = float( c1.array[i] );
			outc.array[i] = uint( f0 + (f1-f0) * inT );
		}
	}

	void EdgeEnds( Coord & outC0, Coord & outC1, GraphState * gs, edge e ) {
		node n0 = gs->g->source(e);
		node n1 = gs->g->target(e);
		outC0   = gs->layout->getNodeValue( n0 );
		outC1   = gs->layout->getNodeValue( n1 );
	}

	bool AssociateEdges( GraphState  * g0,
						 LayoutProxy * e0,
						 GraphState  * g1,
						 LayoutProxy * e1,
						 edge e )
	{
		if( e0->getEdgeStringValue(e) == e1->getEdgeStringValue(e) )
			return false;

		std::vector<Coord> ec0, ec1;
		ec0 = e0->getEdgeValue( e );
		ec1 = e1->getEdgeValue( e );

		if( ec0.size() == ec1.size() )
			return true;
		else if( ec0.size() > ec1.size() ) {
			std::swap( g0, g1 );
			std::swap( e0, e1 );
			std::swap( ec0, ec1 );
		}

		Coord c0, c1;
		EdgeEnds( c0, c1, g0, e );
		int n  = ec1.size() - ec0.size();
		int n0 = n >> 1;
		int n1 = n - n0;
		ec0.insert( ec0.begin(), n0, c0 );
		ec0.insert( ec0.end(), n1, c1 );
		e0->setEdgeValue( e, ec0 );
		return true;
	}

	struct Morphing {
		GraphState   * g0, * g1;
		LayoutProxy  * e0, * e1;
		int			 tid;
		float		 t;
		QTime		 qt0;
		int			 frameCpt;
		Morphing() {
			g0 = g1 = 0;
			tid = -1;
			t = 0.0f;
		}
		bool start( GlGraphWidget * outGlgw, GraphState * inG0, GraphState * inG1, int inTID ) {
			assert( outGlgw );
			assert( inG0 );
			assert( inG1 );
			SuperGraph * g = outGlgw->getSuperGraph();
			assert( g );
			stop();

			bool hasdiff = GraphState::setupDiff( g, inG0, inG1 );
			if( !hasdiff )
				return false;

			g0  = inG0;
			g1  = inG1;
			e0  = 0;
			e1  = 0;
			tid = inTID;
			qt0.start();
		    frameCpt = 0;

			// Edges association
			if( g0->layout && g1->layout ) {
				e0 = new LayoutProxy( g0->g );
				e1 = new LayoutProxy( g1->g );
				*e0 = *(g0->layout);
				*e1 = *(g1->layout);
				e0->setAllNodeValue( Coord(0,0,0) );
				e1->setAllNodeValue( Coord(0,0,0) );

				bool haveSameValues = true;
				Iterator<edge> * edgeIt = g->getEdges();
				while( edgeIt->hasNext() ) {
					edge e = edgeIt->next();
					bool isDiff = AssociateEdges( g0, e0, g1, e1, e );
					if( isDiff )
						haveSameValues = false;
				}
				delete edgeIt;

				if( haveSameValues ) {
					delete e0;
					delete e1;
					e0 = e1 = 0;
				}
			}

		    interpolate( outGlgw, 0.0f );
		    return true;
		}
		void stop( ) {
			if( g0 )	delete g0;
			if( g1 )	delete g1;
			if( e0 )	delete e0;
			if( e1 )	delete e1;
			g0 = g1 = 0;
			e0 = e1 = 0;
			tid = -1;
		}
		float fps() {
			float dt = float(qt0.elapsed()) / 1000.0f;
			return float(frameCpt) / dt;
		}
		void interpolate( GlGraphWidget * outGlgw, float inT )
		{
			frameCpt++;

			assert( outGlgw );
			SuperGraph * g = outGlgw->getSuperGraph();
			assert( g );
			GlGraph * glg = outGlgw->getGlGraph();
			assert( glg );

			inT = inT >? 0.0f;
			inT = inT <? 1.0f;
			t = inT;

			assert( outG );
			LayoutProxy * outLayout = g->getProperty<LayoutProxy>( "viewLayout" );
			SizesProxy  * outSize   = g->getProperty<SizesProxy>( "viewSize" );
			ColorsProxy * outColor  = g->getProperty<ColorsProxy>( "viewColor" );

			// Nodes interpolation
			Iterator<node> * nodeIt = g->getNodes();
			while( nodeIt->hasNext() ) {
				node n = nodeIt->next();

				// Layout
				if( g0->layout && g1->layout ) {
					Coord c0, c1;
					c0 = g0->layout->getNodeValue( n );
					c1 = g1->layout->getNodeValue( n );
					c0 += (c1-c0) * inT;
					outLayout->setNodeValue( n, c0 );
				}

				// Size
				if( g0->size && g1->size ) {
					Size s0, s1;
					s0 = g0->size->getNodeValue( n );
					s1 = g1->size->getNodeValue( n );
					s0 += (s1-s0) * inT;
					outSize->setNodeValue( n, s0 );
				}

				// Color
				if( g0->color && g1->color ) {
					Color c0, c1;
					c0 = g0->color->getNodeValue( n );
					c1 = g1->color->getNodeValue( n );
					InterpolateColors( c0, c0, c1, inT );
					outColor->setNodeValue( n, c0 );
				}
			}
			delete nodeIt;

			// Edges interpolation
			std::vector<Coord> edgeControls, edgeControls0, edgeControls1;
			Iterator<edge> * edgeIt = g->getEdges();
			while( edgeIt->hasNext() ) {
				edge e = edgeIt->next();

				// Layout
				if( g0->layout && g1->layout ) {
					if( inT <= 0.0f ) {
						edgeControls = g0->layout->getEdgeValue( e );
						outLayout->setEdgeValue( e, edgeControls );
					} else if( inT >= 1.0f ) {
						edgeControls = g1->layout->getEdgeValue( e );
						outLayout->setEdgeValue( e, edgeControls );
					} else if( e0 && e1 ) {
						edgeControls0 = e0->getEdgeValue( e );
						edgeControls1 = e1->getEdgeValue( e );
						int n = edgeControls0.size();
						edgeControls.resize( n );
						for( int ec = 0 ; ec < n ; ec++ )
							edgeControls[ec] = edgeControls0[ec] + (edgeControls1[ec]-edgeControls0[ec]) * inT;
						outLayout->setEdgeValue( e, edgeControls );
					}
				}

				// Size
				if( g0->size && g1->size ) {
					Size s0, s1;
					s0 = g0->size->getEdgeValue( e );
					s1 = g1->size->getEdgeValue( e );
					s0 += (s1-s0) * inT;
					outSize->setEdgeValue( e, s0 );
				}

				// Color
				if( g0->color && g1->color ) {
					Color c0, c1;
					c0 = g0->color->getEdgeValue( e );
					c1 = g1->color->getEdgeValue( e );
					InterpolateColors( c0, c0, c1, inT );
					outColor->setEdgeValue( e, c0 );
				}
			}
			delete edgeIt;

			// Camera
			Coord cam_center = g0->camera.center 		+ (g1->camera.center - g0->camera.center) * inT;
			Coord cam_eyes   = g0->camera.eyes   		+ (g1->camera.eyes - g0->camera.eyes) * inT;
			Coord cam_up     = g0->camera.up     		+ (g1->camera.up - g0->camera.up) * inT;
			float zoomf      = g0->camera.zoomFactor    + (g1->camera.zoomFactor - g0->camera.zoomFactor) * inT;
			float radius     = g0->camera.sceneRadius   + (g1->camera.sceneRadius - g0->camera.sceneRadius) * inT;
			Camera c;
			c = g0->camera;
			glg->setCamera( Camera(cam_center,cam_eyes,cam_up,zoomf,radius) );

			// Scene
			Coord scn_t      = g0->sceneT + (g1->sceneT - g0->sceneT) * inT;
			Coord scn_r      = g0->sceneR + (g1->sceneR - g0->sceneR) * inT;
			glg->setSceneTranslation( scn_t );
			glg->setSceneRotation( scn_r );
		}

	};

	// Global morphing data
	Morphing morph;

}


void viewGl::timerEvent( QTimerEvent * te )
{
	if( te->timerId() == morph.tid )
	{
		 float fps        = morph.fps();
		 float min_frames = MORPHING_MIN_DURATION * fps;
		 float max_frames = MORPHING_MAX_DURATION * fps;
		 float it;
		 if( max_frames < MORPHING_MIN_FRAMES )
		 	it = 1.0f / MORPHING_MIN_FRAMES;
		 else if( min_frames < MORPHING_MIN_FRAMES )
		 	it = 1.0f / max_frames;
		 else
		 	it = 1.0f / min_frames;

		 float t  = morph.t + it;

	//	  cout << "FPS = " << fps << ", it=" << it << endl;

		 Observable::holdObservers();
		 if( glWidget )
			 morph.interpolate( glWidget, t );
		 Observable::unholdObservers();
		 redrawView();

		 if( t >= 1.0f ) {
			killTimer( te->timerId() );
			morph.stop();
		}
	}
}



//==============================================================

//**********************************************************************
///Make a new clustering of the view graph
void viewGl::makeClustering(int id) {
  clearObservers();
  if (glWidget==0) return;
  Observable::holdObservers();
  string name(clusteringMenu->text(id).ascii());
  string erreurMsg;
  DataSet dataSet;
  SuperGraph *graph=glWidget->getSuperGraph();
  StructDef parameter = tlp::clusteringFactory.getParam(name);
  parameter.buildDefaultDataSet( dataSet, graph );
  bool result = tlp::openDataSetDialog(dataSet, parameter, &dataSet, "Tulip Parameter Editor", graph );
  MyProgress myProgress(this,name);
  myProgress.hide();

  if (!tlp::clusterizeGraph(graph, erreurMsg, &dataSet, name, &myProgress  )) {
    QMessageBox::critical( 0, "Tulip Algorithm Check Failed",QString((name + "::" + erreurMsg).c_str()));
  }
  clusterTreeWidget->update();
  clusterTreeWidget->setSuperGraph(graph);
  Observable::unholdObservers();
  initObservers();
}
//======================================================================
//**********************************************************************
//Management of properties
//**********************************************************************
template<typename PROPERTY>
bool viewGl::changeProperty(string name, string destination, bool query) {
  if( !glWidget )
    return false;

  Observable::holdObservers();
  MyProgress myProgress(this,name);
  myProgress.hide();
  SuperGraph *graph       = clusterTreeWidget->getSuperGraph();

  string erreurMsg;
  bool   resultBool,cached;  
  DataSet *dataSet =0;
  if (query) {
    dataSet = new DataSet();
    StructDef parameter = PROPERTY::factory.getParam(name);
    parameter.buildDefaultDataSet( *dataSet, graph );
    resultBool = tlp::openDataSetDialog(*dataSet, parameter, dataSet, "Tulip Parameter Editor", graph );
  }
  if (resultBool) {
    PROPERTY* dest = graph->template getLocalProperty<PROPERTY>(destination);
    PROPERTY* tmp = new PROPERTY(graph);
    resultBool = graph->computeProperty(name,tmp,erreurMsg,  &myProgress, dataSet);
    if (!resultBool) {
      QMessageBox::critical( 0, "Tulip Algorithm Check Failed", QString((name + "::" + erreurMsg).c_str()) );
    }
    else {
      if (myProgress.compute==true) {
	*dest=*tmp;
      }
      else 
	resultBool = false;
    }
    delete tmp;
    Observable::unholdObservers();
  }
  if (dataSet!=0) delete dataSet;

  propertiesWidget->setSuperGraph(graph);
  return resultBool;
}
//**********************************************************************
void viewGl::changeString(int id) {
  clearObservers();
  string name(stringMenu.text(id).ascii());
  bool result = changeProperty<StringProxy>(name,"viewLabel");
  initObservers();
}
//**********************************************************************
void viewGl::changeSelection(int id) {
  clearObservers();
  string name(selectMenu.text(id).ascii());
  bool result = changeProperty<SelectionProxy>(name,"viewSelection");
  initObservers();
}
//**********************************************************************
void viewGl::changeMetric(int id) {
  clearObservers();
  string name(metricMenu.text(id).ascii());
  bool result = changeProperty<MetricProxy>(name,"viewMetric");
  if (result && map_metric->isOn()) {
    changeProperty<ColorsProxy>("Metric Mapping","viewColor", false);
  }
  initObservers();
}
//**********************************************************************
void viewGl::changeLayout(int id) {
  clearObservers();

  GraphState * g0 = 0;
  if( enable_morphing->isOn() )
    g0 = new GraphState( glWidget );

  string name(layoutMenu.text(id).ascii());
  bool result = changeProperty<LayoutProxy>(name,"viewLayout");
  if (result) {
    if( force_ratio->isOn() )
      glWidget->getSuperGraph()->getLocalProperty<LayoutProxy>("viewLayout")->perfectAspectRatio();

	SuperGraph *graph=glWidget->getSuperGraph();
	Observable::holdObservers();
	glWidget->getGlGraph()->centerScene();
	overviewWidget->setObservedView(glWidget->getGlGraph());
	updateSatutBar();
	Observable::unholdObservers();

    if( enable_morphing->isOn() ) {
	    GraphState * g1 = new GraphState( glWidget );
	    bool morphable = morph.start( glWidget, g0, g1, startTimer(1000/MORPHING_MAX_FPS) );
	    if( !morphable ) {
	      delete g1;
	      g1 = 0;
	    } else {
	      g0 = 0;	// state remains in morph data ...
	    }
    }

	redrawView();
  }

  if( g0 )
	delete g0;

  initObservers();
}
//**********************************************************************
void viewGl::changeInt(int id) {
  clearObservers();
  string name(intMenu.text(id).ascii());
  bool result = changeProperty<IntProxy>(name,"viewInt");
  initObservers();
}
//**********************************************************************
void viewGl::changeColors(int id) {
  clearObservers();

  GraphState * g0 = 0;
  if( enable_morphing->isOn() )
    g0 = new GraphState( glWidget );

  string name(colorsMenu.text(id).ascii());
  bool result = changeProperty<ColorsProxy>(name,"viewColor");
  if( result ) {
    if( enable_morphing->isOn() ) {
	    GraphState * g1 = new GraphState( glWidget );
	    bool morphable = morph.start( glWidget, g0, g1, startTimer(1000/MORPHING_MAX_FPS) );
	    if( !morphable ) {
	      delete g1;
	      g1 = 0;
	    } else {
	      g0 = 0;	// state remains in morph data ...
	    }
    }
	redrawView();
  }

  if( g0 )
	delete g0;

  initObservers();
}
//**********************************************************************
void viewGl::changeSizes(int id) {
  clearObservers();

  GraphState * g0 = 0;
  if( enable_morphing->isOn() )
    g0 = new GraphState( glWidget );

  string name(sizesMenu.text(id).ascii());
  bool result = changeProperty<SizesProxy>(name,"viewSize");
  if( result ) {
    if( enable_morphing->isOn() ) {
	    GraphState * g1 = new GraphState( glWidget );
	    bool morphable = morph.start( glWidget, g0, g1, startTimer(1000/MORPHING_MAX_FPS) );
	    if( !morphable ) {
	      delete g1;
	      g1 = 0;
	    } else {
	      g0 = 0;	// state remains in morph data ...
	    }
    }
	redrawView();
  }

  if( g0 )
	delete g0;

  initObservers();
}

