
/*
 * Licensed Materials - Property of IBM
 *
 * trousers - An open source TCG Software Stack
 *
 * (C) Copyright International Business Machines Corp. 2004-2006
 *
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <assert.h>

#include "trousers/tss.h"
#include "trousers/trousers.h"
#include "spi_internal_types.h"
#include "spi_utils.h"
#include "capabilities.h"
#include "tsplog.h"
#include "hosttable.h"
#include "tcsd_wrap.h"
#include "obj.h"
#include "tcsd.h"

TSS_RESULT send_init(struct host_table_entry *);
TSS_RESULT sendit(struct host_table_entry *);

void
initData(struct tcsd_comm_data *comm, int parm_count)
{
	/* min packet size should be the size of the header */
	memset(&comm->hdr, 0, sizeof(struct tcsd_packet_hdr));
	comm->hdr.packet_size = sizeof(struct tcsd_packet_hdr);
	if (parm_count > 0) {
		comm->hdr.type_offset = sizeof(struct tcsd_packet_hdr);
		comm->hdr.parm_offset = comm->hdr.type_offset +
					(sizeof(TCSD_PACKET_TYPE) * parm_count);
		comm->hdr.packet_size = comm->hdr.parm_offset;
	}

	memset(comm->buf, 0, comm->buf_size);
}

int
loadData(UINT64 *offset, TCSD_PACKET_TYPE data_type, void *data, int data_size, BYTE *blob)
{
	switch (data_type) {
	case TCSD_PACKET_TYPE_BYTE:
		Trspi_LoadBlob_BYTE(offset, *((BYTE *) (data)), blob);
		break;
	case TCSD_PACKET_TYPE_BOOL:
		Trspi_LoadBlob_BOOL(offset, *((TSS_BOOL *) (data)), blob);
		break;
	case TCSD_PACKET_TYPE_UINT16:
		Trspi_LoadBlob_UINT16(offset, *((UINT16 *) (data)), blob);
		break;
	case TCSD_PACKET_TYPE_UINT32:
		Trspi_LoadBlob_UINT32(offset, *((UINT32 *) (data)), blob);
		break;
	case TCSD_PACKET_TYPE_PBYTE:
		Trspi_LoadBlob(offset, data_size, blob, (BYTE *)data);
		break;
	case TCSD_PACKET_TYPE_NONCE:
		Trspi_LoadBlob(offset, 20, blob, ((TCPA_NONCE *)data)->nonce);
		break;
	case TCSD_PACKET_TYPE_DIGEST:
		Trspi_LoadBlob(offset, 20, blob, ((TCPA_DIGEST *)data)->digest);
		break;
	case TCSD_PACKET_TYPE_AUTH:
		LoadBlob_AUTH(offset, blob, ((TPM_AUTH *)data));
		break;
	case TCSD_PACKET_TYPE_UUID:
		Trspi_LoadBlob_UUID(offset, blob, *((TSS_UUID *)data));
		break;
	case TCSD_PACKET_TYPE_ENCAUTH:
		Trspi_LoadBlob(offset, 20, blob, ((TCPA_ENCAUTH *)data)->authdata);
		break;
	case TCSD_PACKET_TYPE_VERSION:
		Trspi_LoadBlob_TCPA_VERSION(offset, blob, *((TCPA_VERSION *)data));
		break;
	case TCSD_PACKET_TYPE_LOADKEY_INFO:
		LoadBlob_LOADKEY_INFO(offset, blob, ((TCS_LOADKEY_INFO *)data));
		break;
	case TCSD_PACKET_TYPE_PCR_EVENT:
		Trspi_LoadBlob_PCR_EVENT(offset, blob, ((TSS_PCR_EVENT *)data));
		break;
	default:
		LogError("TCSD packet type unknown! (0x%x)", data_type & 0xff);
		return TCSERR(TSS_E_INTERNAL_ERROR);
	}

	return TSS_SUCCESS;
}

int
setData(TCSD_PACKET_TYPE dataType, int index, void *theData, int theDataSize, struct tcsd_comm_data *comm)
{
	UINT64 old_offset, offset;
	TSS_RESULT result;
	TCSD_PACKET_TYPE *type;

	/* Calculate the size of the area needed (use NULL for blob address) */
	offset = 0;
	if ((result = loadData(&offset, dataType, theData, theDataSize, NULL)) != TSS_SUCCESS)
		return result;
	if (((int)comm->hdr.packet_size + (int)offset) < 0) {
		LogError("Too much data to be transmitted!");
		return TCSERR(TSS_E_INTERNAL_ERROR);
	}
	if (((int)comm->hdr.packet_size + (int)offset) > comm->buf_size) {
		/* reallocate the buffer */
		BYTE *buffer;
		int buffer_size = comm->hdr.packet_size + offset;

		LogDebug("Increasing communication buffer to %d bytes.", buffer_size);
		buffer = realloc(comm->buf, buffer_size);
		if (buffer == NULL) {
			LogError("realloc of %d bytes failed.", buffer_size);
			return TCSERR(TSS_E_INTERNAL_ERROR);
		}
		comm->buf_size = buffer_size;
		comm->buf = buffer;
	}

	offset = old_offset = comm->hdr.parm_offset + comm->hdr.parm_size;
	if ((result = loadData(&offset, dataType, theData, theDataSize, comm->buf)) != TSS_SUCCESS)
		return result;
	type = (TCSD_PACKET_TYPE *)(comm->buf + comm->hdr.type_offset) + index;
	*type = dataType;
	comm->hdr.type_size += sizeof(TCSD_PACKET_TYPE);
	comm->hdr.parm_size += (offset - old_offset);

	comm->hdr.packet_size = offset;
	comm->hdr.num_parms++;

	return TSS_SUCCESS;
}

UINT32
getData(TCSD_PACKET_TYPE dataType, int index, void *theData, int theDataSize, struct tcsd_comm_data *comm)
{
	UINT64 old_offset, offset;
	TCSD_PACKET_TYPE *type = (TCSD_PACKET_TYPE *)(comm->buf + comm->hdr.type_offset) +
				 index;

	if ((UINT32)index >= comm->hdr.num_parms ||
	    dataType != *type) {
		LogDebug("Data type of TCS packet element %d doesn't match.", index);
		return TSS_TCP_RPC_BAD_PACKET_TYPE;
	}
	old_offset = offset = comm->hdr.parm_offset;
	switch (dataType) {
	case TCSD_PACKET_TYPE_BYTE:
		Trspi_UnloadBlob_BYTE(&offset, (BYTE *)theData, comm->buf);
		break;
	case TCSD_PACKET_TYPE_BOOL:
		Trspi_UnloadBlob_BOOL(&offset, (TSS_BOOL *)theData, comm->buf);
		break;
	case TCSD_PACKET_TYPE_UINT16:
		Trspi_UnloadBlob_UINT16(&offset, (UINT16 *)theData, comm->buf);
		break;
	case TCSD_PACKET_TYPE_UINT32:
		Trspi_UnloadBlob_UINT32(&offset, (UINT32 *)theData, comm->buf);
		break;
	case TCSD_PACKET_TYPE_PBYTE:
		Trspi_UnloadBlob(&offset, theDataSize, comm->buf, (BYTE *)theData);
		break;
	case TCSD_PACKET_TYPE_NONCE:
		Trspi_UnloadBlob(&offset, sizeof(TCPA_NONCE), comm->buf, ((TCPA_NONCE *)theData)->nonce);
		break;
	case TCSD_PACKET_TYPE_DIGEST:
		Trspi_UnloadBlob(&offset, sizeof(TCPA_DIGEST), comm->buf, ((TCPA_DIGEST *)theData)->digest);
		break;
	case TCSD_PACKET_TYPE_AUTH:
		UnloadBlob_AUTH(&offset, comm->buf, ((TPM_AUTH *)theData));
		break;
	case TCSD_PACKET_TYPE_UUID:
		Trspi_UnloadBlob_UUID(&offset, comm->buf, ((TSS_UUID *)theData));
		break;
	case TCSD_PACKET_TYPE_ENCAUTH:
		Trspi_UnloadBlob(&offset, sizeof(TPM_AUTH), comm->buf, ((TCPA_ENCAUTH *)theData)->authdata);
		break;
	case TCSD_PACKET_TYPE_VERSION:
		Trspi_UnloadBlob_TCPA_VERSION(&offset, comm->buf, ((TCPA_VERSION *) theData));
		break;
	case TCSD_PACKET_TYPE_KM_KEYINFO:
		Trspi_UnloadBlob_KM_KEYINFO(&offset, comm->buf, ((TSS_KM_KEYINFO *)theData ) );
		break;
	case TCSD_PACKET_TYPE_LOADKEY_INFO:
		UnloadBlob_LOADKEY_INFO(&offset, comm->buf, ((TCS_LOADKEY_INFO *)theData));
		break;
	case TCSD_PACKET_TYPE_PCR_EVENT:
		Trspi_UnloadBlob_PCR_EVENT(&offset, comm->buf, ((TSS_PCR_EVENT *)theData));
		break;
	default:
		LogError("unknown data type (%d) in TCSD packet!", dataType);
		return -1;
	}
	comm->hdr.parm_offset = offset;
	comm->hdr.parm_size -= (offset - old_offset);

	return TSS_SUCCESS;
}

#if 0
void
printBuffer(BYTE * b, int size)
{
	int i;
	return;
	for (i = 0; i < size; i++) {
		if ((i % 16) == 0)
			printf("\n");
		printf("%.2X ", b[i]);
	}
	printf("\n");
	return;
}
#endif

TSS_RESULT
sendTCSDPacket(struct host_table_entry *hte)
{
	TSS_RESULT rc;
	UINT64 offset = 0;

	Trspi_LoadBlob_UINT32(&offset, hte->comm.hdr.packet_size, hte->comm.buf);
	Trspi_LoadBlob_UINT32(&offset, hte->comm.hdr.u.ordinal, hte->comm.buf);
	Trspi_LoadBlob_UINT32(&offset, hte->comm.hdr.num_parms, hte->comm.buf);
	Trspi_LoadBlob_UINT32(&offset, hte->comm.hdr.type_size, hte->comm.buf);
	Trspi_LoadBlob_UINT32(&offset, hte->comm.hdr.type_offset, hte->comm.buf);
	Trspi_LoadBlob_UINT32(&offset, hte->comm.hdr.parm_size, hte->comm.buf);
	Trspi_LoadBlob_UINT32(&offset, hte->comm.hdr.parm_offset, hte->comm.buf);

#if 0
	/* ---  Send it */
	printBuffer(hte->comm.buf, hte->comm.hdr.packet_size);
	LogInfo("Sending Packet with TCSD ordinal 0x%X", hte->comm.hdr.u.ordinal);
#endif
	/* if the ordinal is open context, there are some host table entry
	 * manipulations that must be done, so call _init
	 */
	if (hte->comm.hdr.u.ordinal == TCSD_ORD_OPENCONTEXT) {
		if ((rc = send_init(hte))) {
			LogError("Failed to send packet");
			return rc;
		}
	} else {
		if ((rc = sendit(hte))) {
			LogError("Failed to send packet");
			return rc;
		}
	}

	/* create a platform version of the tcsd header */
	offset = 0;
	Trspi_UnloadBlob_UINT32(&offset, &hte->comm.hdr.packet_size, hte->comm.buf);
	Trspi_UnloadBlob_UINT32(&offset, &hte->comm.hdr.u.result, hte->comm.buf);
	Trspi_UnloadBlob_UINT32(&offset, &hte->comm.hdr.num_parms, hte->comm.buf);
	Trspi_UnloadBlob_UINT32(&offset, &hte->comm.hdr.type_size, hte->comm.buf);
	Trspi_UnloadBlob_UINT32(&offset, &hte->comm.hdr.type_offset, hte->comm.buf);
	Trspi_UnloadBlob_UINT32(&offset, &hte->comm.hdr.parm_size, hte->comm.buf);
	Trspi_UnloadBlob_UINT32(&offset, &hte->comm.hdr.parm_offset, hte->comm.buf);

	return TSS_SUCCESS;
}

/* ---------------------------------------------- */
/*	TCS Commands                              */
/* ---------------------------------------------- */

TSS_RESULT
TCS_OpenContext_RPC_TP(struct host_table_entry *hte, TCS_CONTEXT_HANDLE *hContext)
{
	TSS_RESULT result;

	initData(&hte->comm, 0);

	hte->comm.hdr.u.ordinal = TCSD_ORD_OPENCONTEXT;
	result = sendTCSDPacket(hte);

	if (result == TSS_SUCCESS)
		result = hte->comm.hdr.u.result;

	if (result == TSS_SUCCESS) {
		if (getData(TCSD_PACKET_TYPE_UINT32, 0, hContext, 0, &hte->comm))
			return TSPERR(TSS_E_INTERNAL_ERROR);

		LogDebugFn("Received TCS Context: 0x%x", *hContext);
	}

	return result;
}

TSS_RESULT
TCSP_GetRegisteredKeyByPublicInfo_TP(struct host_table_entry *hte,
				     TCPA_ALGORITHM_ID algID,	/* in */
				     UINT32 ulPublicInfoLength,	/* in */
				     BYTE * rgbPublicInfo,	/* in */
				     UINT32 * keySize,		/* out */
				     BYTE ** keyBlob)		/* out */
{
	TSS_RESULT result;

	initData(&hte->comm, 4);

	hte->comm.hdr.u.ordinal = TCSD_ORD_GETREGISTEREDKEYBYPUBLICINFO;
	LogDebugFn("TCS Context: 0x%x", hte->tcsContext);

	if (setData(TCSD_PACKET_TYPE_UINT32, 0, &hte->tcsContext, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_UINT32, 1, &algID, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_UINT32, 2, &ulPublicInfoLength, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_PBYTE, 3, rgbPublicInfo, ulPublicInfoLength, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);

	result = sendTCSDPacket(hte);

	if (result == TSS_SUCCESS)
		result = hte->comm.hdr.u.result;

	if (result == TSS_SUCCESS) {
		if (getData(TCSD_PACKET_TYPE_UINT32, 0, keySize, 0, &hte->comm))
			return TSPERR(TSS_E_INTERNAL_ERROR);
		*keyBlob = (BYTE *) malloc(*keySize);
		if (*keyBlob == NULL) {
			LogError("malloc of %u bytes failed.", *keySize);
			return TSPERR(TSS_E_OUTOFMEMORY);
		}
		if (getData(TCSD_PACKET_TYPE_PBYTE, 1, *keyBlob, *keySize, &hte->comm)) {
			free(*keyBlob);
			return TSPERR(TSS_E_INTERNAL_ERROR);
		}
	}

	return result;
}

TSS_RESULT
TSC_PhysicalPresence_TP(UINT16 physPres)
{
	return TSPERR(TSS_E_NOTIMPL);
}

TSS_RESULT
TCS_CloseContext_TP(struct host_table_entry *hte)
{
	TSS_RESULT result;

	initData(&hte->comm, 1);

	hte->comm.hdr.u.ordinal = TCSD_ORD_CLOSECONTEXT;
	LogDebugFn("TCS Context: 0x%x", hte->tcsContext);

	if (setData(TCSD_PACKET_TYPE_UINT32, 0, &hte->tcsContext, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);

	result = sendTCSDPacket(hte);

	if (result == TSS_SUCCESS)
		result = hte->comm.hdr.u.result;

	return result;
}

TSS_RESULT
TCS_FreeMemory_TP(struct host_table_entry *hte,
			      BYTE * pMemory	/*  in */
    ) {
	free(pMemory);

	return TSS_SUCCESS;
}

TSS_RESULT
TCS_LogPcrEvent_TP(struct host_table_entry *hte,
			       TSS_PCR_EVENT Event,	/*  in  */
			       UINT32 * pNumber	/*  out */
    ) {
	TSS_RESULT result;

	initData(&hte->comm, 2);

	hte->comm.hdr.u.ordinal = TCSD_ORD_LOGPCREVENT;
	LogDebugFn("TCS Context: 0x%x", hte->tcsContext);

	if (setData(TCSD_PACKET_TYPE_UINT32, 0, &hte->tcsContext, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);

	if (setData(TCSD_PACKET_TYPE_PCR_EVENT, 1, &Event, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);

	result = sendTCSDPacket(hte);

	if (result == TSS_SUCCESS)
		result = hte->comm.hdr.u.result;

	if (result == TSS_SUCCESS) {
		if (getData(TCSD_PACKET_TYPE_UINT32, 0, pNumber, 0, &hte->comm))
			result = TSPERR(TSS_E_INTERNAL_ERROR);
	}

	return result;
}

TSS_RESULT
TCS_GetPcrEvent_TP(struct host_table_entry *hte,
			       UINT32 PcrIndex,	/* in */
			       UINT32 * pNumber,	/* in, out */
			       TSS_PCR_EVENT ** ppEvent	/* out */
    ) {
	TSS_RESULT result;
	BYTE lengthOnly = (ppEvent == NULL) ? TRUE : FALSE;

	initData(&hte->comm, 4);

	hte->comm.hdr.u.ordinal = TCSD_ORD_GETPCREVENT;
	LogDebugFn("TCS Context: 0x%x", hte->tcsContext);

	if (setData(TCSD_PACKET_TYPE_UINT32, 0, &hte->tcsContext, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);

	if (setData(TCSD_PACKET_TYPE_UINT32, 1, &PcrIndex, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);

	if (setData(TCSD_PACKET_TYPE_UINT32, 2, pNumber, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);

	if (setData(TCSD_PACKET_TYPE_BYTE, 3, &lengthOnly, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);

	result = sendTCSDPacket(hte);

	if (result == TSS_SUCCESS)
		result = hte->comm.hdr.u.result;

	if (result == TSS_SUCCESS) {
		if (getData(TCSD_PACKET_TYPE_UINT32, 0, pNumber, 0, &hte->comm))
			return TSPERR(TSS_E_INTERNAL_ERROR);

		if (ppEvent) {
			*ppEvent = malloc(sizeof(TSS_PCR_EVENT));
			if (*ppEvent == NULL) {
				LogError("malloc of %zd bytes failed.",
					 sizeof(TSS_PCR_EVENT));
				return TSPERR(TSS_E_OUTOFMEMORY);
			}

			if (getData(TCSD_PACKET_TYPE_PCR_EVENT, 1, *ppEvent, 0, &hte->comm)) {
				free(*ppEvent);
				*ppEvent = NULL;
				return TSPERR(TSS_E_INTERNAL_ERROR);
			}
		}
	}

	return result;
}

TSS_RESULT
TCS_GetPcrEventsByPcr_TP(struct host_table_entry *hte,
				     UINT32 PcrIndex,	/* in */
				     UINT32 FirstEvent,	/* in */
				     UINT32 * pEventCount,	/* in, out */
				     TSS_PCR_EVENT ** ppEvents	/* out */
    ) {
	TSS_RESULT result;
	UINT32 i, j;

	initData(&hte->comm, 4);

	hte->comm.hdr.u.ordinal = TCSD_ORD_GETPCREVENTBYPCR;
	LogDebugFn("TCS Context: 0x%x", hte->tcsContext);

	if (setData(TCSD_PACKET_TYPE_UINT32, 0, &hte->tcsContext, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);

	if (setData(TCSD_PACKET_TYPE_UINT32, 1, &PcrIndex, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);

	if (setData(TCSD_PACKET_TYPE_UINT32, 2, &FirstEvent, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);

	if (setData(TCSD_PACKET_TYPE_UINT32, 3, pEventCount, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);

	result = sendTCSDPacket(hte);

	if (result == TSS_SUCCESS)
		result = hte->comm.hdr.u.result;

	if (result == TSS_SUCCESS) {
		if (getData(TCSD_PACKET_TYPE_UINT32, 0, pEventCount, 0, &hte->comm))
			return TSPERR(TSS_E_INTERNAL_ERROR);

		if (*pEventCount > 0) {
			*ppEvents = calloc_tspi(hte->tspContext,
					        sizeof(TSS_PCR_EVENT) * (*pEventCount));
			if (*ppEvents == NULL) {
				LogError("malloc of %zd bytes failed.",
					 sizeof(TSS_PCR_EVENT) * (*pEventCount));
				return TSPERR(TSS_E_OUTOFMEMORY);
			}

			i = 1;
			for (j = 0; j < (*pEventCount); j++) {
				if (getData(TCSD_PACKET_TYPE_PCR_EVENT, i++, &((*ppEvents)[j]), 0,
					    &hte->comm)) {
					free(*ppEvents);
					*ppEvents = NULL;
					return TSPERR(TSS_E_INTERNAL_ERROR);
				}
			}
		} else {
			*ppEvents = NULL;
		}
	}

	return result;
}

TSS_RESULT
TCS_GetPcrEventLog_TP(struct host_table_entry *hte,
				  UINT32 * pEventCount,	/* out */
				  TSS_PCR_EVENT ** ppEvents	/* out */
    ) {
	TSS_RESULT result;
	int i, j;

	initData(&hte->comm, 1);

	hte->comm.hdr.u.ordinal = TCSD_ORD_GETPCREVENTLOG;
	LogDebugFn("TCS Context: 0x%x", hte->tcsContext);

	if (setData(TCSD_PACKET_TYPE_UINT32, 0, &hte->tcsContext, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);

	result = sendTCSDPacket(hte);

	if (result == TSS_SUCCESS)
		result = hte->comm.hdr.u.result;

	if (result == TSS_SUCCESS) {
		if (getData(TCSD_PACKET_TYPE_UINT32, 0, pEventCount, 0, &hte->comm))
			return TSPERR(TSS_E_INTERNAL_ERROR);

		if (*pEventCount > 0) {
			*ppEvents = calloc_tspi(hte->tspContext,
						sizeof(TSS_PCR_EVENT) * (*pEventCount));
			if (*ppEvents == NULL) {
				LogError("malloc of %zd bytes failed.", sizeof(TSS_PCR_EVENT) * (*pEventCount));
				return TSPERR(TSS_E_OUTOFMEMORY);
			}

			i = 1;
			for (j = 0; (UINT32)j < (*pEventCount); j++) {
				if (getData(TCSD_PACKET_TYPE_PCR_EVENT, i++, &((*ppEvents)[j]), 0, &hte->comm)) {
					free(*ppEvents);
					*ppEvents = NULL;
					return TSPERR(TSS_E_INTERNAL_ERROR);
				}
			}
		} else {
			*ppEvents = NULL;
		}
	}

	return result;
}

TSS_RESULT
TCS_RegisterKey_TP(struct host_table_entry *hte,
			       TSS_UUID WrappingKeyUUID,	/* in */
			       TSS_UUID KeyUUID,	/* in */
			       UINT32 cKeySize,	/* in */
			       BYTE * rgbKey,	/* in */
			       UINT32 cVendorData,	/* in */
			       BYTE * gbVendorData	/* in */
    ) {
	TSS_RESULT result;

	initData(&hte->comm, 7);

	hte->comm.hdr.u.ordinal = TCSD_ORD_REGISTERKEY;
	LogDebugFn("TCS Context: 0x%x", hte->tcsContext);

	if (setData(TCSD_PACKET_TYPE_UINT32, 0, &hte->tcsContext, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_UUID, 1, &WrappingKeyUUID, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_UUID, 2, &KeyUUID, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_UINT32, 3, &cKeySize, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_PBYTE, 4, rgbKey, cKeySize, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_UINT32, 5, &cVendorData, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_PBYTE, 6, gbVendorData, cVendorData, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);

	result = sendTCSDPacket(hte);

	if (result == TSS_SUCCESS)
		result = hte->comm.hdr.u.result;

	return result;
}

TSS_RESULT
TCSP_UnregisterKey_TP(struct host_table_entry *hte,
				  TSS_UUID KeyUUID	/* in */
    ) {
	TSS_RESULT result;

	initData(&hte->comm, 2);

	hte->comm.hdr.u.ordinal = TCSD_ORD_UNREGISTERKEY;
	LogDebugFn("TCS Context: 0x%x", hte->tcsContext);

	if (setData(TCSD_PACKET_TYPE_UINT32, 0, &hte->tcsContext, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_UUID, 1, &KeyUUID, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);

	result = sendTCSDPacket(hte);

	if (result == TSS_SUCCESS)
		result = hte->comm.hdr.u.result;

	return result;
}

TSS_RESULT
TCS_EnumRegisteredKeys_TP(struct host_table_entry *hte,
				      TSS_UUID * pKeyUUID,	/* in */
				      UINT32 * pcKeyHierarchySize,	/* out */
				      TSS_KM_KEYINFO ** ppKeyHierarchy	/* out */
    ) {
	TSS_RESULT result;
	int i, j;

	initData(&hte->comm, 2);

	hte->comm.hdr.u.ordinal = TCSD_ORD_ENUMREGISTEREDKEYS;
	LogDebugFn("TCS Context: 0x%x", hte->tcsContext);

	if (setData(TCSD_PACKET_TYPE_UINT32, 0, &hte->tcsContext, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);

	if (pKeyUUID != NULL) {
		if (setData(TCSD_PACKET_TYPE_UUID, 1, pKeyUUID, 0, &hte->comm))
			return TSPERR(TSS_E_INTERNAL_ERROR);
	}

	result = sendTCSDPacket(hte);

	if (result == TSS_SUCCESS)
		result = hte->comm.hdr.u.result;

	if (result == TSS_SUCCESS) {
		i = 0;
		if (getData(TCSD_PACKET_TYPE_UINT32, i++, pcKeyHierarchySize, 0, &hte->comm))
			return TSPERR(TSS_E_INTERNAL_ERROR);

		if (*pcKeyHierarchySize > 0) {
			*ppKeyHierarchy = malloc((*pcKeyHierarchySize) * sizeof(TSS_KM_KEYINFO));
			if (*ppKeyHierarchy == NULL) {
				LogError("malloc of %zu bytes failed.", (*pcKeyHierarchySize) *
					 sizeof(TSS_KM_KEYINFO));
				return TSPERR(TSS_E_OUTOFMEMORY);
			}
			for (j = 0; (UINT32)j < *pcKeyHierarchySize; j++) {
				if (getData(TCSD_PACKET_TYPE_KM_KEYINFO, i++,
					    &((*ppKeyHierarchy)[j]), 0, &hte->comm)) {
					free(*ppKeyHierarchy);
					return TSPERR(TSS_E_INTERNAL_ERROR);
				}
			}
		} else {
			*ppKeyHierarchy = NULL;
		}
	}

	return result;
}

TSS_RESULT
TCS_GetRegisteredKey_TP(struct host_table_entry *hte,
				    TSS_UUID KeyUUID,	/* in */
				    TSS_KM_KEYINFO ** ppKeyInfo	/* out */
    ) {
	return TSPERR(TSS_E_NOTIMPL);
}

TSS_RESULT
TCS_GetRegisteredKeyBlob_TP(struct host_table_entry *hte,
					TSS_UUID KeyUUID,	/* in */
					UINT32 * pcKeySize,	/* out */
					BYTE ** prgbKey	/* out */
    ) {
	TSS_RESULT result;

	initData(&hte->comm, 2);

	hte->comm.hdr.u.ordinal = TCSD_ORD_GETREGISTEREDKEYBLOB;
	LogDebugFn("TCS Context: 0x%x", hte->tcsContext);

	if (setData(TCSD_PACKET_TYPE_UINT32, 0, &hte->tcsContext, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_UUID, 1, &KeyUUID, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);

	result = sendTCSDPacket(hte);

	if (result == TSS_SUCCESS)
		result = hte->comm.hdr.u.result;

	if (result == TSS_SUCCESS) {
		if (getData(TCSD_PACKET_TYPE_UINT32, 0, pcKeySize, 0, &hte->comm))
			result = TSPERR(TSS_E_INTERNAL_ERROR);
		*prgbKey = malloc(*pcKeySize);
		if (*prgbKey == NULL) {
			LogError("malloc of %u bytes failed.", *pcKeySize);
			return TSPERR(TSS_E_OUTOFMEMORY);
		}
		if (getData(TCSD_PACKET_TYPE_PBYTE, 1, *prgbKey, *pcKeySize, &hte->comm)) {
			free(*prgbKey);
			return TSPERR(TSS_E_INTERNAL_ERROR);
		}
	}

	return result;
}

TSS_RESULT
TCSP_LoadKeyByBlob_TP(struct host_table_entry *hte,
				  TCS_KEY_HANDLE hUnwrappingKey,	/* in */
				  UINT32 cWrappedKeyBlobSize,	/* in */
				  BYTE * rgbWrappedKeyBlob,	/* in */
				  TPM_AUTH * pAuth,	/* in, out */
				  TCS_KEY_HANDLE * phKeyTCSI,	/* out */
				  TCS_KEY_HANDLE * phKeyHMAC	/* out */
    ) {
	TSS_RESULT result;
	int i;

	initData(&hte->comm, 5);

	hte->comm.hdr.u.ordinal = TCSD_ORD_LOADKEYBYBLOB;
	LogDebugFn("IN: TCS Context: 0x%x", hte->tcsContext);

	if (setData(TCSD_PACKET_TYPE_UINT32, 0, &hte->tcsContext, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_UINT32, 1, &hUnwrappingKey, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_UINT32, 2, &cWrappedKeyBlobSize, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_PBYTE, 3, rgbWrappedKeyBlob, cWrappedKeyBlobSize, &hte->comm))
			return TSPERR(TSS_E_INTERNAL_ERROR);

	if (pAuth != NULL) {
		if (setData(TCSD_PACKET_TYPE_AUTH, 4, pAuth, 0, &hte->comm))
			return TSPERR(TSS_E_INTERNAL_ERROR);
	}

	result = sendTCSDPacket(hte);

	if (result == TSS_SUCCESS)
		result = hte->comm.hdr.u.result;

	if (result == TSS_SUCCESS) {
		i = 0;
		if (pAuth != NULL) {
			if (getData(TCSD_PACKET_TYPE_AUTH, i++, pAuth, 0, &hte->comm))
				return TSPERR(TSS_E_INTERNAL_ERROR);
		}
		if (getData(TCSD_PACKET_TYPE_UINT32, i++, phKeyTCSI, 0, &hte->comm))
			return TSPERR(TSS_E_INTERNAL_ERROR);
		if (getData(TCSD_PACKET_TYPE_UINT32, i++, phKeyHMAC, 0, &hte->comm))
			return TSPERR(TSS_E_INTERNAL_ERROR);

		LogDebugFn("OUT: TCS key handle: 0x%x, TPM key slot: 0x%x", *phKeyTCSI,
			   *phKeyHMAC);
	}

	return result;
}

TSS_RESULT
TCSP_LoadKeyByUUID_TP(struct host_table_entry *hte,
				  TSS_UUID KeyUUID,	/* in */
				  TCS_LOADKEY_INFO * pLoadKeyInfo,	/* in, out */
				  TCS_KEY_HANDLE * phKeyTCSI	/* out */
    ) {
	TSS_RESULT result;

	initData(&hte->comm, 3);

	hte->comm.hdr.u.ordinal = TCSD_ORD_LOADKEYBYUUID;
	LogDebugFn("TCS Context: 0x%x", hte->tcsContext);

	if (setData(TCSD_PACKET_TYPE_UINT32, 0, &hte->tcsContext, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_UUID, 1, &KeyUUID, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);

	if (pLoadKeyInfo != NULL) {
		if (setData(TCSD_PACKET_TYPE_LOADKEY_INFO, 2, pLoadKeyInfo, 0, &hte->comm))
			return TSPERR(TSS_E_INTERNAL_ERROR);
	}

	result = sendTCSDPacket(hte);

	if (result == TSS_SUCCESS)
		result = hte->comm.hdr.u.result;

	if (result == TSS_SUCCESS) {
		if (getData(TCSD_PACKET_TYPE_UINT32, 0, phKeyTCSI, 0, &hte->comm))
			return TSPERR(TSS_E_INTERNAL_ERROR);

		LogDebugFn("TCS key handle: 0x%x", *phKeyTCSI);
	} else if (pLoadKeyInfo && (result == TCSERR(TCS_E_KM_LOADFAILED))) {
		if (getData(TCSD_PACKET_TYPE_LOADKEY_INFO, 0, pLoadKeyInfo, 0, &hte->comm))
			return TSPERR(TSS_E_INTERNAL_ERROR);
	}

	return result;
}

TSS_RESULT
TCSP_EvictKey_TP(struct host_table_entry *hte,
			     TCS_KEY_HANDLE hKey	/* in */
    ) {
	TSS_RESULT result;

	initData(&hte->comm, 2);

	hte->comm.hdr.u.ordinal = TCSD_ORD_EVICTKEY;
	LogDebugFn("TCS Context: 0x%x", hte->tcsContext);

	if (setData(TCSD_PACKET_TYPE_UINT32, 0, &hte->tcsContext, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_UINT32, 1, &hKey, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);

	result = sendTCSDPacket(hte);

	if (result == TSS_SUCCESS)
		result = hte->comm.hdr.u.result;

	return result;
}

TSS_RESULT
TCSP_CreateWrapKey_TP(struct host_table_entry *hte,
				  TCS_KEY_HANDLE hWrappingKey,	/* in */
				  TCPA_ENCAUTH KeyUsageAuth,	/* in */
				  TCPA_ENCAUTH KeyMigrationAuth,	/* in */
				  UINT32 keyInfoSize,	/* in */
				  BYTE * keyInfo,	/* in */
				  UINT32 * keyDataSize,	/* out */
				  BYTE ** keyData,	/* out */
				  TPM_AUTH * pAuth	/* in, out */
    ) {
	TSS_RESULT result;

	initData(&hte->comm, 7);

	hte->comm.hdr.u.ordinal = TCSD_ORD_CREATEWRAPKEY;
	LogDebugFn("TCS Context: 0x%x", hte->tcsContext);

	if (setData(TCSD_PACKET_TYPE_UINT32, 0, &hte->tcsContext, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_UINT32, 1, &hWrappingKey, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_ENCAUTH, 2, &KeyUsageAuth, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_ENCAUTH, 3, &KeyMigrationAuth, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_UINT32, 4, &keyInfoSize, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_PBYTE, 5, keyInfo, keyInfoSize, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_AUTH, 6, pAuth, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);

	result = sendTCSDPacket(hte);

	if (result == TSS_SUCCESS)
		result = hte->comm.hdr.u.result;

	if (result == TSS_SUCCESS) {
		if (getData(TCSD_PACKET_TYPE_UINT32, 0, keyDataSize, 0, &hte->comm))
			return TSPERR(TSS_E_INTERNAL_ERROR);
		*keyData = (BYTE *) malloc(*keyDataSize);
		if (*keyData == NULL) {
			LogError("malloc of %u bytes failed.", *keyDataSize);
			return TSPERR(TSS_E_OUTOFMEMORY);
		}
		if (getData(TCSD_PACKET_TYPE_PBYTE, 1, *keyData, *keyDataSize, &hte->comm)) {
			free(*keyData);
			return TSPERR(TSS_E_INTERNAL_ERROR);
		}
		if (getData(TCSD_PACKET_TYPE_AUTH, 2, pAuth, 0, &hte->comm)) {
			free(*keyData);
			return TSPERR(TSS_E_INTERNAL_ERROR);
		}
	}

	return result;
}

TSS_RESULT
TCSP_GetPubKey_TP(struct host_table_entry *hte,
			      TCS_KEY_HANDLE hKey,	/* in */
			      TPM_AUTH * pAuth,	/* in, out */
			      UINT32 * pcPubKeySize,	/* out */
			      BYTE ** prgbPubKey	/* out */
    ) {
	TSS_RESULT result;
	int i;

	initData(&hte->comm, 3);

	hte->comm.hdr.u.ordinal = TCSD_ORD_GETPUBKEY;
	LogDebugFn("TCS Context: 0x%x", hte->tcsContext);

	if (setData(TCSD_PACKET_TYPE_UINT32, 0, &hte->tcsContext, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_UINT32, 1, &hKey, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (pAuth != NULL) {
		if (setData(TCSD_PACKET_TYPE_AUTH, 2, pAuth, 0, &hte->comm))
			return TSPERR(TSS_E_INTERNAL_ERROR);
	}

	result = sendTCSDPacket(hte);

	if (result == TSS_SUCCESS)
		result = hte->comm.hdr.u.result;

	i = 0;
	if (result == TSS_SUCCESS) {
		if (pAuth != NULL) {
			if (getData(TCSD_PACKET_TYPE_AUTH, i++, pAuth, 0, &hte->comm))
				return TSPERR(TSS_E_INTERNAL_ERROR);
		}
		if (getData(TCSD_PACKET_TYPE_UINT32, i++, pcPubKeySize, 0, &hte->comm))
			return TSPERR(TSS_E_INTERNAL_ERROR);

		*prgbPubKey = (BYTE *) calloc_tspi(hte->tspContext, *pcPubKeySize);
		if (*prgbPubKey == NULL) {
			LogError("malloc of %u bytes failed.", *pcPubKeySize);
			return TSPERR(TSS_E_OUTOFMEMORY);
		}
		if (getData(TCSD_PACKET_TYPE_PBYTE, i++, *prgbPubKey, *pcPubKeySize, &hte->comm)) {
			free_tspi(hte->tspContext, *prgbPubKey);
			return TSPERR(TSS_E_INTERNAL_ERROR);
		}
	}

	return result;
}

TSS_RESULT
TCSP_MakeIdentity_TP(struct host_table_entry *hte,
				 TCPA_ENCAUTH identityAuth,	/* in */
				 TCPA_CHOSENID_HASH IDLabel_PrivCAHash,	/* in */
				 UINT32 idKeyInfoSize,	/* in */
				 BYTE * idKeyInfo,	/* in */
				 TPM_AUTH * pSrkAuth,	/* in, out */
				 TPM_AUTH * pOwnerAuth,	/* in, out */
				 UINT32 * idKeySize,	/* out */
				 BYTE ** idKey,	/* out */
				 UINT32 * pcIdentityBindingSize,	/* out */
				 BYTE ** prgbIdentityBinding,	/* out */
				 UINT32 * pcEndorsementCredentialSize,	/* out */
				 BYTE ** prgbEndorsementCredential,	/* out */
				 UINT32 * pcPlatformCredentialSize,	/* out */
				 BYTE ** prgbPlatformCredential,	/* out */
				 UINT32 * pcConformanceCredentialSize,	/* out */
				 BYTE ** prgbConformanceCredential	/* out */
    ) {
	TSS_RESULT result;
	int i;

	initData(&hte->comm, 7);

	hte->comm.hdr.u.ordinal = TCSD_ORD_MAKEIDENTITY;
	LogDebugFn("TCS Context: 0x%x", hte->tcsContext);

	if (setData(TCSD_PACKET_TYPE_UINT32, 0, &hte->tcsContext, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_ENCAUTH, 1, &identityAuth, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_DIGEST, 2, &IDLabel_PrivCAHash, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_UINT32, 3, &idKeyInfoSize, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_PBYTE, 4, idKeyInfo, idKeyInfoSize, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	i = 5;
	if (pSrkAuth) {
		if (setData(TCSD_PACKET_TYPE_AUTH, i++, pSrkAuth, 0, &hte->comm))
			return TSPERR(TSS_E_INTERNAL_ERROR);
	}
	if (setData(TCSD_PACKET_TYPE_AUTH, i++, pOwnerAuth, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);

	result = sendTCSDPacket(hte);

	if (result == TSS_SUCCESS)
		result = hte->comm.hdr.u.result;

	i = 0;
	if (result == TSS_SUCCESS) {
		i = 0;
		if (pSrkAuth) {
			if (getData(TCSD_PACKET_TYPE_AUTH, i++, pSrkAuth, 0, &hte->comm))
				return TSPERR(TSS_E_INTERNAL_ERROR);
		}
		if (getData(TCSD_PACKET_TYPE_AUTH, i++, pOwnerAuth, 0, &hte->comm))
			return TSPERR(TSS_E_INTERNAL_ERROR);
		if (getData(TCSD_PACKET_TYPE_UINT32, i++, idKeySize, 0, &hte->comm))
			return TSPERR(TSS_E_INTERNAL_ERROR);

		*idKey = (BYTE *) malloc(*idKeySize);
		if (*idKey == NULL) {
			LogError("malloc of %u bytes failed.", *idKeySize);
			return TSPERR(TSS_E_OUTOFMEMORY);
		}
		if (getData(TCSD_PACKET_TYPE_PBYTE, i++, *idKey, *idKeySize, &hte->comm)) {
			free(*idKey);
			return TSPERR(TSS_E_INTERNAL_ERROR);
		}
		if (getData(TCSD_PACKET_TYPE_UINT32, i++, pcIdentityBindingSize, 0, &hte->comm)) {
			free(*idKey);
			return TSPERR(TSS_E_INTERNAL_ERROR);
		}

		*prgbIdentityBinding = (BYTE *) malloc(*pcIdentityBindingSize);
		if (*prgbIdentityBinding == NULL) {
			LogError("malloc of %u bytes failed.", *pcIdentityBindingSize);
			free(*idKey);
			return TSPERR(TSS_E_OUTOFMEMORY);
		}
		if (getData(TCSD_PACKET_TYPE_PBYTE, i++, *prgbIdentityBinding, *pcIdentityBindingSize, &hte->comm)) {
			free(*idKey);
			free(*prgbIdentityBinding);
			return TSPERR(TSS_E_INTERNAL_ERROR);
		}
		if (getData(TCSD_PACKET_TYPE_UINT32, i++, pcEndorsementCredentialSize, 0, &hte->comm)) {
			free(*idKey);
			free(*prgbIdentityBinding);
			return TSPERR(TSS_E_INTERNAL_ERROR);
		}

		*prgbEndorsementCredential = (BYTE *) malloc(*pcEndorsementCredentialSize);
		if (*prgbEndorsementCredential == NULL) {
			LogError("malloc of %u bytes failed.", *pcEndorsementCredentialSize);
			free(*idKey);
			free(*prgbIdentityBinding);
			return TSPERR(TSS_E_OUTOFMEMORY);
		}
		if (getData(TCSD_PACKET_TYPE_PBYTE, i++, *prgbEndorsementCredential, *pcEndorsementCredentialSize, &hte->comm)) {
			free(*idKey);
			free(*prgbIdentityBinding);
			free(*prgbEndorsementCredential);
			return TSPERR(TSS_E_INTERNAL_ERROR);
		}
		if (getData(TCSD_PACKET_TYPE_UINT32, i++, pcPlatformCredentialSize, 0, &hte->comm)) {
			free(*idKey);
			free(*prgbIdentityBinding);
			free(*prgbEndorsementCredential);
			return TSPERR(TSS_E_INTERNAL_ERROR);
		}

		*prgbPlatformCredential = (BYTE *) malloc(*pcPlatformCredentialSize);
		if (*prgbPlatformCredential == NULL) {
			LogError("malloc of %u bytes failed.", *pcPlatformCredentialSize);
			free(*idKey);
			free(*prgbIdentityBinding);
			free(*prgbEndorsementCredential);
			return TSPERR(TSS_E_OUTOFMEMORY);
		}
		if (getData(TCSD_PACKET_TYPE_PBYTE, i++, *prgbPlatformCredential, *pcPlatformCredentialSize, &hte->comm)) {
			free(*idKey);
			free(*prgbIdentityBinding);
			free(*prgbEndorsementCredential);
			free(*prgbPlatformCredential);
			return TSPERR(TSS_E_INTERNAL_ERROR);
		}
		if (getData(TCSD_PACKET_TYPE_UINT32, i++, pcConformanceCredentialSize, 0, &hte->comm)) {
			free(*idKey);
			free(*prgbIdentityBinding);
			free(*prgbEndorsementCredential);
			free(*prgbPlatformCredential);
			return TSPERR(TSS_E_INTERNAL_ERROR);
		}

		*prgbConformanceCredential = (BYTE *) malloc(*pcConformanceCredentialSize);
		if (*prgbConformanceCredential == NULL) {
			LogError("malloc of %u bytes failed.", *pcConformanceCredentialSize);
			free(*idKey);
			free(*prgbIdentityBinding);
			free(*prgbEndorsementCredential);
			free(*prgbPlatformCredential);
			return TSPERR(TSS_E_OUTOFMEMORY);
		}
		if (getData(TCSD_PACKET_TYPE_PBYTE, i++, *prgbConformanceCredential, *pcConformanceCredentialSize, &hte->comm)) {
			free(*idKey);
			free(*prgbIdentityBinding);
			free(*prgbEndorsementCredential);
			free(*prgbPlatformCredential);
			free(*prgbConformanceCredential);
			return TSPERR(TSS_E_INTERNAL_ERROR);
		}
	}

	return result;
}

TSS_RESULT
TCSP_SetOwnerInstall_TP(struct host_table_entry *hte,
				    TSS_BOOL state	/* in */
    ) {
	TSS_RESULT result;

	initData(&hte->comm, 2);

	hte->comm.hdr.u.ordinal = TCSD_ORD_SETOWNERINSTALL;
	LogDebugFn("TCS Context: 0x%x", hte->tcsContext);

	if (setData(TCSD_PACKET_TYPE_UINT32, 0, &hte->tcsContext, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_BOOL, 1, &state, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);

	result = sendTCSDPacket(hte);

	if (result == TSS_SUCCESS)
		result = hte->comm.hdr.u.result;

	return result;
}

TSS_RESULT
TCSP_TakeOwnership_TP(struct host_table_entry *hte,
				  UINT16 protocolID,	/* in */
				  UINT32 encOwnerAuthSize,	/* in */
				  BYTE * encOwnerAuth,	/* in */
				  UINT32 encSrkAuthSize,	/* in */
				  BYTE * encSrkAuth,	/* in */
				  UINT32 srkInfoSize,	/* in */
				  BYTE * srkInfo,	/* in */
				  TPM_AUTH * ownerAuth,	/* in, out */
				  UINT32 * srkKeySize, BYTE ** srkKey) {
	TSS_RESULT result;

	initData(&hte->comm, 9);

	hte->comm.hdr.u.ordinal = TCSD_ORD_TAKEOWNERSHIP;
	LogDebugFn("TCS Context: 0x%x", hte->tcsContext);

	if (setData(TCSD_PACKET_TYPE_UINT32, 0, &hte->tcsContext, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_UINT16, 1, &protocolID, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_UINT32, 2, &encOwnerAuthSize, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_PBYTE, 3, encOwnerAuth, encOwnerAuthSize, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_UINT32, 4, &encSrkAuthSize, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_PBYTE, 5, encSrkAuth, encSrkAuthSize, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_UINT32, 6, &srkInfoSize, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_PBYTE, 7, srkInfo, srkInfoSize, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_AUTH, 8, ownerAuth, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);

	result = sendTCSDPacket(hte);

	if (result == TSS_SUCCESS)
		result = hte->comm.hdr.u.result;

	if (result == TSS_SUCCESS) {
		if (getData(TCSD_PACKET_TYPE_AUTH, 0, ownerAuth, 0, &hte->comm))
			return TSPERR(TSS_E_INTERNAL_ERROR);
		if (getData(TCSD_PACKET_TYPE_UINT32, 1, srkKeySize, 0, &hte->comm))
			return TSPERR(TSS_E_INTERNAL_ERROR);

		*srkKey = (BYTE *) malloc(*srkKeySize);
		if (*srkKey == NULL) {
			LogError("malloc of %u bytes failed.", *srkKeySize);
			return TSPERR(TSS_E_OUTOFMEMORY);
		}
		if (getData(TCSD_PACKET_TYPE_PBYTE, 2, *srkKey, *srkKeySize, &hte->comm)) {
			free(*srkKey);
			return TSPERR(TSS_E_INTERNAL_ERROR);
		}
	}

	return result;
}

TSS_RESULT
TCSP_OIAP_TP(struct host_table_entry *hte,
			 TCS_AUTHHANDLE * authHandle,	/* out */
			 TCPA_NONCE * nonce0	/* out */
    ) {
	TSS_RESULT result;

	initData(&hte->comm, 1);

	hte->comm.hdr.u.ordinal = TCSD_ORD_OIAP;
	LogDebugFn("TCS Context: 0x%x", hte->tcsContext);

	if (setData(TCSD_PACKET_TYPE_UINT32, 0, &hte->tcsContext, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);

	result = sendTCSDPacket(hte);

	if (result == TSS_SUCCESS)
		result = hte->comm.hdr.u.result;

	if (result == TSS_SUCCESS) {
		if (getData(TCSD_PACKET_TYPE_UINT32, 0, authHandle, 0, &hte->comm))
			return TSPERR(TSS_E_INTERNAL_ERROR);
		if (getData(TCSD_PACKET_TYPE_NONCE, 1, nonce0, 0, &hte->comm))
			return TSPERR(TSS_E_INTERNAL_ERROR);
	}

	return result;
}

TSS_RESULT
TCSP_OSAP_TP(struct host_table_entry *hte,
			 TCPA_ENTITY_TYPE entityType,	/* in */
			 UINT32 entityValue,	/* in */
			 TCPA_NONCE nonceOddOSAP,	/* in */
			 TCS_AUTHHANDLE * authHandle,	/* out */
			 TCPA_NONCE * nonceEven,	/* out */
			 TCPA_NONCE * nonceEvenOSAP	/* out */
    ) {
	TSS_RESULT result;

	initData(&hte->comm, 4);

	hte->comm.hdr.u.ordinal = TCSD_ORD_OSAP;
	LogDebugFn("TCS Context: 0x%x", hte->tcsContext);

	if (setData(TCSD_PACKET_TYPE_UINT32, 0, &hte->tcsContext, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_UINT16, 1, &entityType, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_UINT32, 2, &entityValue, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_NONCE, 3, &nonceOddOSAP, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);

	result = sendTCSDPacket(hte);

	if (result == TSS_SUCCESS)
		result = hte->comm.hdr.u.result;

	if (result == TSS_SUCCESS) {
		if (getData(TCSD_PACKET_TYPE_UINT32, 0, authHandle, 0, &hte->comm))
			return TSPERR(TSS_E_INTERNAL_ERROR);
		if (getData(TCSD_PACKET_TYPE_NONCE, 1, nonceEven, 0, &hte->comm))
			return TSPERR(TSS_E_INTERNAL_ERROR);
		if (getData(TCSD_PACKET_TYPE_NONCE, 2, nonceEvenOSAP, 0, &hte->comm))
			return TSPERR(TSS_E_INTERNAL_ERROR);
	}

	return result;
}

TSS_RESULT
TCSP_ChangeAuth_TP(struct host_table_entry *hte,
			       TCS_KEY_HANDLE parentHandle,	/* in */
			       TCPA_PROTOCOL_ID protocolID,	/* in */
			       TCPA_ENCAUTH newAuth,	/* in */
			       TCPA_ENTITY_TYPE entityType,	/* in */
			       UINT32 encDataSize,	/* in */
			       BYTE * encData,	/* in */
			       TPM_AUTH * ownerAuth,	/* in, out */
			       TPM_AUTH * entityAuth,	/* in, out */
			       UINT32 * outDataSize,	/* out */
			       BYTE ** outData	/* out */
    ) {
	TSS_RESULT result;

	initData(&hte->comm, 9);

	hte->comm.hdr.u.ordinal = TCSD_ORD_CHANGEAUTH;
	LogDebugFn("TCS Context: 0x%x", hte->tcsContext);

	if (setData(TCSD_PACKET_TYPE_UINT32, 0, &hte->tcsContext, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_UINT32, 1, &parentHandle, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_UINT16, 2, &protocolID, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_ENCAUTH, 3, &newAuth, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_UINT16, 4, &entityType, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_UINT32, 5, &encDataSize, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_PBYTE, 6, encData, encDataSize, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_AUTH, 7, ownerAuth, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_AUTH, 8, entityAuth, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);

	result = sendTCSDPacket(hte);

	if (result == TSS_SUCCESS)
		result = hte->comm.hdr.u.result;

	if (result == TSS_SUCCESS) {
		if (getData(TCSD_PACKET_TYPE_AUTH, 0, ownerAuth, 0, &hte->comm))
			return TSPERR(TSS_E_INTERNAL_ERROR);
		if (getData(TCSD_PACKET_TYPE_AUTH, 1, entityAuth, 0, &hte->comm))
			return TSPERR(TSS_E_INTERNAL_ERROR);
		if (getData(TCSD_PACKET_TYPE_UINT32, 2, outDataSize, 0, &hte->comm))
			return TSPERR(TSS_E_INTERNAL_ERROR);

		*outData = (BYTE *) malloc(*outDataSize);
		if (*outData == NULL) {
			LogError("malloc of %u bytes failed.", *outDataSize);
			return TSPERR(TSS_E_OUTOFMEMORY);
		}
		if (getData(TCSD_PACKET_TYPE_PBYTE, 3, *outData, *outDataSize, &hte->comm)) {
			free(*outData);
			return TSPERR(TSS_E_INTERNAL_ERROR);
		}
	}

	return result;
}

TSS_RESULT
TCSP_ChangeAuthOwner_TP(struct host_table_entry *hte,
				    TCPA_PROTOCOL_ID protocolID,	/* in */
				    TCPA_ENCAUTH newAuth,	/* in */
				    TCPA_ENTITY_TYPE entityType,	/* in */
				    TPM_AUTH * ownerAuth	/* in, out */
    ) {
	TSS_RESULT result;

	initData(&hte->comm, 5);

	hte->comm.hdr.u.ordinal = TCSD_ORD_CHANGEAUTHOWNER;
	LogDebugFn("TCS Context: 0x%x", hte->tcsContext);

	if (setData(TCSD_PACKET_TYPE_UINT32, 0, &hte->tcsContext, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_UINT16, 1, &protocolID, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_ENCAUTH, 2, &newAuth, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_UINT16, 3, &entityType, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_AUTH, 4, ownerAuth, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);

	result = sendTCSDPacket(hte);

	if (result == TSS_SUCCESS)
		result = hte->comm.hdr.u.result;

	if (result == TSS_SUCCESS) {
		if (getData(TCSD_PACKET_TYPE_AUTH, 0, ownerAuth, 0, &hte->comm))
			return TSPERR(TSS_E_INTERNAL_ERROR);
	}

	return result;
}

TSS_RESULT
TCSP_ChangeAuthAsymStart_TP(struct host_table_entry *hte,
					TCS_KEY_HANDLE idHandle,	/* in */
					TCPA_NONCE antiReplay,	/* in */
					UINT32 KeySizeIn,	/* in */
					BYTE * KeyDataIn,	/* in */
					TPM_AUTH * pAuth,	/* in, out */
					UINT32 * KeySizeOut,	/* out */
					BYTE ** KeyDataOut,	/* out */
					UINT32 * CertifyInfoSize,	/* out */
					BYTE ** CertifyInfo,	/* out */
					UINT32 * sigSize,	/* out */
					BYTE ** sig,	/* out */
					TCS_KEY_HANDLE * ephHandle	/* out */
    ) {
	return TSPERR(TSS_E_NOTIMPL);
}

TSS_RESULT
TCSP_ChangeAuthAsymFinish_TP(struct host_table_entry *hte,
					 TCS_KEY_HANDLE parentHandle,	/* in */
					 TCS_KEY_HANDLE ephHandle,	/* in */
					 TCPA_ENTITY_TYPE entityType,	/* in */
					 TCPA_HMAC newAuthLink,	/* in */
					 UINT32 newAuthSize,	/* in */
					 BYTE * encNewAuth,	/* in */
					 UINT32 encDataSizeIn,	/* in */
					 BYTE * encDataIn,	/* in */
					 TPM_AUTH * ownerAuth,	/* in, out */
					 UINT32 * encDataSizeOut,	/* out */
					 BYTE ** encDataOut,	/* out */
					 TCPA_SALT_NONCE * saltNonce,	/* out */
					 TCPA_DIGEST * changeProof	/* out */
    ) {
	return TSPERR(TSS_E_NOTIMPL);
}

TSS_RESULT
TCSP_TerminateHandle_TP(struct host_table_entry *hte,
				    TCS_AUTHHANDLE handle	/* in */
    ) {
	TSS_RESULT result;

	initData(&hte->comm, 2);

	hte->comm.hdr.u.ordinal = TCSD_ORD_TERMINATEHANDLE;
	LogDebugFn("TCS Context: 0x%x", hte->tcsContext);

	if (setData(TCSD_PACKET_TYPE_UINT32, 0, &hte->tcsContext, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_UINT32, 1, &handle, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);

	result = sendTCSDPacket(hte);

	if (result == TSS_SUCCESS)
		result = hte->comm.hdr.u.result;

	return result;
}

TSS_RESULT
TCSP_ActivateTPMIdentity_TP(struct host_table_entry *hte,
					TCS_KEY_HANDLE idKey,	/* in */
					UINT32 blobSize,	/* in */
					BYTE * blob,	/* in */
					TPM_AUTH * idKeyAuth,	/* in, out */
					TPM_AUTH * ownerAuth,	/* in, out */
					UINT32 * SymmetricKeySize,	/* out */
					BYTE ** SymmetricKey	/* out */
    ) {
	TSS_RESULT result;
	int i = 0;

	initData(&hte->comm, 6);

	hte->comm.hdr.u.ordinal = TCSD_ORD_ACTIVATETPMIDENTITY;
	LogDebugFn("TCS Context: 0x%x", hte->tcsContext);

	if (setData(TCSD_PACKET_TYPE_UINT32, i++, &hte->tcsContext, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_UINT32, i++, &idKey, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_UINT32, i++, &blobSize, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_PBYTE, i++, blob, blobSize, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);

	if (idKeyAuth) {
		if (setData(TCSD_PACKET_TYPE_AUTH, i++, idKeyAuth, 0, &hte->comm))
			return TSPERR(TSS_E_INTERNAL_ERROR);
	}
	if (setData(TCSD_PACKET_TYPE_AUTH, i++, ownerAuth, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);

	result = sendTCSDPacket(hte);

	if (result == TSS_SUCCESS)
		result = hte->comm.hdr.u.result;

	if (result == TSS_SUCCESS) {
		i = 0;
		if (idKeyAuth) {
			if (getData(TCSD_PACKET_TYPE_AUTH, i++, idKeyAuth, 0, &hte->comm))
				return TSPERR(TSS_E_INTERNAL_ERROR);
		}
		if (getData(TCSD_PACKET_TYPE_AUTH, i++, ownerAuth, 0, &hte->comm))
			return TSPERR(TSS_E_INTERNAL_ERROR);
		if (getData(TCSD_PACKET_TYPE_UINT32, i++, SymmetricKeySize, 0, &hte->comm))
			return TSPERR(TSS_E_INTERNAL_ERROR);

		*SymmetricKey = malloc(*SymmetricKeySize);
		if (*SymmetricKey == NULL) {
			LogError("malloc of %u bytes failed.", *SymmetricKeySize);
			return TSPERR(TSS_E_OUTOFMEMORY);
		}
		if (getData(TCSD_PACKET_TYPE_PBYTE, i++, *SymmetricKey, *SymmetricKeySize, &hte->comm)) {
			free(*SymmetricKey);
			return TSPERR(TSS_E_INTERNAL_ERROR);
		}
	}

	return result;
}

TSS_RESULT
TCSP_Extend_TP(struct host_table_entry *hte,
			   TCPA_PCRINDEX pcrNum,	/* in */
			   TCPA_DIGEST inDigest,	/* in */
			   TCPA_PCRVALUE * outDigest	/* out */
    ) {
	TSS_RESULT result;

	initData(&hte->comm, 3);

	hte->comm.hdr.u.ordinal = TCSD_ORD_EXTEND;
	LogDebugFn("TCS Context: 0x%x", hte->tcsContext);

	if (setData(TCSD_PACKET_TYPE_UINT32, 0, &hte->tcsContext, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_UINT32, 1, &pcrNum, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_DIGEST, 2, &inDigest, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);

	result = sendTCSDPacket(hte);

	if (result == TSS_SUCCESS)
		result = hte->comm.hdr.u.result;

	if (result == TSS_SUCCESS) {
		if (getData(TCSD_PACKET_TYPE_DIGEST, 0, outDigest, 0, &hte->comm))
			return TSPERR(TSS_E_INTERNAL_ERROR);
	}

	return result;
}

TSS_RESULT
TCSP_PcrRead_TP(struct host_table_entry *hte,
			    TCPA_PCRINDEX pcrNum,	/* in */
			    TCPA_PCRVALUE * outDigest	/* out */
    ) {
	TSS_RESULT result;

	initData(&hte->comm, 2);

	hte->comm.hdr.u.ordinal = TCSD_ORD_PCRREAD;
	LogDebugFn("TCS Context: 0x%x", hte->tcsContext);

	if (setData(TCSD_PACKET_TYPE_UINT32, 0, &hte->tcsContext, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_UINT32, 1, &pcrNum, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);

	result = sendTCSDPacket(hte);

	if (result == TSS_SUCCESS)
		result = hte->comm.hdr.u.result;

	if (result == TSS_SUCCESS) {
		if (getData(TCSD_PACKET_TYPE_DIGEST, 0, outDigest, 0, &hte->comm))
			return TSPERR(TSS_E_INTERNAL_ERROR);
	}

	return result;
}

TSS_RESULT
TCSP_Quote_TP(struct host_table_entry *hte,
			  TCS_KEY_HANDLE keyHandle,	/* in */
			  TCPA_NONCE antiReplay,	/* in */
			  UINT32 pcrDataSizeIn,	/* in */
			  BYTE * pcrDataIn,	/* in */
			  TPM_AUTH * privAuth,	/* in, out */
			  UINT32 * pcrDataSizeOut,	/* out */
			  BYTE ** pcrDataOut,	/* out */
			  UINT32 * sigSize,	/* out */
			  BYTE ** sig	/* out */
    ) {
	TSS_RESULT result;
	int i;

	initData(&hte->comm, 6);

	hte->comm.hdr.u.ordinal = TCSD_ORD_QUOTE;
	LogDebugFn("TCS Context: 0x%x", hte->tcsContext);

	if (setData(TCSD_PACKET_TYPE_UINT32, 0, &hte->tcsContext, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_UINT32, 1, &keyHandle, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_NONCE, 2, &antiReplay, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_UINT32, 3, &pcrDataSizeIn, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_PBYTE, 4, pcrDataIn, pcrDataSizeIn, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);

	if (privAuth) {
		if (setData(TCSD_PACKET_TYPE_AUTH, 5, privAuth, 0, &hte->comm))
			return TSPERR(TSS_E_INTERNAL_ERROR);
	}

	result = sendTCSDPacket(hte);

	if (result == TSS_SUCCESS)
		result = hte->comm.hdr.u.result;

	if (result == TSS_SUCCESS) {
		i = 0;
		if (privAuth) {
			if (getData(TCSD_PACKET_TYPE_AUTH, i++, privAuth, 0, &hte->comm))
				return TSPERR(TSS_E_INTERNAL_ERROR);
		}
		if (getData(TCSD_PACKET_TYPE_UINT32, i++, pcrDataSizeOut, 0, &hte->comm))
			return TSPERR(TSS_E_INTERNAL_ERROR);

		*pcrDataOut = (BYTE *) malloc(*pcrDataSizeOut);
		if (*pcrDataOut == NULL) {
			LogError("malloc of %u bytes failed.", *pcrDataSizeOut);
			return TSPERR(TSS_E_OUTOFMEMORY);
		}
		if (getData(TCSD_PACKET_TYPE_PBYTE, i++, *pcrDataOut, *pcrDataSizeOut, &hte->comm)) {
			free(*pcrDataOut);
			return TSPERR(TSS_E_INTERNAL_ERROR);
		}
		if (getData(TCSD_PACKET_TYPE_UINT32, i++, sigSize, 0, &hte->comm)) {
			free(*pcrDataOut);
			return TSPERR(TSS_E_INTERNAL_ERROR);
		}
		*sig = (BYTE *) malloc(*sigSize);
		if (*sig == NULL) {
			LogError("malloc of %u bytes failed.", *sigSize);
			free(*pcrDataOut);
			return TSPERR(TSS_E_OUTOFMEMORY);
		}
		if (getData(TCSD_PACKET_TYPE_PBYTE, i++, *sig, *sigSize, &hte->comm)) {
			free(*pcrDataOut);
			free(*sig);
			return TSPERR(TSS_E_INTERNAL_ERROR);
		}
	}

	return result;
}

TSS_RESULT
TCSP_DirWriteAuth_TP(struct host_table_entry *hte,
				 TCPA_DIRINDEX dirIndex,	/* in */
				 TCPA_DIRVALUE newContents,	/* in */
				 TPM_AUTH * ownerAuth	/* in, out */
    ) {
	TSS_RESULT result;

	initData(&hte->comm, 4);

	hte->comm.hdr.u.ordinal = TCSD_ORD_DIRWRITEAUTH;
	LogDebugFn("TCS Context: 0x%x", hte->tcsContext);

	if (setData(TCSD_PACKET_TYPE_UINT32, 0, &hte->tcsContext, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_UINT32, 1, &dirIndex, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_DIGEST, 2, &newContents, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_AUTH, 3, ownerAuth, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);

	result = sendTCSDPacket(hte);

	if (result == TSS_SUCCESS)
		result = hte->comm.hdr.u.result;

	if (result == TSS_SUCCESS) {
		if (getData(TCSD_PACKET_TYPE_AUTH, 0, ownerAuth, 0, &hte->comm))
			return TSPERR(TSS_E_INTERNAL_ERROR);
	}

	return result;
}

TSS_RESULT
TCSP_DirRead_TP(struct host_table_entry *hte,
			    TCPA_DIRINDEX dirIndex,	/* in */
			    TCPA_DIRVALUE * dirValue	/* out */
    ) {
	TSS_RESULT result;

	initData(&hte->comm, 2);

	hte->comm.hdr.u.ordinal = TCSD_ORD_DIRREAD;
	LogDebugFn("TCS Context: 0x%x", hte->tcsContext);

	if (setData(TCSD_PACKET_TYPE_UINT32, 0, &hte->tcsContext, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_UINT32, 1, &dirIndex, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);

	result = sendTCSDPacket(hte);

	if (result == TSS_SUCCESS)
		result = hte->comm.hdr.u.result;

	if (result == TSS_SUCCESS) {
		if (getData(TCSD_PACKET_TYPE_DIGEST, 0, dirValue, 0, &hte->comm))
			return TSPERR(TSS_E_INTERNAL_ERROR);
	}

	return result;
}

TSS_RESULT
TCSP_Seal_TP(struct host_table_entry *hte,
			 TCS_KEY_HANDLE keyHandle,	/* in */
			 TCPA_ENCAUTH encAuth,	/* in */
			 UINT32 pcrInfoSize,	/* in */
			 BYTE * PcrInfo,	/* in */
			 UINT32 inDataSize,	/* in */
			 BYTE * inData,	/* in */
			 TPM_AUTH * pubAuth,	/* in, out */
			 UINT32 * SealedDataSize,	/* out */
			 BYTE ** SealedData	/* out */
    ) {
	TSS_RESULT result;
	int i = 0;

	initData(&hte->comm, 8);

	hte->comm.hdr.u.ordinal = TCSD_ORD_SEAL;
	LogDebugFn("TCS Context: 0x%x", hte->tcsContext);

	if (setData(TCSD_PACKET_TYPE_UINT32, i++, &hte->tcsContext, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_UINT32, i++, &keyHandle, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_ENCAUTH, i++, &encAuth, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_UINT32, i++, &pcrInfoSize, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (pcrInfoSize > 0) {
		if (setData(TCSD_PACKET_TYPE_PBYTE, i++, PcrInfo, pcrInfoSize, &hte->comm))
			return TSPERR(TSS_E_INTERNAL_ERROR);
	}
	if (setData(TCSD_PACKET_TYPE_UINT32, i++, &inDataSize, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (inDataSize > 0) {
		if (setData(TCSD_PACKET_TYPE_PBYTE, i++, inData, inDataSize, &hte->comm))
			return TSPERR(TSS_E_INTERNAL_ERROR);
	}

	if (setData(TCSD_PACKET_TYPE_AUTH, i, pubAuth, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);

	result = sendTCSDPacket(hte);

	if (result == TSS_SUCCESS)
		result = hte->comm.hdr.u.result;

	if (result == TSS_SUCCESS) {
		if (getData(TCSD_PACKET_TYPE_AUTH, 0, pubAuth, 0, &hte->comm))
			return TSPERR(TSS_E_INTERNAL_ERROR);

		if (getData(TCSD_PACKET_TYPE_UINT32, 1, SealedDataSize, 0, &hte->comm))
			return TSPERR(TSS_E_INTERNAL_ERROR);

		*SealedData = (BYTE *) malloc(*SealedDataSize);
		if (*SealedData == NULL) {
			LogError("malloc of %u bytes failed.", *SealedDataSize);
			return TSPERR(TSS_E_OUTOFMEMORY);
		}
		if (getData(TCSD_PACKET_TYPE_PBYTE, 2, *SealedData, *SealedDataSize, &hte->comm)) {
			free(*SealedData);
			return TSPERR(TSS_E_INTERNAL_ERROR);
		}
	}

	return result;
}

TSS_RESULT
TCSP_Unseal_TP(struct host_table_entry *hte,
			   TCS_KEY_HANDLE parentHandle,	/* in */
			   UINT32 SealedDataSize,	/* in */
			   BYTE * SealedData,	/* in */
			   TPM_AUTH * parentAuth,	/* in, out */
			   TPM_AUTH * dataAuth,	/* in, out */
			   UINT32 * DataSize,	/* out */
			   BYTE ** Data	/* out */
    ) {
	TSS_RESULT result;

	initData(&hte->comm, 6);

	hte->comm.hdr.u.ordinal = TCSD_ORD_UNSEAL;
	LogDebugFn("TCS Context: 0x%x", hte->tcsContext);

	if (setData(TCSD_PACKET_TYPE_UINT32, 0, &hte->tcsContext, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_UINT32, 1, &parentHandle, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_UINT32, 2, &SealedDataSize, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_PBYTE, 3, SealedData, SealedDataSize, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);

	if (parentAuth != NULL) {
		if (setData(TCSD_PACKET_TYPE_AUTH, 4, parentAuth, 0, &hte->comm))
			return TSPERR(TSS_E_INTERNAL_ERROR);
	}

	if (setData(TCSD_PACKET_TYPE_AUTH, 5, dataAuth, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);

	result = sendTCSDPacket(hte);

	if (result == TSS_SUCCESS)
		result = hte->comm.hdr.u.result;

	if (result == TSS_SUCCESS) {
		if (parentAuth != NULL) {
			if (getData(TCSD_PACKET_TYPE_AUTH, 0, parentAuth, 0, &hte->comm))
				return TSPERR(TSS_E_INTERNAL_ERROR);
		}

		if (getData(TCSD_PACKET_TYPE_AUTH, 1, dataAuth, 0, &hte->comm))
			return TSPERR(TSS_E_INTERNAL_ERROR);

		if (getData(TCSD_PACKET_TYPE_UINT32, 2, DataSize, 0, &hte->comm))
			return TSPERR(TSS_E_INTERNAL_ERROR);

		*Data = (BYTE *) calloc_tspi(hte->tspContext, *DataSize);
		if (*Data == NULL) {
			LogError("malloc of %u bytes failed.", *DataSize);
			return TSPERR(TSS_E_OUTOFMEMORY);
		}
		if (getData(TCSD_PACKET_TYPE_PBYTE, 3, *Data, *DataSize, &hte->comm)) {
			free_tspi(hte->tspContext, *Data);
			return TSPERR(TSS_E_INTERNAL_ERROR);
		}
	}

	return result;
}

TSS_RESULT
TCSP_UnBind_TP(struct host_table_entry *hte,
			   TCS_KEY_HANDLE keyHandle,	/* in */
			   UINT32 inDataSize,	/* in */
			   BYTE * inData,	/* in */
			   TPM_AUTH * privAuth,	/* in, out */
			   UINT32 * outDataSize,	/* out */
			   BYTE ** outData	/* out */
    ) {
	TSS_RESULT result;
	int i;

	initData(&hte->comm, 5);

	hte->comm.hdr.u.ordinal = TCSD_ORD_UNBIND;
	LogDebugFn("TCS Context: 0x%x", hte->tcsContext);

	if (setData(TCSD_PACKET_TYPE_UINT32, 0, &hte->tcsContext, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_UINT32, 1, &keyHandle, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_UINT32, 2, &inDataSize, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_PBYTE, 3, inData, inDataSize, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);

	if (privAuth != NULL) {
		if (setData(TCSD_PACKET_TYPE_AUTH, 4, privAuth, 0, &hte->comm))
			return TSPERR(TSS_E_INTERNAL_ERROR);
	}

	result = sendTCSDPacket(hte);

	if (result == TSS_SUCCESS)
		result = hte->comm.hdr.u.result;

	if (result == TSS_SUCCESS) {
		i = 0;
		if (privAuth != NULL) {
			if (getData(TCSD_PACKET_TYPE_AUTH, i++, privAuth, 0, &hte->comm))
				return TSPERR(TSS_E_INTERNAL_ERROR);
		}
		if (getData(TCSD_PACKET_TYPE_UINT32, i++, outDataSize, 0, &hte->comm))
			return TSPERR(TSS_E_INTERNAL_ERROR);

		*outData = (BYTE *) calloc_tspi(hte->tspContext, *outDataSize);
		if (*outData == NULL) {
			LogError("malloc of %u bytes failed.", *outDataSize);
			return TSPERR(TSS_E_OUTOFMEMORY);
		}
		if (getData(TCSD_PACKET_TYPE_PBYTE, i++, *outData, *outDataSize, &hte->comm)) {
			free_tspi(hte->tspContext, *outData);
			return TSPERR(TSS_E_INTERNAL_ERROR);
		}
	}

	return result;
}

TSS_RESULT
TCSP_CreateMigrationBlob_TP(struct host_table_entry *hte,
					TCS_KEY_HANDLE parentHandle,	/* in */
					TSS_MIGRATE_SCHEME migrationType,	/* in */
					UINT32 MigrationKeyAuthSize,	/* in */
					BYTE * MigrationKeyAuth,	/* in */
					UINT32 encDataSize,	/* in */
					BYTE * encData,	/* in */
					TPM_AUTH * parentAuth,	/* in, out */
					TPM_AUTH * entityAuth,	/* in, out */
					UINT32 * randomSize,	/* out */
					BYTE ** random,	/* out */
					UINT32 * outDataSize,	/* out */
					BYTE ** outData	/* out */
    ) {
	TSS_RESULT result;
	TPM_AUTH null_auth;
	UINT32 i;

	initData(&hte->comm, 9);

	hte->comm.hdr.u.ordinal = TCSD_ORD_CREATEMIGRATIONBLOB;
	LogDebugFn("TCS Context: 0x%x", hte->tcsContext);

	memset(&null_auth, 0, sizeof(TPM_AUTH));
	i = 0;
	if (setData(TCSD_PACKET_TYPE_UINT32, i++, &hte->tcsContext, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_UINT32, i++, &parentHandle, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_UINT16, i++, &migrationType, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_UINT32, i++, &MigrationKeyAuthSize, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_PBYTE, i++, MigrationKeyAuth, MigrationKeyAuthSize, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_UINT32, i++, &encDataSize, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_PBYTE, i++, encData, encDataSize, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);

	if (parentAuth) {
		if (setData(TCSD_PACKET_TYPE_AUTH, i++, parentAuth, 0, &hte->comm))
			return TSPERR(TSS_E_INTERNAL_ERROR);
	}

	if (setData(TCSD_PACKET_TYPE_AUTH, i++, entityAuth, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);

	result = sendTCSDPacket(hte);

	if (result == TSS_SUCCESS)
		result = hte->comm.hdr.u.result;

	if (result == TSS_SUCCESS) {
		i = 0;
		if (parentAuth) {
			if (getData(TCSD_PACKET_TYPE_AUTH, i++, parentAuth, 0, &hte->comm))
				return TSPERR(TSS_E_INTERNAL_ERROR);
		}
		if (getData(TCSD_PACKET_TYPE_AUTH, i++, entityAuth, 0, &hte->comm))
			return TSPERR(TSS_E_INTERNAL_ERROR);

		if (getData(TCSD_PACKET_TYPE_UINT32, i++, randomSize, 0, &hte->comm))
			return TSPERR(TSS_E_INTERNAL_ERROR);

		if (*randomSize > 0) {
			*random = (BYTE *)calloc_tspi(hte->tspContext, *randomSize);
			if (*random == NULL) {
				LogError("malloc of %u bytes failed.", *randomSize);
				return TSPERR(TSS_E_OUTOFMEMORY);
			}
			if (getData(TCSD_PACKET_TYPE_PBYTE, i++, *random, *randomSize, &hte->comm)) {
				free_tspi(hte->tspContext, *random);
				return TSPERR(TSS_E_INTERNAL_ERROR);
			}
		}

		if (getData(TCSD_PACKET_TYPE_UINT32, i++, outDataSize, 0, &hte->comm)) {
			if (*randomSize > 0)
				free_tspi(hte->tspContext, *random);
			return TSPERR(TSS_E_INTERNAL_ERROR);
		}

		*outData = (BYTE *)calloc_tspi(hte->tspContext, *outDataSize);
		if (*outData == NULL) {
			if (*randomSize > 0)
				free_tspi(hte->tspContext, *random);
			LogError("malloc of %u bytes failed.", *outDataSize);
			return TSPERR(TSS_E_OUTOFMEMORY);
		}
		if (getData(TCSD_PACKET_TYPE_PBYTE, i++, *outData, *outDataSize, &hte->comm)) {
			if (*randomSize > 0)
				free_tspi(hte->tspContext, *random);
			free_tspi(hte->tspContext, *outData);
			return TSPERR(TSS_E_INTERNAL_ERROR);
		}
	}

	return result;
}

TSS_RESULT
TCSP_ConvertMigrationBlob_TP(struct host_table_entry *hte,
					 TCS_KEY_HANDLE parentHandle,	/* in */
					 UINT32 inDataSize,	/* in */
					 BYTE * inData,	/* in */
					 UINT32 randomSize,	/* in */
					 BYTE * random,	/* in */
					 TPM_AUTH * parentAuth,	/* in, out */
					 UINT32 * outDataSize,	/* out */
					 BYTE ** outData	/* out */
    ) {
	TSS_RESULT result;
	UINT32 i;

	initData(&hte->comm, 7);

	hte->comm.hdr.u.ordinal = TCSD_ORD_CONVERTMIGRATIONBLOB;
	LogDebugFn("TCS Context: 0x%x", hte->tcsContext);

	if (setData(TCSD_PACKET_TYPE_UINT32, 0, &hte->tcsContext, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_UINT32, 1, &parentHandle, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_UINT32, 2, &inDataSize, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_PBYTE, 3, inData, inDataSize, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_UINT32, 4, &randomSize, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_PBYTE, 5, random, randomSize, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);

	if (parentAuth) {
		if (setData(TCSD_PACKET_TYPE_AUTH, 6, parentAuth, 0, &hte->comm))
			return TSPERR(TSS_E_INTERNAL_ERROR);
	}

	result = sendTCSDPacket(hte);

	if (result == TSS_SUCCESS)
		result = hte->comm.hdr.u.result;

	if (result == TSS_SUCCESS) {
		i = 0;
		if (parentAuth) {
			if (getData(TCSD_PACKET_TYPE_AUTH, i++, parentAuth, 0, &hte->comm))
				return TSPERR(TSS_E_INTERNAL_ERROR);
		}

		if (getData(TCSD_PACKET_TYPE_UINT32, i++, outDataSize, 0, &hte->comm))
			return TSPERR(TSS_E_INTERNAL_ERROR);

		*outData = (BYTE *)malloc(*outDataSize);
		if (*outData == NULL) {
			LogError("malloc of %u bytes failed.", *outDataSize);
			return TSPERR(TSS_E_OUTOFMEMORY);
		}
		if (getData(TCSD_PACKET_TYPE_PBYTE, i++, *outData, *outDataSize, &hte->comm)) {
			free(*outData);
			return TSPERR(TSS_E_INTERNAL_ERROR);
		}
	}

	return result;
}

TSS_RESULT
TCSP_AuthorizeMigrationKey_TP(struct host_table_entry *hte,
					  TSS_MIGRATE_SCHEME migrateScheme,	/* in */
					  UINT32 MigrationKeySize,	/* in */
					  BYTE * MigrationKey,	/* in */
					  TPM_AUTH * ownerAuth,	/* in, out */
					  UINT32 * MigrationKeyAuthSize,	/* out */
					  BYTE ** MigrationKeyAuth	/* out */
    ) {
	TSS_RESULT result;

	initData(&hte->comm, 5);

	hte->comm.hdr.u.ordinal = TCSD_ORD_AUTHORIZEMIGRATIONKEY;
	LogDebugFn("TCS Context: 0x%x", hte->tcsContext);

	if (setData(TCSD_PACKET_TYPE_UINT32, 0, &hte->tcsContext, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_UINT16, 1, &migrateScheme, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_UINT32, 2, &MigrationKeySize, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_PBYTE, 3, MigrationKey, MigrationKeySize, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_AUTH, 4, ownerAuth, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);

	result = sendTCSDPacket(hte);

	if (result == TSS_SUCCESS)
		result = hte->comm.hdr.u.result;

	if (result == TSS_SUCCESS) {
		if (getData(TCSD_PACKET_TYPE_AUTH, 0, ownerAuth, 0, &hte->comm))
			return TSPERR(TSS_E_INTERNAL_ERROR);
		if (getData(TCSD_PACKET_TYPE_UINT32, 1, MigrationKeyAuthSize, 0, &hte->comm))
			return TSPERR(TSS_E_INTERNAL_ERROR);

		*MigrationKeyAuth = (BYTE *)calloc_tspi(hte->tspContext, *MigrationKeyAuthSize);
		if (*MigrationKeyAuth == NULL) {
			LogError("malloc of %u bytes failed.", *MigrationKeyAuthSize);
			return TSPERR(TSS_E_OUTOFMEMORY);
		}
		if (getData(TCSD_PACKET_TYPE_PBYTE, 2, *MigrationKeyAuth, *MigrationKeyAuthSize,
			    &hte->comm)) {
			free(*MigrationKeyAuth);
			return TSPERR(TSS_E_INTERNAL_ERROR);
		}
	}

	return result;
}

TSS_RESULT
TCSP_CertifyKey_TP(struct host_table_entry *hte,
			       TCS_KEY_HANDLE certHandle,	/* in */
			       TCS_KEY_HANDLE keyHandle,	/* in */
			       TCPA_NONCE antiReplay,	/* in */
			       TPM_AUTH * certAuth,	/* in, out */
			       TPM_AUTH * keyAuth,	/* in, out */
			       UINT32 * CertifyInfoSize,	/* out */
			       BYTE ** CertifyInfo,	/* out */
			       UINT32 * outDataSize,	/* out */
			       BYTE ** outData	/* out */
    ) {
	TSS_RESULT result;
	TPM_AUTH null_auth;
	int i;

	initData(&hte->comm, 6);

	hte->comm.hdr.u.ordinal = TCSD_ORD_CERTIFYKEY;
	LogDebugFn("TCS Context: 0x%x", hte->tcsContext);

	memset(&null_auth, 0, sizeof(TPM_AUTH));
	if (setData(TCSD_PACKET_TYPE_UINT32, 0, &hte->tcsContext, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_UINT32, 1, &certHandle, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_UINT32, 2, &keyHandle, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_NONCE, 3, &antiReplay, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (certAuth) {
		if (setData(TCSD_PACKET_TYPE_AUTH, 4, certAuth, 0, &hte->comm))
			return TSPERR(TSS_E_INTERNAL_ERROR);
	} else {
		if (setData(TCSD_PACKET_TYPE_AUTH, 4, &null_auth, 0, &hte->comm))
			return TSPERR(TSS_E_INTERNAL_ERROR);
	}
	if (keyAuth) {
		if (setData(TCSD_PACKET_TYPE_AUTH, 5, keyAuth, 0, &hte->comm))
			return TSPERR(TSS_E_INTERNAL_ERROR);
	} else {
		if (setData(TCSD_PACKET_TYPE_AUTH, 5, &null_auth, 0, &hte->comm))
			return TSPERR(TSS_E_INTERNAL_ERROR);
	}

	result = sendTCSDPacket(hte);

	if (result == TSS_SUCCESS)
		result = hte->comm.hdr.u.result;

	if (result == TSS_SUCCESS) {
		i = 0;
		if (certAuth) {
			if (getData(TCSD_PACKET_TYPE_AUTH, i++, certAuth, 0, &hte->comm))
				return TSPERR(TSS_E_INTERNAL_ERROR);
		}
		if (keyAuth) {
			if (getData(TCSD_PACKET_TYPE_AUTH, i++, keyAuth, 0, &hte->comm))
				return TSPERR(TSS_E_INTERNAL_ERROR);
		}
		if (getData(TCSD_PACKET_TYPE_UINT32, i++, CertifyInfoSize, 0, &hte->comm))
			return TSPERR(TSS_E_INTERNAL_ERROR);

		*CertifyInfo = (BYTE *) malloc(*CertifyInfoSize);
		if (*CertifyInfo == NULL) {
			LogError("malloc of %u bytes failed.", *CertifyInfoSize);
			return TSPERR(TSS_E_OUTOFMEMORY);
		}
		if (getData(TCSD_PACKET_TYPE_PBYTE, i++, *CertifyInfo, *CertifyInfoSize, &hte->comm)) {
			free(*CertifyInfo);
			return TSPERR(TSS_E_INTERNAL_ERROR);
		}
		if (getData(TCSD_PACKET_TYPE_UINT32, i++, outDataSize, 0, &hte->comm)) {
			free(*CertifyInfo);
			return TSPERR(TSS_E_INTERNAL_ERROR);
		}

		*outData = (BYTE *) malloc(*outDataSize);
		if (*outData == NULL) {
			LogError("malloc of %u bytes failed.", *outDataSize);
			free(*CertifyInfo);
			return TSPERR(TSS_E_OUTOFMEMORY);
		}
		if (getData(TCSD_PACKET_TYPE_PBYTE, i++, *outData, *outDataSize, &hte->comm)) {
			free(*CertifyInfo);
			free(*outData);
			return TSPERR(TSS_E_INTERNAL_ERROR);
		}
	}

	return result;
}

TSS_RESULT
TCSP_Sign_TP(struct host_table_entry *hte,
			 TCS_KEY_HANDLE keyHandle,	/* in */
			 UINT32 areaToSignSize,	/* in */
			 BYTE * areaToSign,	/* in */
			 TPM_AUTH * privAuth,	/* in, out */
			 UINT32 * sigSize,	/* out */
			 BYTE ** sig	/* out */
    ) {
	TSS_RESULT result;
	int i;

	initData(&hte->comm, 5);

	hte->comm.hdr.u.ordinal = TCSD_ORD_SIGN;
	LogDebugFn("TCS Context: 0x%x", hte->tcsContext);

	if (setData(TCSD_PACKET_TYPE_UINT32, 0, &hte->tcsContext, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_UINT32, 1, &keyHandle, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_UINT32, 2, &areaToSignSize, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_PBYTE, 3, areaToSign, areaToSignSize, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);

	if (privAuth) {
		if (setData(TCSD_PACKET_TYPE_AUTH, 4, privAuth, 0, &hte->comm))
			return TSPERR(TSS_E_INTERNAL_ERROR);
	}

	result = sendTCSDPacket(hte);

	if (result == TSS_SUCCESS)
		result = hte->comm.hdr.u.result;

	if (result == TSS_SUCCESS) {
		i = 0;
		if (privAuth) {
			if (getData(TCSD_PACKET_TYPE_AUTH, i++, privAuth, 0, &hte->comm))
				return TSPERR(TSS_E_INTERNAL_ERROR);
		}
		if (getData(TCSD_PACKET_TYPE_UINT32, i++, sigSize, 0, &hte->comm))
			return TSPERR(TSS_E_INTERNAL_ERROR);

		*sig = (BYTE *) calloc_tspi(hte->tspContext, *sigSize);
		if (*sig == NULL) {
			LogError("malloc of %u bytes failed.", *sigSize);
			return TSPERR(TSS_E_OUTOFMEMORY);
		}
		if (getData(TCSD_PACKET_TYPE_PBYTE, i++, *sig, *sigSize, &hte->comm)) {
			free_tspi(hte->tspContext, *sig);
			return TSPERR(TSS_E_INTERNAL_ERROR);
		}
	}

	return result;
}

TSS_RESULT
TCSP_GetRandom_TP(struct host_table_entry *hte,
			      UINT32 bytesRequested,	/* in */
			      BYTE ** randomBytes	/* out */
    ) {
	TSS_RESULT result;

	initData(&hte->comm, 2);

	hte->comm.hdr.u.ordinal = TCSD_ORD_GETRANDOM;
	LogDebugFn("TCS Context: 0x%x", hte->tcsContext);

	if (setData(TCSD_PACKET_TYPE_UINT32, 0, &hte->tcsContext, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_UINT32, 1, &bytesRequested, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);

	result = sendTCSDPacket(hte);

	if (result == TSS_SUCCESS)
		result = hte->comm.hdr.u.result;

	if (result == TSS_SUCCESS) {
		if (getData(TCSD_PACKET_TYPE_UINT32, 0, &bytesRequested, 0, &hte->comm))
			return TSPERR(TSS_E_INTERNAL_ERROR);
		*randomBytes = (BYTE *) calloc_tspi(hte->tspContext, bytesRequested);
		if (*randomBytes == NULL) {
			LogError("malloc of %u bytes failed.", bytesRequested);
			return TSPERR(TSS_E_OUTOFMEMORY);
		}
		if (getData(TCSD_PACKET_TYPE_PBYTE, 1, *randomBytes, bytesRequested, &hte->comm)) {
			free_tspi(hte->tspContext, *randomBytes);
			return TSPERR(TSS_E_INTERNAL_ERROR);
		}
	}

	return result;
}

TSS_RESULT
TCSP_StirRandom_TP(struct host_table_entry *hte,
			       UINT32 inDataSize,	/* in */
			       BYTE * inData	/* in */
    ) {
	TSS_RESULT result;

	initData(&hte->comm, 3);

	hte->comm.hdr.u.ordinal = TCSD_ORD_STIRRANDOM;
	LogDebugFn("TCS Context: 0x%x", hte->tcsContext);

	if (setData(TCSD_PACKET_TYPE_UINT32, 0, &hte->tcsContext, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_UINT32, 1, &inDataSize, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_PBYTE, 2, inData, inDataSize, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);

	result = sendTCSDPacket(hte);

	if (result == TSS_SUCCESS)
		result = hte->comm.hdr.u.result;

	return result;
}

TSS_RESULT
TCSP_GetCapability_TP(struct host_table_entry *hte,
				  TCPA_CAPABILITY_AREA capArea,	/* in */
				  UINT32 subCapSize,	/* in */
				  BYTE * subCap,	/* in */
				  UINT32 * respSize,	/* out */
				  BYTE ** resp	/* out */
    ) {
	TSS_RESULT result;

	initData(&hte->comm, 4);

	hte->comm.hdr.u.ordinal = TCSD_ORD_GETCAPABILITY;
	LogDebugFn("TCS Context: 0x%x", hte->tcsContext);

	if (setData(TCSD_PACKET_TYPE_UINT32, 0, &hte->tcsContext, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_UINT32, 1, &capArea, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_UINT32, 2, &subCapSize, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_PBYTE, 3, subCap, subCapSize, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);

	result = sendTCSDPacket(hte);

	if (result == TSS_SUCCESS)
		result = hte->comm.hdr.u.result;

	if (result == TSS_SUCCESS) {
		if (getData(TCSD_PACKET_TYPE_UINT32, 0, respSize, 0, &hte->comm))
			return TSPERR(TSS_E_INTERNAL_ERROR);

		*resp = (BYTE *) malloc(*respSize);
		if (*resp == NULL) {
			LogError("malloc of %u bytes failed.", *respSize);
			return TSPERR(TSS_E_OUTOFMEMORY);
		}
		if (getData(TCSD_PACKET_TYPE_PBYTE, 1, *resp, *respSize, &hte->comm)) {
			free(*resp);
			return TSPERR(TSS_E_INTERNAL_ERROR);
		}
	}

	return result;
}

TSS_RESULT
TCS_GetCapability_TP(struct host_table_entry *hte,
				 TCPA_CAPABILITY_AREA capArea,	/* in */
				 UINT32 subCapSize,	/* in */
				 BYTE * subCap,	/* in */
				 UINT32 * respSize,	/* out */
				 BYTE ** resp	/* out */
    ) {
	TSS_RESULT result;

	initData(&hte->comm, 4);

	hte->comm.hdr.u.ordinal = TCSD_ORD_TCSGETCAPABILITY;
	LogDebugFn("TCS Context: 0x%x", hte->tcsContext);

	if (setData(TCSD_PACKET_TYPE_UINT32, 0, &hte->tcsContext, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_UINT32, 1, &capArea, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_UINT32, 2, &subCapSize, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_PBYTE, 3, subCap, subCapSize, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);

	result = sendTCSDPacket(hte);

	if (result == TSS_SUCCESS)
		result = hte->comm.hdr.u.result;

	if (result == TSS_SUCCESS) {
		if (getData(TCSD_PACKET_TYPE_UINT32, 0, respSize, 0, &hte->comm))
			return TSPERR(TSS_E_INTERNAL_ERROR);

		*resp = (BYTE *) calloc_tspi(hte->tspContext, *respSize);
		if (*resp == NULL) {
			LogError("malloc of %u bytes failed.", *respSize);
			return TSPERR(TSS_E_OUTOFMEMORY);
		}
		if (getData(TCSD_PACKET_TYPE_PBYTE, 1, *resp, *respSize, &hte->comm)) {
			free_tspi(hte->tspContext, *resp);
			return TSPERR(TSS_E_INTERNAL_ERROR);
		}
	}

	return result;
}

TSS_RESULT
TCSP_GetCapabilitySigned_TP(struct host_table_entry *hte,
					TCS_KEY_HANDLE keyHandle,	/* in */
					TCPA_NONCE antiReplay,	/* in */
					TCPA_CAPABILITY_AREA capArea,	/* in */
					UINT32 subCapSize,	/* in */
					BYTE * subCap,	/* in */
					TPM_AUTH * privAuth,	/* in, out */
					TCPA_VERSION * Version,	/* out */
					UINT32 * respSize,	/* out */
					BYTE ** resp,	/* out */
					UINT32 * sigSize,	/* out */
					BYTE ** sig	/* out */
    ) {
	return TSPERR(TSS_E_NOTIMPL);
}

TSS_RESULT
TCSP_GetCapabilityOwner_TP(struct host_table_entry *hte,
				       TPM_AUTH * pOwnerAuth,	/* out */
				       TCPA_VERSION * pVersion,	/* out */
				       UINT32 * pNonVolatileFlags,	/* out */
				       UINT32 * pVolatileFlags	/* out */
    ) {
	TSS_RESULT result;

	initData(&hte->comm, 2);

	hte->comm.hdr.u.ordinal = TCSD_ORD_GETCAPABILITYOWNER;
	LogDebugFn("TCS Context: 0x%x", hte->tcsContext);

	if (setData(TCSD_PACKET_TYPE_UINT32, 0, &hte->tcsContext, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_AUTH, 1, pOwnerAuth, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);

	result = sendTCSDPacket(hte);

	if (result == TSS_SUCCESS)
		result = hte->comm.hdr.u.result;

	if (result == TSS_SUCCESS) {
		if (getData(TCSD_PACKET_TYPE_VERSION, 0, pVersion, 0, &hte->comm))
			return TSPERR(TSS_E_INTERNAL_ERROR);
		if (getData(TCSD_PACKET_TYPE_UINT32, 1, pNonVolatileFlags, 0, &hte->comm))
			return TSPERR(TSS_E_INTERNAL_ERROR);
		if (getData(TCSD_PACKET_TYPE_UINT32, 2, pVolatileFlags, 0, &hte->comm))
			return TSPERR(TSS_E_INTERNAL_ERROR);
		if (getData(TCSD_PACKET_TYPE_AUTH, 3, pOwnerAuth, 0, &hte->comm))
			return TSPERR(TSS_E_INTERNAL_ERROR);
	}

	return result;
}

TSS_RESULT
TCSP_CreateEndorsementKeyPair_TP(struct host_table_entry *hte,
					     TCPA_NONCE antiReplay,	/* in */
					     UINT32 endorsementKeyInfoSize,	/* in */
					     BYTE * endorsementKeyInfo,	/* in */
					     UINT32 * endorsementKeySize,	/* out */
					     BYTE ** endorsementKey,	/* out */
					     TCPA_DIGEST * checksum	/* out */
    ) {

	TSS_RESULT result;

	initData(&hte->comm, 4);

	hte->comm.hdr.u.ordinal = TCSD_ORD_CREATEENDORSEMENTKEYPAIR;
	LogDebugFn("TCS Context: 0x%x", hte->tcsContext);

	if (setData(TCSD_PACKET_TYPE_UINT32, 0, &hte->tcsContext, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_NONCE, 1, &antiReplay, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_UINT32, 2, &endorsementKeyInfoSize, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_PBYTE, 3, endorsementKeyInfo, endorsementKeyInfoSize, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);

	result = sendTCSDPacket(hte);

	if (result == TSS_SUCCESS)
		result = hte->comm.hdr.u.result;

	if (result == TSS_SUCCESS) {
		if (getData(TCSD_PACKET_TYPE_UINT32, 0, endorsementKeySize, 0, &hte->comm))
			return TSPERR(TSS_E_INTERNAL_ERROR);

		*endorsementKey = (BYTE *) malloc(*endorsementKeySize);
		if (*endorsementKey == NULL) {
			LogError("malloc of %u bytes failed.", *endorsementKeySize);
			return TSPERR(TSS_E_OUTOFMEMORY);
		}
		if (getData(TCSD_PACKET_TYPE_PBYTE, 1, *endorsementKey, *endorsementKeySize, &hte->comm)) {
			free(*endorsementKey);
			return TSPERR(TSS_E_INTERNAL_ERROR);
		}
		if (getData(TCSD_PACKET_TYPE_DIGEST, 2, &(checksum->digest), 0, &hte->comm)) {
			free(*endorsementKey);
			return TSPERR(TSS_E_INTERNAL_ERROR);
		}
	}

	return result;
}

TSS_RESULT
TCSP_ReadPubek_TP(struct host_table_entry *hte,
			      TCPA_NONCE antiReplay,	/* in */
			      UINT32 * pubEndorsementKeySize,	/* out */
			      BYTE ** pubEndorsementKey,	/* out */
			      TCPA_DIGEST * checksum	/* out */
    ) {
	TSS_RESULT result;

	initData(&hte->comm, 2);

	hte->comm.hdr.u.ordinal = TCSD_ORD_READPUBEK;
	LogDebugFn("TCS Context: 0x%x", hte->tcsContext);

	if (setData(TCSD_PACKET_TYPE_UINT32, 0, &hte->tcsContext, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_NONCE, 1, &antiReplay, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);

	result = sendTCSDPacket(hte);

	if (result == TSS_SUCCESS)
		result = hte->comm.hdr.u.result;

	if (result == TSS_SUCCESS) {
		if (getData(TCSD_PACKET_TYPE_UINT32, 0, pubEndorsementKeySize, 0, &hte->comm))
			return TSPERR(TSS_E_INTERNAL_ERROR);

		*pubEndorsementKey = (BYTE *) malloc(*pubEndorsementKeySize);
		if (*pubEndorsementKey == NULL) {
			LogError("malloc of %u bytes failed.", *pubEndorsementKeySize);
			return TSPERR(TSS_E_OUTOFMEMORY);
		}
		if (getData(TCSD_PACKET_TYPE_PBYTE, 1, *pubEndorsementKey, *pubEndorsementKeySize, &hte->comm)) {
			free(*pubEndorsementKey);
			return TSPERR(TSS_E_INTERNAL_ERROR);
		}
		if (getData(TCSD_PACKET_TYPE_DIGEST, 2, &(checksum->digest), 0, &hte->comm)) {
			free(*pubEndorsementKey);
			return TSPERR(TSS_E_INTERNAL_ERROR);
		}
	}

	return result;
}

TSS_RESULT
TCSP_DisablePubekRead_TP(struct host_table_entry *hte,
				     TPM_AUTH * ownerAuth	/* in, out */
    ) {
	TSS_RESULT result;

	initData(&hte->comm, 2);

	hte->comm.hdr.u.ordinal = TCSD_ORD_DISABLEPUBEKREAD;
	LogDebugFn("TCS Context: 0x%x", hte->tcsContext);

        if (setData(TCSD_PACKET_TYPE_UINT32, 0, &hte->tcsContext, 0, &hte->comm))
                return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_AUTH, 1, ownerAuth, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);

        result = sendTCSDPacket(hte);

        if (result == TSS_SUCCESS)
                result = hte->comm.hdr.u.result;

	if (result == TSS_SUCCESS) {
		if (getData(TCSD_PACKET_TYPE_AUTH, 0, ownerAuth, 0, &hte->comm))
			return TSPERR(TSS_E_INTERNAL_ERROR);
	}

	return result;
}

TSS_RESULT
TCSP_OwnerReadPubek_TP(struct host_table_entry *hte,
				   TPM_AUTH * ownerAuth,	/* in, out */
				   UINT32 * pubEndorsementKeySize,	/* out */
				   BYTE ** pubEndorsementKey	/* out */
    ) {

        TSS_RESULT result;

	initData(&hte->comm, 2);

	hte->comm.hdr.u.ordinal = TCSD_ORD_OWNERREADPUBEK;
	LogDebugFn("TCS Context: 0x%x", hte->tcsContext);

        if (setData(TCSD_PACKET_TYPE_UINT32, 0, &hte->tcsContext, 0, &hte->comm))
                return TSPERR(TSS_E_INTERNAL_ERROR);
        if (setData(TCSD_PACKET_TYPE_AUTH, 1, ownerAuth, 0, &hte->comm))
                return TSPERR(TSS_E_INTERNAL_ERROR);

        result = sendTCSDPacket(hte);

        if (result == TSS_SUCCESS)
                result = hte->comm.hdr.u.result;

        if (result == TSS_SUCCESS) {
                if (getData(TCSD_PACKET_TYPE_AUTH, 0, ownerAuth, 0, &hte->comm))
                        return TSPERR(TSS_E_INTERNAL_ERROR);

                if (getData(TCSD_PACKET_TYPE_UINT32, 1, pubEndorsementKeySize, 0, &hte->comm))
                        return TSPERR(TSS_E_INTERNAL_ERROR);

                *pubEndorsementKey = (BYTE *) malloc(*pubEndorsementKeySize);
                if (*pubEndorsementKey == NULL) {
                        LogError("malloc of %u bytes failed.", *pubEndorsementKeySize);
                        return TSPERR(TSS_E_OUTOFMEMORY);
                }

                if (getData(TCSD_PACKET_TYPE_PBYTE, 2, *pubEndorsementKey, *pubEndorsementKeySize, &hte->comm)) {
                        free(*pubEndorsementKey);
                        return TSPERR(TSS_E_INTERNAL_ERROR);
                }
        }

	return result;
}

TSS_RESULT
TCSP_SelfTestFull_TP(struct host_table_entry *hte)
{
	TSS_RESULT result;

	initData(&hte->comm, 1);

	hte->comm.hdr.u.ordinal = TCSD_ORD_SELFTESTFULL;
	LogDebugFn("TCS Context: 0x%x", hte->tcsContext);

	if (setData(TCSD_PACKET_TYPE_UINT32, 0, &hte->tcsContext, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);

	result = sendTCSDPacket(hte);

	if (result == TSS_SUCCESS)
		result = hte->comm.hdr.u.result;

	return result;
}

TSS_RESULT
TCSP_CertifySelfTest_TP(struct host_table_entry *hte,
				    TCS_KEY_HANDLE keyHandle,	/* in */
				    TCPA_NONCE antiReplay,	/* in */
				    TPM_AUTH * privAuth,	/* in, out */
				    UINT32 * sigSize,	/* out */
				    BYTE ** sig	/* out */
    ) {
	TSS_RESULT result;
	int i;

	initData(&hte->comm, 4);

	hte->comm.hdr.u.ordinal = TCSD_ORD_CERTIFYSELFTEST;
	LogDebugFn("TCS Context: 0x%x", hte->tcsContext);

	if (setData(TCSD_PACKET_TYPE_UINT32, 0, &hte->tcsContext, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_UINT32, 1, &keyHandle, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_NONCE, 2, &antiReplay, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);

	if (privAuth) {
		if (setData(TCSD_PACKET_TYPE_AUTH, 3, privAuth, 0, &hte->comm))
			return TSPERR(TSS_E_INTERNAL_ERROR);
	}

	result = sendTCSDPacket(hte);

	if (result == TSS_SUCCESS)
		result = hte->comm.hdr.u.result;

	if (result == TSS_SUCCESS) {
		i = 0;
		if (privAuth) {
			if (getData(TCSD_PACKET_TYPE_AUTH, i++, privAuth, 0, &hte->comm)) {
				LogDebug("privAuth");
				return TSPERR(TSS_E_INTERNAL_ERROR);
			}
		}
		if (getData(TCSD_PACKET_TYPE_UINT32, i++, sigSize, 0, &hte->comm)) {
			LogDebug("sigSize");
			return TSPERR(TSS_E_INTERNAL_ERROR);
		}
		*sig = (BYTE *) malloc(*sigSize);
		if (*sig == NULL) {
			LogError("malloc of %u bytes failed.", *sigSize);
			return TSPERR(TSS_E_OUTOFMEMORY);
		}
		if (getData(TCSD_PACKET_TYPE_PBYTE, i++, *sig, *sigSize, &hte->comm)) {
			LogDebug("sig");
			free(*sig);
			return TSPERR(TSS_E_INTERNAL_ERROR);
		}
	}

	return result;
}

TSS_RESULT
TCSP_GetTestResult_TP(struct host_table_entry *hte,
				  UINT32 * outDataSize,	/* out */
				  BYTE ** outData	/* out */
    ) {
	TSS_RESULT result;

	initData(&hte->comm, 1);

	hte->comm.hdr.u.ordinal = TCSD_ORD_GETTESTRESULT;
	LogDebugFn("TCS Context: 0x%x", hte->tcsContext);

	LogDebug("TCSP_GetTestResult_TP");
	if (setData(TCSD_PACKET_TYPE_UINT32, 0, &hte->tcsContext, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);

	result = sendTCSDPacket(hte);

	if (result == TSS_SUCCESS)
		result = hte->comm.hdr.u.result;

	if (result == TSS_SUCCESS) {
		LogDebug("sendTCSDPacket succeeded");
		if (getData(TCSD_PACKET_TYPE_UINT32, 0, outDataSize, 0, &hte->comm))
			return TSPERR(TSS_E_INTERNAL_ERROR);

		*outData = calloc_tspi(hte->tspContext, *outDataSize);
		if (*outData == NULL) {
			LogError("malloc of %u bytes failed.", *outDataSize);
			return TSPERR(TSS_E_OUTOFMEMORY);
		}

		if (getData(TCSD_PACKET_TYPE_PBYTE, 1, *outData, *outDataSize, &hte->comm)) {
			free_tspi(hte->tspContext, *outData);
			*outData = NULL;
			return TSPERR(TSS_E_INTERNAL_ERROR);
		}
	}
	LogDebug("TCSP_GetTestResult_TP exit");

	return result;
}

TSS_RESULT
TCSP_OwnerClear_TP(struct host_table_entry *hte,
			       TPM_AUTH * ownerAuth	/* in, out */
    ) {
        TSS_RESULT result;

	initData(&hte->comm, 2);

	hte->comm.hdr.u.ordinal = TCSD_ORD_OWNERCLEAR;
	LogDebugFn("TCS Context: 0x%x", hte->tcsContext);

        if (setData(TCSD_PACKET_TYPE_UINT32, 0, &hte->tcsContext, 0, &hte->comm))
                return TSPERR(TSS_E_INTERNAL_ERROR);
        if (setData(TCSD_PACKET_TYPE_AUTH, 1, ownerAuth, 0, &hte->comm))
                return TSPERR(TSS_E_INTERNAL_ERROR);

        result = sendTCSDPacket(hte);

        if (result == TSS_SUCCESS)
                result = hte->comm.hdr.u.result;

        if (result == TSS_SUCCESS ){
                if (getData(TCSD_PACKET_TYPE_AUTH, 0, ownerAuth, 0, &hte->comm))
                        return TSPERR(TSS_E_INTERNAL_ERROR);
        }

        return result;
}

TSS_RESULT
TCSP_DisableOwnerClear_TP(struct host_table_entry *hte,
				      TPM_AUTH * ownerAuth	/* in, out */
    ) {
        TSS_RESULT result;

	initData(&hte->comm, 2);

	hte->comm.hdr.u.ordinal = TCSD_ORD_DISABLEOWNERCLEAR;
	LogDebugFn("TCS Context: 0x%x", hte->tcsContext);

        if (setData(TCSD_PACKET_TYPE_UINT32, 0, &hte->tcsContext, 0, &hte->comm))
                return TSPERR(TSS_E_INTERNAL_ERROR);
        if (setData(TCSD_PACKET_TYPE_AUTH, 1, ownerAuth, 0, &hte->comm))
                return TSPERR(TSS_E_INTERNAL_ERROR);

        result = sendTCSDPacket(hte);

        if (result == TSS_SUCCESS)
                result = hte->comm.hdr.u.result;

        if (result == TSS_SUCCESS ){
                if (getData(TCSD_PACKET_TYPE_AUTH, 0, ownerAuth, 0, &hte->comm))
                        return TSPERR(TSS_E_INTERNAL_ERROR);
        }

        return result;
}

TSS_RESULT
TCSP_ForceClear_TP(struct host_table_entry *hte)
{
	TSS_RESULT result;

	initData(&hte->comm, 1);

	hte->comm.hdr.u.ordinal = TCSD_ORD_FORCECLEAR;
	LogDebugFn("TCS Context: 0x%x", hte->tcsContext);

	if (setData(TCSD_PACKET_TYPE_UINT32, 0, &hte->tcsContext, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);

	result = sendTCSDPacket(hte);

	if (result == TSS_SUCCESS)
		result = hte->comm.hdr.u.result;

	return result;
}

TSS_RESULT
TCSP_DisableForceClear_TP(struct host_table_entry *hte)
{
        TSS_RESULT result;

	initData(&hte->comm, 1);

	hte->comm.hdr.u.ordinal = TCSD_ORD_DISABLEFORCECLEAR;
	LogDebugFn("TCS Context: 0x%x", hte->tcsContext);

        if (setData(TCSD_PACKET_TYPE_UINT32, 0, &hte->tcsContext, 0, &hte->comm))
                return TSPERR(TSS_E_INTERNAL_ERROR);

        result = sendTCSDPacket(hte);

        if (result == TSS_SUCCESS)
                result = hte->comm.hdr.u.result;

        return result;
}

TSS_RESULT
TCSP_PhysicalDisable_TP(struct host_table_entry *hte)
{
	TSS_RESULT result;

	initData(&hte->comm, 1);

	hte->comm.hdr.u.ordinal = TCSD_ORD_PHYSICALDISABLE;
	LogDebugFn("TCS Context: 0x%x", hte->tcsContext);

	if (setData(TCSD_PACKET_TYPE_UINT32, 0, &hte->tcsContext, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);

	result = sendTCSDPacket(hte);

	if (result == TSS_SUCCESS)
		result = hte->comm.hdr.u.result;

	return result;
}

TSS_RESULT
TCSP_PhysicalEnable_TP(struct host_table_entry *hte)
{
	TSS_RESULT result;

	initData(&hte->comm, 1);

	hte->comm.hdr.u.ordinal = TCSD_ORD_PHYSICALENABLE;
	LogDebugFn("TCS Context: 0x%x", hte->tcsContext);

	if (setData(TCSD_PACKET_TYPE_UINT32, 0, &hte->tcsContext, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);

	result = sendTCSDPacket(hte);

	if (result == TSS_SUCCESS)
		result = hte->comm.hdr.u.result;

	return result;
}

TSS_RESULT
TCSP_OwnerSetDisable_TP(struct host_table_entry *hte,
			TSS_BOOL disableState,     /*  in */
			TPM_AUTH * ownerAuth   /*  in, out */
) {
	TSS_RESULT result;

	initData(&hte->comm, 3);

	hte->comm.hdr.u.ordinal = TCSD_ORD_OWNERSETDISABLE;
	LogDebugFn("TCS Context: 0x%x", hte->tcsContext);

	if (setData(TCSD_PACKET_TYPE_UINT32, 0, &hte->tcsContext, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_BOOL, 1, &disableState, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_AUTH, 2, ownerAuth, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);

	result = sendTCSDPacket(hte);

	if (result == TSS_SUCCESS)
		result = hte->comm.hdr.u.result;

	if (result == TSS_SUCCESS) {
		if (getData(TCSD_PACKET_TYPE_AUTH, 0, ownerAuth, 0, &hte->comm))
			return TSPERR(TSS_E_INTERNAL_ERROR);
	}

	return result;
}


TSS_RESULT
TCSP_PhysicalSetDeactivated_TP(struct host_table_entry *hte,
					   TSS_BOOL state	/* in */
    ) {
	TSS_RESULT result;

	initData(&hte->comm, 2);

	hte->comm.hdr.u.ordinal = TCSD_ORD_PHYSICALSETDEACTIVATED;
	LogDebugFn("TCS Context: 0x%x", hte->tcsContext);

	if (setData(TCSD_PACKET_TYPE_UINT32, 0, &hte->tcsContext, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_BOOL, 1, &state, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);

	result = sendTCSDPacket(hte);

	if (result == TSS_SUCCESS)
		result = hte->comm.hdr.u.result;

	return result;
}

TSS_RESULT
TCSP_PhysicalPresence_TP(struct host_table_entry *hte,
				TCPA_PHYSICAL_PRESENCE fPhysicalPresence	/* in */
    ) {
	TSS_RESULT result;

	initData(&hte->comm, 2);

	hte->comm.hdr.u.ordinal = TCSD_ORD_PHYSICALPRESENCE;
	LogDebugFn("TCS Context: 0x%x", hte->tcsContext);

	if (setData(TCSD_PACKET_TYPE_UINT32, 0, &hte->tcsContext, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_UINT16, 1, &fPhysicalPresence, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);

	result = sendTCSDPacket(hte);

	if (result == TSS_SUCCESS)
		result = hte->comm.hdr.u.result;

	return result;
}

TSS_RESULT
TCSP_SetTempDeactivated_TP(struct host_table_entry *hte)
{
	TSS_RESULT result;

	initData(&hte->comm, 1);

	hte->comm.hdr.u.ordinal = TCSD_ORD_SETTEMPDEACTIVATED;
	LogDebugFn("TCS Context: 0x%x", hte->tcsContext);

	if (setData(TCSD_PACKET_TYPE_UINT32, 0, &hte->tcsContext, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);

	result = sendTCSDPacket(hte);

	if (result == TSS_SUCCESS)
		result = hte->comm.hdr.u.result;

	return result;
}

TSS_RESULT
TCSP_FieldUpgrade_TP(struct host_table_entry *hte,
				 UINT32 dataInSize,	/* in */
				 BYTE * dataIn,	/* in */
				 UINT32 * dataOutSize,	/* out */
				 BYTE ** dataOut,	/* out */
				 TPM_AUTH * ownerAuth	/* in, out */
    ) {
	return TSPERR(TSS_E_NOTIMPL);
}

TSS_RESULT
TCSP_SetRedirection_TP(struct host_table_entry *hte,
				   TCS_KEY_HANDLE keyHandle,	/* in */
				   UINT32 c1,	/* in */
				   UINT32 c2,	/* in */
				   TPM_AUTH * privAuth	/* in, out */
    ) {
	return TSPERR(TSS_E_NOTIMPL);

}

TSS_RESULT
TCSP_CreateMaintenanceArchive_TP(struct host_table_entry *hte,
					     TSS_BOOL generateRandom,	/* in */
					     TPM_AUTH * ownerAuth,	/* in, out */
					     UINT32 * randomSize,	/* out */
					     BYTE ** random,	/* out */
					     UINT32 * archiveSize,	/* out */
					     BYTE ** archive	/* out */
    ) {
	TSS_RESULT result;

	initData(&hte->comm, 3);

	hte->comm.hdr.u.ordinal = TCSD_ORD_CREATEMAINTENANCEARCHIVE;
	LogDebugFn("TCS Context: 0x%x", hte->tcsContext);

	if (setData(TCSD_PACKET_TYPE_UINT32, 0, &hte->tcsContext, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_BOOL, 1, &generateRandom, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_AUTH, 2, ownerAuth, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);

	result = sendTCSDPacket(hte);

	if (result == TSS_SUCCESS)
		result = hte->comm.hdr.u.result;

	if (result == TSS_SUCCESS) {
		if (getData(TCSD_PACKET_TYPE_AUTH, 0, ownerAuth, 0, &hte->comm))
			return TSPERR(TSS_E_INTERNAL_ERROR);
		if (getData(TCSD_PACKET_TYPE_UINT32, 1, randomSize, 0, &hte->comm))
			return TSPERR(TSS_E_INTERNAL_ERROR);

		if (*randomSize > 0) {
			*random = calloc_tspi(hte->tspContext, *randomSize);
			if (*random == NULL) {
				LogError("malloc of %u bytes failed.", *randomSize);
				return TSPERR(TSS_E_OUTOFMEMORY);
			}

			if (getData(TCSD_PACKET_TYPE_PBYTE, 2, *random, *randomSize, &hte->comm)) {
				free_tspi(hte->tspContext, *random);
				return TSPERR(TSS_E_INTERNAL_ERROR);
			}
		} else {
			*random = NULL;
		}

		/* Assume all elements are in the list, even when *randomSize == 0. */
		if (getData(TCSD_PACKET_TYPE_UINT32, 3, archiveSize, 0, &hte->comm)) {
			free_tspi(hte->tspContext, *random);
			return TSPERR(TSS_E_INTERNAL_ERROR);
		}

		if (*archiveSize > 0) {
			*archive = calloc_tspi(hte->tspContext, *archiveSize);
			if (*archive == NULL) {
				LogError("malloc of %u bytes failed.", *archiveSize);
				free_tspi(hte->tspContext, *random);
				return TSPERR(TSS_E_OUTOFMEMORY);
			}

			if (getData(TCSD_PACKET_TYPE_PBYTE, 4, *archive, *archiveSize, &hte->comm)) {
				free_tspi(hte->tspContext, *random);
				free_tspi(hte->tspContext, *archive);
				return TSPERR(TSS_E_INTERNAL_ERROR);
			}
		} else {
			*archive = NULL;
		}
	}

	return result;
}

TSS_RESULT
TCSP_LoadMaintenanceArchive_TP(struct host_table_entry *hte,
					   UINT32 dataInSize,	/* in */
					   BYTE * dataIn,	/* in */
					   TPM_AUTH * ownerAuth,	/* in, out */
					   UINT32 * dataOutSize,	/* out */
					   BYTE ** dataOut	/* out */
    ) {
	TSS_RESULT result;

	initData(&hte->comm, 4);

	hte->comm.hdr.u.ordinal = TCSD_ORD_LOADMAINTENANCEARCHIVE;
	LogDebugFn("TCS Context: 0x%x", hte->tcsContext);

	if (setData(TCSD_PACKET_TYPE_UINT32, 0, &hte->tcsContext, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_UINT32, 1, &dataInSize, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_PBYTE, 2, &dataIn, dataInSize, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_AUTH, 3, ownerAuth, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);

	result = sendTCSDPacket(hte);

	if (result == TSS_SUCCESS)
		result = hte->comm.hdr.u.result;

	if (result == TSS_SUCCESS) {
		if (getData(TCSD_PACKET_TYPE_AUTH, 0, ownerAuth, 0, &hte->comm))
			return TSPERR(TSS_E_INTERNAL_ERROR);
		if (getData(TCSD_PACKET_TYPE_UINT32, 1, dataOutSize, 0, &hte->comm))
			return TSPERR(TSS_E_INTERNAL_ERROR);

		if (*dataOutSize > 0) {
			*dataOut = malloc(*dataOutSize);
			if (*dataOut == NULL) {
				LogError("malloc of %u bytes failed.", *dataOutSize);
				return TSPERR(TSS_E_OUTOFMEMORY);
			}

			if (getData(TCSD_PACKET_TYPE_PBYTE, 2, *dataOut, *dataOutSize, &hte->comm)) {
				free(*dataOut);
				return TSPERR(TSS_E_INTERNAL_ERROR);
			}
		} else {
			*dataOut = NULL;
		}
	}

	return result;
}

TSS_RESULT
TCSP_KillMaintenanceFeature_TP(struct host_table_entry *hte,
					   TPM_AUTH * ownerAuth	/* in , out */
    ) {
	TSS_RESULT result;

	initData(&hte->comm, 2);

	hte->comm.hdr.u.ordinal = TCSD_ORD_KILLMAINTENANCEFEATURE;
	LogDebugFn("TCS Context: 0x%x", hte->tcsContext);

	if (setData(TCSD_PACKET_TYPE_UINT32, 0, &hte->tcsContext, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_AUTH, 1, ownerAuth, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);

	result = sendTCSDPacket(hte);

	if (result == TSS_SUCCESS)
		result = hte->comm.hdr.u.result;

	if (result == TSS_SUCCESS) {
		if (getData(TCSD_PACKET_TYPE_AUTH, 0, ownerAuth, 0, &hte->comm))
			return TSPERR(TSS_E_INTERNAL_ERROR);
	}

	return result;
}

TSS_RESULT
TCSP_LoadManuMaintPub_TP(struct host_table_entry *hte,
				     TCPA_NONCE antiReplay,	/* in */
				     UINT32 PubKeySize,	/* in */
				     BYTE * PubKey,	/* in */
				     TCPA_DIGEST * checksum	/* out */
    ) {
	TSS_RESULT result;

	initData(&hte->comm, 4);

	hte->comm.hdr.u.ordinal = TCSD_ORD_LOADMANUFACTURERMAINTENANCEPUB;
	LogDebugFn("TCS Context: 0x%x", hte->tcsContext);

	if (setData(TCSD_PACKET_TYPE_UINT32, 0, &hte->tcsContext, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_NONCE, 1, &antiReplay, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_UINT32, 2, &PubKeySize, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_PBYTE, 3, PubKey, PubKeySize, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);

	result = sendTCSDPacket(hte);

	if (result == TSS_SUCCESS)
		result = hte->comm.hdr.u.result;

	if (result == TSS_SUCCESS) {
		if (getData(TCSD_PACKET_TYPE_DIGEST, 0, checksum, 0, &hte->comm))
			return TSPERR(TSS_E_INTERNAL_ERROR);
	}

	return result;
}

TSS_RESULT
TCSP_ReadManuMaintPub_TP(struct host_table_entry *hte,
				     TCPA_NONCE antiReplay,	/* in */
				     TCPA_DIGEST * checksum	/* out */
    ) {
	TSS_RESULT result;

	initData(&hte->comm, 2);

	hte->comm.hdr.u.ordinal = TCSD_ORD_READMANUFACTURERMAINTENANCEPUB;
	LogDebugFn("TCS Context: 0x%x", hte->tcsContext);

	if (setData(TCSD_PACKET_TYPE_UINT32, 0, &hte->tcsContext, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);
	if (setData(TCSD_PACKET_TYPE_NONCE, 1, &antiReplay, 0, &hte->comm))
		return TSPERR(TSS_E_INTERNAL_ERROR);

	result = sendTCSDPacket(hte);

	if (result == TSS_SUCCESS)
		result = hte->comm.hdr.u.result;

	if (result == TSS_SUCCESS) {
		if (getData(TCSD_PACKET_TYPE_DIGEST, 0, checksum, 0, &hte->comm))
			return TSPERR(TSS_E_INTERNAL_ERROR);
	}

	return result;
}
