
/*
 * Licensed Materials - Property of IBM
 *
 * trousers - An open source TCG Software Stack
 *
 * (C) Copyright International Business Machines Corp. 2004
 *
 */


#include <stdlib.h>
#include <stdio.h>
#include <syslog.h>
#include <pthread.h>
#include <errno.h>
#include <string.h>
#include <unistd.h>

#include "trousers/tss.h"
#include "trousers_types.h"
#include "tcs_int_literals.h"
#include "tcs_tsp.h"
#include "tcs_utils.h"
#include "tcsd_wrap.h"
#include "tcsd.h"
#include "tcslog.h"

struct tcsd_thread_mgr *tm = NULL;

TSS_RESULT
tcsd_threads_final()
{
	int rc;
	UINT32 i;

	pthread_mutex_lock(&(tm->lock));

	tm->shutdown = 1;

	pthread_mutex_unlock(&(tm->lock));

	/* wait for all currently running threads to exit */
	for (i = 0; i < tm->max_threads; i++) {
		if (tm->thread_data[i].thread_id != (pthread_t)0) {
			if ((rc = pthread_join(tm->thread_data[i].thread_id, NULL))) {
				LogError("pthread_join() failed: error: %d", rc);
			}
		}
	}

	free(tm->thread_data);
	free(tm);

	return TSS_SUCCESS;
}

TSS_RESULT
tcsd_threads_init(void)
{
	/* allocate the thread mgmt structure */
	tm = calloc(1, sizeof(struct tcsd_thread_mgr));
	if (tm == NULL) {
		LogError("malloc of %zd bytes failed.", sizeof(struct tcsd_thread_mgr));
		return TCSERR(TSS_E_OUTOFMEMORY);
	}

	/* set the max threads variable from config */
	tm->max_threads = tcsd_options.num_threads;

	/* allocate each thread's data structure */
	tm->thread_data = calloc(tcsd_options.num_threads, sizeof(struct tcsd_thread_data));
	if (tm->thread_data == NULL) {
		LogError("malloc of %zu bytes failed.",
			 tcsd_options.num_threads * sizeof(struct tcsd_thread_data));
		free(tm);
		return TCSERR(TSS_E_OUTOFMEMORY);
	}

	return TSS_SUCCESS;
}


TSS_RESULT
tcsd_thread_create(int socket, char *hostname)
{
	UINT32 thread_num;
#ifndef TCSD_SINGLE_THREAD_DEBUG
	int rc;
	pthread_attr_t tcsd_thread_attr;

	/* init the thread attribute */
	if ((rc = pthread_attr_init(&tcsd_thread_attr))) {
		LogError("pthread_attr_init failed: error=%d: %s", rc, strerror(rc));
		return TCSERR(TSS_E_INTERNAL_ERROR);
	}
	/* make all threads joinable */
	if ((rc = pthread_attr_setdetachstate(&tcsd_thread_attr, PTHREAD_CREATE_JOINABLE))) {
		LogError("pthread_attr_init failed: error=%d: %s", rc, strerror(rc));
		return TCSERR(TSS_E_INTERNAL_ERROR);
	}

	pthread_mutex_lock(&(tm->lock));
#endif
	if (tm->num_active_threads == tm->max_threads) {
		close(socket);
		if (hostname != NULL) {
			LogError("max number of connections reached (%d), new connection"
				 " from %s refused.", tm->max_threads, hostname);
		} else {
			LogError("max number of connections reached (%d), new connection"
				 " refused.", tm->max_threads);
		}
		free(hostname);
		pthread_mutex_unlock(&(tm->lock));
		return TCSERR(TSS_E_CONNECTION_FAILED);
	}

	/* search for an open slot to store the thread data in */
	for (thread_num = 0; thread_num < tm->max_threads; thread_num++) {
		if (tm->thread_data[thread_num].thread_id == (pthread_t)0)
			break;
	}

	DBG_ASSERT(thread_num != tm->max_threads);

	tm->thread_data[thread_num].sock = socket;
	tm->thread_data[thread_num].context = NULL_TCS_HANDLE;
	if (hostname != NULL)
		tm->thread_data[thread_num].hostname = hostname;

#ifdef TCSD_SINGLE_THREAD_DEBUG
	(void)tcsd_thread_run((void *)(&(tm->thread_data[thread_num])));
#else
	if ((rc = pthread_create(&(tm->thread_data[thread_num].thread_id),
				 &tcsd_thread_attr,
				 tcsd_thread_run,
				 (void *)(&(tm->thread_data[thread_num]))))) {
		LogError("pthread_create() failed: %d", rc);
		pthread_mutex_unlock(&(tm->lock));
		return TCSERR(TSS_E_INTERNAL_ERROR);
	}

	tm->num_active_threads++;

	pthread_mutex_unlock(&(tm->lock));
#endif
	return TSS_SUCCESS;
}

/* since we don't want any of the worker threads to catch any signals,
 * we must mask off any potential signals here after creating the threads.  If any of
 * the created threads catch a signal, they'd eventually call pthread_join() on
 * themselves, causing a deadlock.
 */
void
thread_signal_init()
{
	sigset_t thread_sigmask;
	int rc;

	if ((rc = sigfillset(&thread_sigmask))) {
		LogError("sigfillset failed: error=%d: %s", rc, strerror(rc));
		LogError("worker thread %u is exiting prematurely", (unsigned int)pthread_self());
		pthread_exit(NULL);
	}

	if ((rc = pthread_sigmask(SIG_BLOCK, &thread_sigmask, NULL))) {
		LogError("pthread_sigmask failed: error=%d: %s", rc, strerror(rc));
		LogError("worker thread %u is exiting prematurely", (unsigned int)pthread_self());
		pthread_exit(NULL);
	}
}

void *
tcsd_thread_run(void *v)
{
	struct tcsd_thread_data *data = (struct tcsd_thread_data *)v;
	BYTE *buffer;
	int recv_size, send_size;
	TSS_RESULT result;
	UINT64 offset;
#ifndef TCSD_SINGLE_THREAD_DEBUG
	int rc;

	thread_signal_init();
#endif

	data->comm.buf_size = TCSD_INIT_TXBUF_SIZE;
	data->comm.buf = calloc(1, data->comm.buf_size);
	while (data->comm.buf) {
		/* get the packet header to get the size of the incoming packet */
		buffer = data->comm.buf;
		recv_size = sizeof(struct tcsd_packet_hdr);
		if ((recv_size = recv_from_socket(data->sock, buffer, recv_size)) < 0)
			break;
		buffer += sizeof(struct tcsd_packet_hdr);	/* increment the buffer pointer */

		/* check the packet size */
		recv_size = Decode_UINT32(data->comm.buf);
		if (recv_size < (int)sizeof(struct tcsd_packet_hdr)) {
			LogError("Packet to receive from socket %d is too small (%d bytes)",
				 data->sock, recv_size);
			break;
		}

		if (recv_size > data->comm.buf_size ) {
			BYTE *new_buffer;

			LogDebug("Increasing communication buffer to %d bytes.", recv_size);
			new_buffer = realloc(data->comm.buf, recv_size);
			if (new_buffer == NULL) {
				LogError("realloc of %d bytes failed.", recv_size);
				break;
			}
			buffer = new_buffer + sizeof(struct tcsd_packet_hdr);
			data->comm.buf_size = recv_size;
			data->comm.buf = new_buffer;
		}

		/* get the rest of the packet */
		recv_size -= sizeof(struct tcsd_packet_hdr);	/* already received the header */
		if ((recv_size = recv_from_socket(data->sock, buffer, recv_size)) < 0)
			break;
		LogDebug("Rx'd packet");

		/* create a platform version of the tcsd header */
		offset = 0;
		UnloadBlob_UINT32(&offset, &data->comm.hdr.packet_size, data->comm.buf, NULL);
		UnloadBlob_UINT32(&offset, &data->comm.hdr.u.result, data->comm.buf, NULL);
		UnloadBlob_UINT32(&offset, &data->comm.hdr.num_parms, data->comm.buf, NULL);
		UnloadBlob_UINT32(&offset, &data->comm.hdr.type_size, data->comm.buf, NULL);
		UnloadBlob_UINT32(&offset, &data->comm.hdr.type_offset, data->comm.buf, NULL);
		UnloadBlob_UINT32(&offset, &data->comm.hdr.parm_size, data->comm.buf, NULL);
		UnloadBlob_UINT32(&offset, &data->comm.hdr.parm_offset, data->comm.buf, NULL);

		if ((result = getTCSDPacket(data)) != TSS_SUCCESS) {
			/* something internal to the TCSD went wrong in preparing the packet
			 * to return to the TSP.  Use our already allocated buffer to return a
			 * TSS_E_INTERNAL_ERROR return code to the TSP. In the non-error path,
			 * these LoadBlob's are done in getTCSDPacket().
			 */
			/* set everything to zero, fill in what is non-zero */
			memset(data->comm.buf, 0, data->comm.buf_size);
			offset = 0;
			/* load packet size */
			LoadBlob_UINT32(&offset, sizeof(struct tcsd_packet_hdr), data->comm.buf, NULL);
			/* load result */
			LoadBlob_UINT32(&offset, result, data->comm.buf, NULL);
		}
		send_size = Decode_UINT32(data->comm.buf);
		LogDebug("Sending 0x%X bytes back", send_size);
		send_size = send_to_socket(data->sock, data->comm.buf, send_size);
		if (send_size < 0)
			break;

		/* check for shutdown */
		if (tm->shutdown) {
			LogDebug("Thread %u exiting via shutdown signal!", (unsigned int)pthread_self());
			break;
		}
	}

	LogDebug("Thread exiting.");

	/* Closing connection to TSP */
	close(data->sock);
	data->sock = -1;
	free(data->comm.buf);
	data->comm.buf = NULL;
	data->comm.buf_size = -1;
	/* If the connection was not shut down cleanly, free TCS resources here */
	if (data->context != NULL_TCS_HANDLE) {
		TCS_CloseContext_Internal(data->context);
		data->context = NULL_TCS_HANDLE;
	}

#ifndef TCSD_SINGLE_THREAD_DEBUG
	pthread_mutex_lock(&(tm->lock));
	tm->num_active_threads--;
	/* if we're not in shutdown mode, then nobody is waiting to join this thread, so
	 * detach it so that its resources are free at pthread_exit() time. */
	if (!tm->shutdown) {
		if ((rc = pthread_detach(data->thread_id))) {
			LogError("pthread_detach failed (errno %d)."
				 " Resources may not be properly released.", rc);
		}
	}
	free(data->hostname);
	data->hostname = NULL;
	data->thread_id = (pthread_t)0;
	pthread_mutex_unlock(&(tm->lock));
	pthread_exit(NULL);
#else
	return NULL;
#endif
}
