/***************************************************************************
 
   trm - the TRM digital finger print utilty 
  
   Copyright (C) 2002 Robert Kaye

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   as published by the Free Software Foundation; either version 2
   of the License, or (at your option) any later version. 
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details. 
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software 
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111, USA.

   $Id: main.cpp,v 1.14 2003/02/05 02:41:05 robert Exp $

 ***************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <musicbrainz/mb_c.h>
#include <musicbrainz/browser.h>
#include "config.h"
#include "defs.h"
#include "trm.h"
#include "mbid3.h"

#ifndef WIN32
#define stricmp(s,d) strcasecmp(s,d)
#include <unistd.h>
#else
#include <winsock.h>
#include <io.h>
#endif

char* g_proxy = NULL;
int   g_proxyPort = 80;

enum FileType { eUnknown, eMP3, eOggVorbis, eWav };

char *EscapeFormValue(const char *form_value)
{
    int i, form_value_length, extra_length;
    char *escaped_value, *ptr;

    form_value_length = strlen(form_value);
    for (i = 0, extra_length = 0; i < form_value_length; ++i)
    {
        switch(form_value[i])
        {
            case '"':
	    case '\'':
                extra_length += 5;
                break;
            case '&':
                extra_length += 4;
                break;
            case '<':
            case '>':
                extra_length += 3;
                break;
        }
    }

    if (extra_length == 0)
    {
        // This is necessary since the caller must free the memory.
        return strdup(form_value);
    }

    escaped_value = (char *)malloc(form_value_length + extra_length + 1);
    for (i = 0, ptr = escaped_value; i < form_value_length; ++i)
    {
        switch(form_value[i])
        {
            case '"':
                strcpy(ptr, "&quot;");
                ptr += 6;
                break;
            case '&':
                strcpy(ptr, "&amp;");
                ptr += 5;
                break;
            case '<':
                strcpy(ptr, "&lt;");
                ptr += 4;
                break;
            case '>':
                strcpy(ptr, "&gt;");
                ptr += 4;
                break;
	    case '\'':
		strcpy(ptr, "&apos;");
		ptr += 6;
		break;
            default:
                *(ptr++) = form_value[i];
        }
    }
    *ptr = 0;

    return escaped_value;
}

int printXML(char *sig)
{
    musicbrainz_t o;
    char          error[256], data[256],*args[2], trackURI[256];
    int           ret, index;

    // Create the musicbrainz object, which will be needed for subsequent calls
    o = mb_New();

//#ifdef WIN32
//    mb_WSAInit(o);
//#endif

    // Tell the client library to return data in ISO8859-1 and not UTF-8
    mb_UseUTF8(o, 1);

    // Set the proper server to use. Defaults to mm.musicbrainz.org:80
    if (getenv("MB_SERVER"))
        mb_SetServer(o, getenv("MB_SERVER"), 80);

    // Check to see if the debug env var has been set 
    if (getenv("MB_DEBUG"))
        mb_SetDebug(o, atoi(getenv("MB_DEBUG")));

    if (getenv("MB_PROXY"))
    	mb_SetProxy(o, getenv("MB_PROXY"), atoi(getenv("MB_PROXY_PORT")));

    if (g_proxy)
    	mb_SetProxy(o, g_proxy, g_proxyPort);

    // Set up the args for the trm query
    args[0] = sig;
    args[1] = NULL;

    // Execute the MBQ_TrackInfoFromTRMId query
    ret = mb_QueryWithArgs(o, MBQ_TrackInfoFromTRMId, args);
    if (!ret)
    {
        mb_GetQueryError(o, error, 256);
        printf("<\?xml version='1.0'?>\n<plist version='1.0'>\n\t<array>\n\t\t<string>%s</string>\n\t</array>\n</plist>\n", error);
        mb_Delete(o);
        return 0;
    }
    
    printf("<\?xml version='1.0'?>\n<plist version='1.0'>\n\t<array>\n");    
    for(index = 1;; index++)
    {
        mb_Select(o, MBS_Rewind);

        // Select the first track from the track list 
        if (!mb_Select1(o, MBS_SelectTrack, index))
        {
            if (index == 1)
		printf("\t\t<string>%s</string>\n", error);
            break;
        }
	else printf("\t<dict>\n");
	
        mb_GetResultData(o, MBE_TrackGetTrackId, trackURI, 256);

        // Extract the artist name from the track
        char *dataPtr;
	if (mb_GetResultData(o, MBE_TrackGetArtistName, data, 256))
		dataPtr = EscapeFormValue(data);
		printf("\t\t <key>Artist</key>\n\t\t    <string>%s</string>\n ",dataPtr); 
        free(dataPtr);   
        // Extract the track name
	if (mb_GetResultData(o, MBE_TrackGetTrackName, data, 256))
	   	dataPtr=EscapeFormValue(data);
		printf("\t\t <key>Name</key>\n\t\t    <string>%s</string>\n ",dataPtr); 
	free(dataPtr);
        // Extract the track duration
        int duration =0;
	duration= mb_GetResultInt(o, MBE_TrackGetTrackDuration);
	printf("\t\t <key>Total Time</key>\n\t\t    <integer>%d</integer>\n ",duration);

        mb_Select(o, MBS_SelectTrackAlbum);

	// Extract the track number
	int trackNum=0;
        trackNum = mb_GetOrdinalFromList(o, MBE_AlbumGetTrackList, trackURI);
        if (trackNum > 0 && trackNum < 100)
          // printf("  TrackNum: %d\n", trackNum);
	   printf("\t\t <key>TrackNum</key>\n\t\t    <integer>%d</integer>\n ",trackNum);
	
        // Extract the album name from the track
        if (mb_GetResultData(o, MBE_AlbumGetAlbumName, data, 256))
                dataPtr=EscapeFormValue(data);
		printf("\t\t <key>Album</key>\n\t\t    <string>%s</string>\n ",dataPtr);
	//free(dataPtr);
    
	// The code below (Getting the ID) segfaults       	
	// Extract the album ID 
 /*    	char id[40];
     	if (mb_GetResultData(o, MBE_AlbumGetAlbumId, data, 256))
	        mb_GetIDFromURL(o, data, id, 40);
	  printf("\t\t <key>AlbumID</key>\n\t\t    <string>%s</string>\n ",id);
	//Getting album type  
	free(id);
*/	
	char *typePtr; //strstr requirement
	if(mb_GetResultData(o,MBE_AlbumGetAlbumType,data,256)){
		if (data!=NULL) {
			typePtr=strstr(data,"Type")+4; //look for type and take all chars after the Type string is encountered
			printf("\t\t  <key>AlbumType</key>\n\t\t    <string>%s</string>\n ",typePtr);	
		}
	}
	int totalTracks=0;
	totalTracks = mb_GetResultInt(o, MBE_AlbumGetNumTracks);
	if (totalTracks > 0 && totalTracks < 100)
	   printf("\t\t <key>TracksInAlbum</key>\n\t\t    <integer>%d</integer>\n ",totalTracks);
	
	//Get release date and country 
	if(mb_GetResultInt(o, MBE_AlbumGetNumReleaseDates))
		mb_Select1(o,MBS_SelectReleaseDate,1);
		
		char country[64];	
		mb_GetResultData(o,MBE_ReleaseGetCountry,country,64);
		if((strcmp(country,""))!=0)
	 		printf("\t\t <key>Country</key>\n\t\t    <string>%s</string>\n ",country);
	
	 	char date[15];
        	mb_GetResultData(o,MBE_ReleaseGetDate,date,15);
		int year=atoi(date);
		if(year!=0)
         		printf("\t\t <key>Year</key>\n\t\t    <string>%d</string>\n ",year);
	mb_Select(0,MBS_Rewind);
	
	printf("\t</dict>\n");
    }
	printf("   </array>\n </plist>\n");
	
    // and clean up the musicbrainz object
    mb_Delete(o);

    return 0;
}



int main(int argc, char *argv[])
{
    char          sig[37], *ptr;
    int           index = 1, isLookup = 0, printID3 = 0,printXMLPlist=0;
    unsigned long duration = 0;
    FileType      type = eUnknown;

    if (argc < 2)
    {
        fprintf(stderr,"usage: trmxml [-m|-o|-w] [-x] [-p server port] "
                       "<mp3 | ogg/vorbis | wav file>\n");
        fprintf(stderr,"Options:\n  -m   Input file is mp3\n");
        fprintf(stderr,"  -o   Input file is Ogg/Vorbis\n");
        fprintf(stderr,"  -w   Input file is WAV\n");
        fprintf(stderr,"  -x   Output xml plist with recognized tracks\n");
	fprintf(stderr,"  -p   Set proxy\n");
        return -1;
    }

#ifdef WIN32
    WSADATA  sGawdIHateMicrosoft;
    WSAStartup(0x0002,  &sGawdIHateMicrosoft);
#endif

    for(; index < argc; index++)
    {
        if (stricmp(argv[index], "-m") == 0)
           type = eMP3;
        else
        if (stricmp(argv[index], "-o") == 0)
           type = eOggVorbis;
        else
        if (stricmp(argv[index], "-w") == 0)
           type = eWav;
        else 
	if (stricmp(argv[index], "-x") == 0)
	   printXMLPlist=1;
	else
	if (stricmp(argv[index], "-p") == 0) {
	   if (index>argc-2) {
	       fprintf(stderr,"error: incorrect proxy parameters.\n");
	       exit(-1);
	   }
	   g_proxy = argv[++index];
	   g_proxyPort = atoi(argv[++index]);
	}
	else
           break;
    }

    if (index >= argc)
    {
        fprintf(stderr, "error: no lookup file specified.\n");
        exit(-1);
    }
    if (access(argv[index], 0))
    {
        fprintf(stderr, "Cannot open file %s\n", argv[index]);
        return (-1);
    }

    if (type == eUnknown)
    {
        ptr = strrchr(argv[index], '.');
        if (ptr && stricmp(ptr, ".mp3") == 0)
           type = eMP3;

        if (ptr && stricmp(ptr, ".ogg") == 0)
           type = eOggVorbis;

        if (ptr && stricmp(ptr, ".wav") == 0)
           type = eWav;

        if (ptr == NULL && type == eUnknown)
        {
            fprintf(stderr, "error: cannot determine audio file type.\n");
            fprintf(stderr, "       use -m, -o, -w to set the file type.\n");
            return(-1); 
        }                             
    }
    if (printXMLPlist)
    {
	switch(type)
	{
		case eMP3:
			{
#ifdef HAVE_LIBMAD
			int ret;
			ret = MP3_generateTRM(argv[index], sig, &duration, "", 0);
			    if (ret == 0)
		    	       printXML(sig);
			    else
			       printf("<?xml version='1.0'?>\n<plist/>\n");
#endif
			 break;
			}
		case eOggVorbis:
			{
#ifdef HAVE_OGGVORBIS
			  int ret;
			  ret = OggVorbis_generateTRM(argv[index], sig, &duration, "", 0);
            		   if (ret == 0)
		                    printf("%s\n", sig);
			   else printf("<?xml version='1.0'?>\n<plist/>\n");
#endif
			   break;
			}
		case eWav:
			{
			int ret;
			ret = Wav_generateTRM(argv[index], sig, &duration, "", 0);  	
			if (ret==0) printXML(sig);
			else printf("<?xml version='1.0'?>\n<plist/>\n");
			break;
			}
	}
    }
    else
    switch(type)
    {
        case eMP3:
        {
#ifdef HAVE_LIBMAD
            int ret;

            ret = MP3_generateTRM(argv[index], sig, &duration, "", 0);
            if (ret == 0)
                printf("%s\n", sig);
            else
                fprintf(stderr, "error: failed to generate TRM id\n");
#else
            fprintf(stderr, "error: mp3 support not built into application\n");
#endif
            break;
        }
        case eOggVorbis:
        {
#ifdef HAVE_OGGVORBIS
            int ret;

            ret = OggVorbis_generateTRM(argv[index], sig, &duration, "", 0);
            if (ret == 0)
                printf("%s\n", sig);
            else
                fprintf(stderr, "error: failed to generate TRM id\n");
#else
            fprintf(stderr, "error: ogg/vorbis support not built into application\n");
#endif
            break;
        }
        case eWav:
        {
            int ret;

            ret = Wav_generateTRM(argv[index], sig, &duration, "", 0);
            if (ret == 0)
                printf("%s\n", sig);
            else
                fprintf(stderr, "error: failed to generate TRM id\n");
            break;
        }
    }

    if (printID3)
    {
#ifdef HAVE_LIBID3
  //      PrintMetadata(argv[index], type, duration);
#else
  //      fprintf(stderr, "error: lookup (id3) support not built into application\n");
#endif
    }

#ifdef WIN32
    WSACleanup();
#endif

    return 0;
}
