//@HEADER
// ************************************************************************
// 
//            NOX: An Object-Oriented Nonlinear Solver Package
//                 Copyright (2002) Sandia Corporation
// 
//            LOCA: Library of Continuation Algorithms Package
//                 Copyright (2005) Sandia Corporation
// 
// Under terms of Contract DE-AC04-94AL85000, there is a non-exclusive
// license for use of this work by or on behalf of the U.S. Government.
// 
// This library is free software; you can redistribute it and/or modify
// it under the terms of the GNU Lesser General Public License as
// published by the Free Software Foundation; either version 2.1 of the
// License, or (at your option) any later version.
//  
// This library is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
// 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA
// 
// Questions? Contact Roger Pawlowski (rppawlo@sandia.gov) or 
// Eric Phipps (etphipp@sandia.gov), Sandia National Laboratories.
// ************************************************************************
//  CVS Information
//  $Source$
//  $Author$
//  $Date$
//  $Revision$
// ************************************************************************
//@HEADER
                                                                                
// Finite Element Problem Class
/* Provides function (F) and Jacobian evaluations for the following equation 
 * via a 1D linear finite element discretization with Epetra objects.
 *
 * d2u 
 * --- - k * u**2 = 0
 * dx2
 *
 * subject to @ x=0, u=1
 */

#ifndef _NOX_EXAMPLE_EPETRA_NONLINEAR_FINITEELEMENTPROBLEM_H
#define _NOX_EXAMPLE_EPETRA_NONLINEAR_FINITEELEMENTPROBLEM_H

// This is needed to get the FillType enum values from NOX_Epetra_Interface
#include "NOX_Epetra.H"
#include "Teuchos_RCP.hpp"

// Forward Declarations
class Epetra_Comm;
class Epetra_Map; 
class Epetra_Vector;
class Epetra_Import;
class Epetra_CrsGraph;
class Epetra_CrsMatrix;

// Finite Element Problem Class
class FiniteElementProblem { 

public:
  
  // Flag to tell the evaluate routine what objects to fill
  enum FillType {F_ONLY, MATRIX_ONLY, ALL}; 

public:

  // Constructor
  FiniteElementProblem(int NumGlobalElements, Epetra_Comm& Comm);
  
  // Destructor
  ~FiniteElementProblem();

  // Evaluates the function (F) and/or the Jacobian using the solution 
  // values in solnVector.
  bool evaluate(FillType f, const Epetra_Vector *solnVector, 
		Epetra_Vector *rhsVector, Epetra_RowMatrix *matrix,
                NOX::Epetra::Interface::Required::FillType fill=NOX::Epetra::Interface::Required::Residual);

  // Return a reference to the Epetra_Vector with the initial guess
  // that is generated by the FiniteElementProblem class.
  Teuchos::RCP<Epetra_Vector> getSolution();
  
  // Return a reference to the Epetra_Vector with the Jacobian
  // that is generated by the FiniteElementProblem class.
  Teuchos::RCP<Epetra_CrsMatrix> getJacobian();

  // Return a reference to the Epetra_CrsGraph that is generated by 
  // the FiniteElementProblem class.
  Teuchos::RCP<Epetra_CrsGraph> getGraph();

private:

  // inserts the global column indices into the Graph
  Epetra_CrsGraph& generateGraph(Epetra_CrsGraph& AA);

private:

  FillType flag;
  Epetra_Map *StandardMap; 
  Epetra_Map *OverlapMap;
  Epetra_Import *Importer;
  Epetra_Import *ColumnToOverlapImporter;
  Teuchos::RCP<Epetra_Vector> initialSolution;
  Epetra_Vector *rhs;
  Teuchos::RCP<Epetra_CrsGraph> AA;
  Teuchos::RCP<Epetra_CrsMatrix> A;
  Epetra_Comm *Comm;

  int MyPID;              // Process number
  int NumProc;            // Total number of processes
  int NumMyElements;      // Number of elements owned by this process
  int NumGlobalElements;  // Total Number of elements

};
#endif




