#!/usr/bin/env python

#****************************************************************************
# printpreview.py, provides a print preview window
#
# Copyright (C) 2005, Douglas W. Bell
#
# This is free software; you can redistribute it and/or modify it under the
# terms of the GNU General Public License, Version 2.  This program is
# distributed in the hope that it will be useful, but WITTHOUT ANY WARRANTY.
#*****************************************************************************

import sys
from qt import Qt, PYSIGNAL, SIGNAL, SLOT, QBrush, QColor, QDialog, \
               QFrame, QHBoxLayout, QLabel, QPainter, QPushButton, QRect, \
               QSizePolicy, QVBoxLayout, QWidget
               

class PrintPrevDlg(QDialog):
    """Provides a generic print preview with page controls"""
    def __init__(self, minPage, maxPage, pageSize, pageAvail, margins, \
                 pageCmd, parent=None, name=None, modal=0, flags=0):
        QDialog.__init__(self, parent, name, modal, flags)
        self.curPage = minPage
        self.minPage = minPage
        self.maxPage = maxPage
        self.resize(300, 560)
        topLayout = QVBoxLayout(self, 5)
        ctrlLayout = QHBoxLayout(topLayout)

        self.prevButton  = QPushButton(_('P&rev. Page'), self)
        ctrlLayout.addWidget(self.prevButton)
        self.connect(self.prevButton, SIGNAL('clicked()'), self.prevPage)

        self.nextButton  = QPushButton(_('&Next Page'), self)
        ctrlLayout.addWidget(self.nextButton)
        self.connect(self.nextButton, SIGNAL('clicked()'), self.nextPage)

        self.statusLabel = QLabel('', self)
        self.statusLabel.setAlignment(Qt.AlignCenter)
        self.statusLabel.setFrameStyle(QFrame.Panel | QFrame.Sunken)
        ctrlLayout.addWidget(self.statusLabel, 1)

        cancelButton  = QPushButton(_('&Cancel'), self)
        ctrlLayout.addWidget(cancelButton)
        self.connect(cancelButton, SIGNAL('clicked()'), self.reject)

        printButton  = QPushButton(_('&Print'), self)
        ctrlLayout.addWidget(printButton)
        self.connect(printButton, SIGNAL('clicked()'), self.accept)

        self.preview = PrintPrev(pageSize, pageAvail, margins, pageCmd, self)
        self.preview.setMinimumSize(200, 400)
        self.preview.setSizePolicy(QSizePolicy(QSizePolicy.Expanding, \
                                               QSizePolicy.Expanding))
        topLayout.addWidget(self.preview, 1)

        self.updatePageNum()
        caption = _('Print Preview')
        if sys.platform == 'win32':
            caption += '  (PyQt)'
        self.setCaption(caption)

    def updatePageNum(self):
        """Enable/disable prev & next buttons, update status label 
           and preview"""
        self.prevButton.setEnabled(self.curPage > self.minPage)
        self.nextButton.setEnabled(self.curPage < self.maxPage)
        self.statusLabel.setText(_('Page %(current)i of %(max)i') % \
                                 {'current':self.curPage, 'max':self.maxPage})
        self.preview.setPageNum(self.curPage)

    def prevPage(self):
        """Go to previous page"""
        if self.curPage > self.minPage:
            self.curPage -= 1
            self.updatePageNum()

    def nextPage(self):
        """Go to next page"""
        if self.curPage < self.maxPage:
            self.curPage += 1
            self.updatePageNum()


class PrintPrev(QWidget):
    """Provides a widget for the paper"""
    def __init__(self, pageSize, pageAvail, margins, pageCmd, parent=None, \
                 name=None, flags=0):
        QWidget.__init__(self, parent, name, flags)
        self.pageSize = pageSize
        self.pageAvail = pageAvail
        self.margins = margins
        self.pageCmd = pageCmd
        self.pageNum = 0

    def setPageNum(self, pageNum):
        """Set new page number and update"""
        self.pageNum = pageNum
        self.update()

    def paintEvent(self, event):
        """Paint the current page"""
        paint = QPainter()   # problem in win with QPainter(self)
        paint.begin(self)
        pgAspect = float(self.pageSize[0]) / self.pageSize[1]
        viewRect = paint.viewport()
        if pgAspect < float(viewRect.width()) / viewRect.height():
            pageSpace = (viewRect.height() * pgAspect, viewRect.height())
        else:               # width controls size
            pageSpace = (viewRect.width(), viewRect.width() / pgAspect)
        pageRect = QRect(viewRect.left() + \
                         (viewRect.width() - pageSpace[0]) / 2, \
                         viewRect.top() + \
                         (viewRect.height() - pageSpace[1]) / 2, \
                         pageSpace[0], pageSpace[1])
        paint.fillRect(pageRect, QBrush(QColor(255, 255, 255), \
                                        QBrush.SolidPattern))
        paint.setWindow(0, 0, self.pageAvail[0], self.pageAvail[1])
        viewMargins = (self.margins[0] * pageSpace[0] / self.pageSize[0], \
                       self.margins[1] * pageSpace[1] / self.pageSize[1])
        paint.setViewport(pageRect.left() + viewMargins[0], \
                          pageRect.top() + viewMargins[1], \
                          self.pageAvail[0] * pageSpace[0] / self.pageSize[0], \
                          self.pageAvail[1] * pageSpace[1] / self.pageSize[1])
        self.pageCmd(self.pageNum, paint)
        paint.end()
