/*****************************************************************************
    TRAVIS - Trajectory Analyzer and Visualizer
    Copyright (C) 2009-2012 Martin Brehm

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*****************************************************************************/

#include "lmwrapper.h"
#include <math.h>
#include "tools.h"

CLMWrapper::CLMWrapper()
{
}


CLMWrapper::~CLMWrapper()
{
}

int g_iFitDegree;

typedef struct {
    const double *x;
    const double *y;
} lmcurve_data_struct;


void lmcurve_evaluate(const double *par, int m_dat, const void *data, double *fvec, int *info)
{
	int z, z2;
	const double eps = 0.001;

	for (z2=0;z2<m_dat;z2++)
	{
		fvec[z2] = 0;
		for (z=0;z<g_iFitDegree;z++)
		{
			fvec[z2] += par[z*2] * exp(par[z*2+1]*((lmcurve_data_struct*)data)->x[z2]);
			if (par[z*2+1] > -eps)
				fvec[z2] += (par[z*2+1]+eps) * 1000.0;
			if (par[z*2] < 0)
				fvec[z2] += fabs(par[z*2]) * 1000.0;
		}
		fvec[z2] = ((lmcurve_data_struct*)data)->y[z2] - fvec[z2];
	}
}


double f_PolyExp(double t, const double *p)
{
	double r;
	int z;

	r = 0;
	for (z=0;z<g_iFitDegree;z++)
		r += p[z*2] * exp(p[z*2+1]*t);
    return r;
}


void CLMWrapper::Fit_PolyExp(int degree, int n, double *x, double *y, double *par, double &r, double &integral, double *fitcurve)
{
    lm_status_struct status;
    lm_control_struct control = lm_control_double;
    control.printflags = 0; // monitor status (+1) and parameters (+2)
	int z, z2, i;
	double ya, sse, sst, t;

	g_iFitDegree = degree;

	mprintf("    Performing Levenberg-Marquardt fit with %d-exponential curve...\n",g_iFitDegree);
	mprintf("      f(x) = c1 * exp(e1*x) + c2 * exp(e2*x) + ...\n");
	mprintf("      Starting values:\n");
 	for (z=0;z<g_iFitDegree;z++)
		mprintf("        c%-2d  = % 11.6f      e%-2d = % 11.5f\n",z+1,par[z*2],z+1,par[z*2+1]);

//	lmcurve_fit(g_iFitDegree*2, par, n, x, y, f_PolyExp, &control, &status );
    lmcurve_data_struct data = { x, y };

    lmmin( g_iFitDegree*2, par, n, (const void*) &data,
           lmcurve_evaluate, &control, &status, lm_printout_std );

	for (z=0;z<g_iFitDegree;z++)
		if (par[z*2] < 0)
			par[z*2] = 0;

	if (fitcurve != NULL)
	{
		for (z=0;z<n;z++)
			fitcurve[z] = f_PolyExp(x[z],par);
	}

	ya = 0;
	for (z=0;z<n;z++)
		ya += y[z];
	ya /= n;

	sse = 0;
	sst = 0;

	for (z=0;z<n;z++)
	{
		sse += (y[z]-f_PolyExp(x[z],par)) * (y[z]-f_PolyExp(x[z],par));
		sst += (y[z]-ya) * (y[z]-ya);
	}

//	mprintf("    sse = %.10G,  sst = %.10G,  sse/sst = %.10G\n",sse,sst,sse/sst);

	r = sqrt(1.0 - sse/sst);

//		par[z*2+1] = -(par[z*2+1] * par[z*2+1]);

	// Stack-Sort nach dem groessten Tau
	for (z=0;z<degree-1;z++)
	{
		t = -9e20;
		i = -1;
		for (z2=z;z2<degree;z2++)
		{
			if (-par[z2*2]/par[z2*2+1] > t)
			{
				t = -par[z2*2]/par[z2*2+1];
				i = z2;
			}
		}
		if ((i != -1) && (i != z))
		{
			t = par[z*2];
			par[z*2] = par[i*2];
			par[i*2] = t;
			t = par[z*2+1];
			par[z*2+1] = par[i*2+1];
			par[i*2+1] = t;
		}
	}

	mprintf("      Status: %s (%d cycles)\n",lm_infmsg[status.info],status.nfev);
	for (z=0;z<g_iFitDegree;z++)
	{
		mprintf("        c%-2d  = % 11.6f      e%-2d = % 11.6f",z+1,par[z*2],z+1,par[z*2+1]);
		if ((par[z*2+1] <= 0) && (par[z*2] >= 0))
			mprintf("    tau(%d) = %.8G ps",z+1,-par[z*2]/par[z*2+1]);
		mprintf("\n");
	}
	mprintf("        R    = % 13.9f",r);
	mprintf("  Chi^2 = % 12.8f\n",status.fnorm);
	integral = 0;
	for (z=0;z<g_iFitDegree;z++)
	{
		if (par[z*2+1] > 0)
		{
			mprintf("      Curve integral is infinite (not all exponents <= 0)!\n");
			integral = -1;
			goto _noint;
		}
		integral -= par[z*2]/par[z*2+1];
	}
	mprintf("      Curve integral: tau(total) = %.8G ps.\n",integral);
_noint:
	mprintf("    Fitting done.\n\n");
}

