/*
 *  TouchFreezeUI.cpp
 *
 *  Copyright (C) 2007, 2008  Stefan Kombrink
 *  Copyright (C) 2008 Pau Garcia i Quiles
 *
 *  This file is part of TouchFreeze.
 *
 *  TouchFreeze is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  TouchFreeze is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with touchfreeze; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 *
 *  On Debian GNU/Linux systems, the complete text of the GNU General
 *  Public License can be found in the /usr/share/common-licenses/GPL file.
 */


#include <QtGui>

#include "TouchFreezeUI.h"
#include "Version.h"

TouchFreezeUI::TouchFreezeUI( QWidget* parent ) : QWidget( parent ), Ui::TouchFreezeUIBase()
{
  setupUi( this );

  systray = new QSystemTrayIcon( this );

  systray->setIcon( QIcon( ":/stuff/touchpad.svg" ) );
  systray->show();

  QMenu* trayMenu = new QMenu( this );
  mEnableTouchpadAction = new QAction( tr( "&Enable touchpad" ), this );
  mEnableTouchpadAction->setCheckable( true );
  mEnableTouchpadAction->setChecked( true );
  QAction* quitAction = new QAction( tr( "&Quit" ), this );

  connect( mEnableTouchpadAction, SIGNAL( toggled(bool) ), qApp, SLOT( setTouchpad(bool) ) );
  connect( mEnableTouchpadAction, SIGNAL( toggled(bool) ), qApp, SLOT( setTouchpad(bool) ) );
  connect( quitAction, SIGNAL( triggered() ), qApp, SLOT( quit() ) );

  trayMenu->addAction( mEnableTouchpadAction );
  trayMenu->addAction( quitAction );
  systray->setContextMenu( trayMenu );

  systray->setToolTip( qAppName() + " - "+ tr( "A utility to avoid touch pad troubles" ) );

  connect( systray, SIGNAL( activated( QSystemTrayIcon::ActivationReason ) ),
           this, SLOT( onSystrayActivation( QSystemTrayIcon::ActivationReason ) ) );


  connect( delaySlider, SIGNAL( valueChanged( int ) ), this, SLOT( onSliderValueChange( int ) ) );
  connect( delaySpinBox, SIGNAL( valueChanged( double ) ), this, SLOT( onSpinBoxValueChange( double ) ) );

  connect( cancelPushButton, SIGNAL( clicked() ), this, SLOT( onCancel() ) );
  connect( applyPushButton, SIGNAL( clicked() ), this, SLOT( onApply() ) );
  
  connect( useDelayCB, SIGNAL( toggled( bool ) ), this, SLOT( onUseDelayChanged( bool ) ) );
  connect( enableTouchPadCB, SIGNAL( toggled( bool ) ), this, SLOT( onEnableTouchPadChanged( bool ) ) );
}

void TouchFreezeUI::setup( const QString& tpStr, const QString& clientStr, unsigned int delay, bool daemonActive, bool touchPadActive )
{
  QString verStr = tr( "version" );
  QString appInfoStr = "<b>" + QString( "<a href='http://qsynaptics.sf.net'>%1</a>").arg(qAppName()) + "</b> " + verStr + " " + VER_STR + "<br>";
  appInfoStr += tr( "using synaptics driver" ) + " " + tpStr + "<br>" + tr( "and" );
  appInfoStr += " <b>" + clientStr + "</b><br>";
  appInfoStr += "(c) 2008 <i>Stefan Kombrink</i>";

  appInfoLabel->setText( appInfoStr );

  setWindowTitle( qAppName() + " " + VER_STR );

  delayGroupBox->setTitle( tr( "delay in seconds" ) );

  delaySlider->setMinimum( 10 );
  delaySlider->setMaximum( 2000 );
  delaySlider->setSingleStep( 200 );
  delaySlider->setTickInterval( 200 );

  const double FACTOR = 0.001;

  delaySpinBox->setMinimum( delaySlider->minimum() * FACTOR );
  delaySpinBox->setMaximum( delaySlider->maximum() * FACTOR );
  delaySpinBox->setValue( delaySlider->value() * FACTOR );
  delaySpinBox->setSingleStep( delaySlider->singleStep() * FACTOR );

  applyPushButton->setText( tr( "A&pply" ) );
  applyPushButton->setDisabled( true );

  cancelPushButton->setText( tr( "C&ancel" ) );
  
  updateGUI( delay, daemonActive, touchPadActive );
}

void TouchFreezeUI::updateGUI( unsigned int delay, bool daemonActive, bool touchPadActive )
{
    const double FACTOR = 0.001;
    
    delaySlider->setValue( delay );
    delaySpinBox->setValue( delay * FACTOR );
    
    useDelayCB->setChecked( daemonActive );
    
    delayGroupBox->setEnabled( touchPadActive );
    applyPushButton->setEnabled( touchPadActive );
    
    if (touchPadActive) setEnabledIcon(); else setDisabledIcon();
    enableTouchPadCB->setChecked( touchPadActive );
    
    mEnableTouchpadAction->setChecked(touchPadActive);
}

void TouchFreezeUI::onSliderValueChange( int value )
{
    const double FACTOR = 0.001;
    
    delaySpinBox->setValue( value * FACTOR );
    applyPushButton->setEnabled( true );
}

void TouchFreezeUI::onSpinBoxValueChange( double value )
{
  const int FACTOR = 1000;

  delaySlider->setValue( qRound( value * FACTOR ) );
  applyPushButton->setEnabled( true );
}

void TouchFreezeUI::closeEvent( QCloseEvent* e )
{
  hide();
  e->ignore();
}

void TouchFreezeUI::onSystrayActivation( QSystemTrayIcon::ActivationReason reason )
{
  if ( reason == QSystemTrayIcon::Trigger ) setVisible( !isVisible() );
}

void TouchFreezeUI::onCancel()
{
  hide();
  applyPushButton->setDisabled( true );
  emit cancelled();
}

void TouchFreezeUI::onApply()
{
  hide();
  applyPushButton->setDisabled( true );
  mEnableTouchpadAction->setChecked( enableTouchPadCB->isChecked());
  emit delayChanged( delaySlider->value() );
  emit useDelayChanged( useDelayCB->isChecked() );
  emit enableTouchPadChanged( enableTouchPadCB->isChecked() );
  emit applied();
}

void TouchFreezeUI::setDisabledIcon() {
  systray->setIcon( QIcon( ":/stuff/touchpad-disabled.svg" ) );
}

void TouchFreezeUI::setEnabledIcon() {
  systray->setIcon( QIcon( ":/stuff/touchpad.svg" ) );	
}

void TouchFreezeUI::onUseDelayChanged( bool isChecked )
{
    delaySlider->setEnabled( isChecked );
    delaySpinBox->setEnabled( isChecked );
    applyPushButton->setEnabled( true );
}

void TouchFreezeUI::onEnableTouchPadChanged( bool b )
{
    delayGroupBox->setEnabled( b );
    applyPushButton->setEnabled( true );
}
