// Copyright (C) 2002 Zbigniew Leyk (zbigniew.leyk@anu.edu.au)
//                and David E. Stewart (david.stewart@anu.edu.au)
//                and Ronan Collobert (collober@iro.umontreal.ca)
//                
//
// This file is part of Torch. Release II.
// [The Ultimate Machine Learning Library]
//
// Torch is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// Torch is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Torch; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include "mx_givens.h"

namespace Torch {

/*
		Files for matrix computations

	Givens operations file. Contains routines for calculating and
	applying givens rotations for/to vectors and also to matrices by
	row and by column.
*/

/* givens -- returns c,s parameters for Givens rotation to
		eliminate y in the vector [ x y ]' */
void mx_givens(real x, real y, real * c, real * s)
{
  real norm = sqrt(x * x + y * y);
  if (norm == 0.0)
  {
    *c = 1.0;
    *s = 0.0;
  }				/* identity */
  else
  {
    *c = x / norm;
    *s = y / norm;
  }
}

/* rot_vec -- apply Givens rotation to x's i & k components */
void mx_rot_vec(Vec * x, int i, int k, real c, real s, Vec * out)
{
  out->copy(x);

  real temp = c * out->ptr[i] + s * out->ptr[k];
  out->ptr[k] = -s * out->ptr[i] + c * out->ptr[k];
  out->ptr[i] = temp;
}

/* rot_rows -- premultiply mat by givens rotation described by c,s */
void mx_rot_rows(Mat * mat, int i, int k, real c, real s, Mat * out)
{
  out->copy(mat);

  for (int j = 0; j < mat->n; j++)
  {
    real temp = c * out->ptr[i][j] + s * out->ptr[k][j];
    out->ptr[k][j] = -s * out->ptr[i][j] + c * out->ptr[k][j];
    out->ptr[i][j] = temp;
  }
}

/* rot_cols -- postmultiply mat by givens rotation described by c,s */
void mx_rot_cols(Mat * mat, int i, int k, real c, real s, Mat * out)
{
  out->copy(mat);

  for (int j = 0; j < mat->m; j++)
  {
    real temp = c * out->ptr[j][i] + s * out->ptr[j][k];
    out->ptr[j][k] = -s * out->ptr[j][i] + c * out->ptr[j][k];
    out->ptr[j][i] = temp;
  }
}

}

