// Copyright (C) 2002 Ronan Collobert (collober@iro.umontreal.ca)
//                
//
// This file is part of Torch. Release II.
// [The Ultimate Machine Learning Library]
//
// Torch is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// Torch is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Torch; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include "general.h"

namespace Torch {

bool is_really_torch_in_native_mode = true;
char xxpetit_message_pour_melanie[10000];

bool isLittleEndian()
{
  int x = 7;
  char *ptr = (char *)&x;

  if(ptr[0] == 0)
    return(false);
  else
    return(true);
}

bool isNativeMode()
{
  return(is_really_torch_in_native_mode);
}

void setNativeMode()
{
  is_really_torch_in_native_mode = true;
}

void setLittleEndianMode()
{
  if(!isLittleEndian())
    is_really_torch_in_native_mode = false;
}

void setBigEndianMode()
{
  if(isLittleEndian())
    is_really_torch_in_native_mode = false;
}

void reverseMemory(void *ptr_, int block_size, int n_blocks)
{
  char *ptr = (char *)ptr_;
  char *ptrr, *ptrw;
  char *buff = (char *)xalloc(block_size);

  for(int i = 0; i < n_blocks; i++)
  {
    ptrr = ptr + ((i+1)*block_size);
    ptrw = buff;

    for(int j = 0; j < block_size; j++)
    {
      ptrr--;
      *ptrw++ = *ptrr;
    }

    ptrr = buff;
    ptrw = ptr + (i*block_size);
    for(int j = 0; j < block_size; j++)
      *ptrw++ = *ptrr++;
  }

  free(buff);
}

int xfread(void *ptr, int block_size, int n_blocks, FILE *f)
{
  int xx = fread(ptr, block_size, n_blocks, f);

  if(!is_really_torch_in_native_mode)
    reverseMemory(ptr, block_size, n_blocks);

  return(xx);
}

int xfwrite(void *ptr, int block_size, int n_blocks, FILE *f)
{
  if(!is_really_torch_in_native_mode)
    reverseMemory(ptr, block_size, n_blocks);

  int xx = fwrite(ptr, block_size, n_blocks, f);

  if(!is_really_torch_in_native_mode)
    reverseMemory(ptr, block_size, n_blocks);

  return(xx);
}

void *xcalloc(int n_blocks, int size)
{
  if( (size <= 0) || (n_blocks <= 0) )
    return(NULL);

  void *ptr = calloc(n_blocks, size);
  if(!ptr)
    error("general: not enough memory. Buy new ram");
  return(ptr);
}

void *xalloc(int size)
{
//  void *ptr = memalign(8, size);
  if(size <= 0)
    return(NULL);

  void *ptr = malloc(size);
  if(!ptr)
    error("general: not enough memory. Buy new ram");
  return(ptr);
}

void *xrealloc(void *ptr, int size)
{
  if(size < 0)
  {
    warning("general: trying to reallocate a block with a negative size");
    return(NULL);
  }

  void *ptrx = NULL;

  if(size == 0)
    free(ptr);
  else
  {
    ptrx = realloc(ptr, size);
    if(!ptrx)
      error("general: not enough memory. Buy new ram");
  }

  return(ptrx);
}

void error(const char* msg, ...)
{
  va_list args;
  va_start(args,msg);
  vsprintf(xxpetit_message_pour_melanie, msg, args);
  printf("\n$ Error: %s\n\n", xxpetit_message_pour_melanie);
  fflush(stdout);
  va_end(args);
  exit(-1);
}

void warning(const char* msg, ...)
{
  va_list args;
  va_start(args,msg);
  vsprintf(xxpetit_message_pour_melanie, msg, args);
  printf("! Warning: %s\n", xxpetit_message_pour_melanie);
  fflush(stdout);
  va_end(args);
}

void message(const char* msg, ...)
{
  va_list args;
  va_start(args,msg);
  vsprintf(xxpetit_message_pour_melanie, msg, args);
  printf("# %s\n", xxpetit_message_pour_melanie);
  fflush(stdout);
  va_end(args);
}

void print(const char* msg, ...)
{
  va_list args;
  va_start(args,msg);
  vsprintf(xxpetit_message_pour_melanie, msg, args);
  printf("%s", xxpetit_message_pour_melanie);
  fflush(stdout);
  va_end(args);
}

void controlBar(int level, int max_level)
{
  if(level == -1)
    print("[");
  else
  {
    if(max_level < 10)
      print(".");
    else
    {
      if( !(level % (max_level/10) ) )
        print(".");
    }
  
    if(level == max_level-1)
      print("]\n");
  }
}

int sparseVectorLength(sreal *line)
{
  int i = 0;
  while(line->index != -1)
  {
    i++;
    line++;
  }
  return(i);
}

long getRuntime(void) 
{
  struct tms buffer;
  times(&buffer);
  return((long)(buffer.tms_utime));//+buffer.tms_stime));
}

}

