// Copyright (C) 2002 Ronan Collobert (collober@iro.umontreal.ca)
//                
//
// This file is part of Torch. Release II.
// [The Ultimate Machine Learning Library]
//
// Torch is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// Torch is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Torch; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include "IOTorch.h"

namespace Torch {

#ifdef USEDOUBLE
#define REAL_FORMAT "%lf"
#else
#define REAL_FORMAT "%f"
#endif

IOTorch::IOTorch()
{
  blocks = NULL;
  n_blocks = 0;
}

int IOTorch::addBlkMem(void *internal_alloc, void **data, void **targets, int n_examples, bool data_is_internal, bool targets_is_internal)
{
  blocks = (BlkMem *)xrealloc((void *)blocks, (n_blocks+1)*sizeof(BlkMem));
  blocks[n_blocks].internal_alloc = internal_alloc;
  blocks[n_blocks].data = data;
  blocks[n_blocks].targets = targets;
  blocks[n_blocks].n_examples = n_examples;
  blocks[n_blocks].data_is_internal = data_is_internal;
  blocks[n_blocks].targets_is_internal = targets_is_internal;
  n_blocks++;
  return((n_blocks-1));
}

void IOTorch::destroyBlkMem(int number)
{
  free(blocks[number].internal_alloc);
  if(!blocks[number].data_is_internal)
  {
    void **ptr = blocks[number].data;
    for(int t = 0; t < blocks[number].n_examples; t++)
    {
      free(*ptr);
      ptr++;
    }
  }
  free(blocks[number].data);

  if(!blocks[number].targets_is_internal)
  {
    void **ptr = blocks[number].targets;
    for(int t = 0; t < blocks[number].n_examples; t++)
    {
      free(*ptr);
      ptr ++;
    }
  }
  free(blocks[number].targets);

  if(number != (n_blocks-1) )
    memmove(&blocks[number], &blocks[number+1], (n_blocks-number-1)*sizeof(BlkMem));

  blocks = (BlkMem *)xrealloc((void *)blocks, (n_blocks-1)*sizeof(BlkMem));
  n_blocks--;

  if(n_blocks == 0)
    blocks = NULL;
}

int IOTorch::loadData(const char *file, real ***data_, real ***y_,
                       int n_inputs, int n_targets, int *n_examples, bool bin, int max_load)
{  
  real **data = NULL;
  real **y = NULL;
  real *internal_alloc = NULL;
  int l, w;
  FILE *f;

  f = fopen(file, "r");
  if(!f)
    error("IOTorch: cannot open the file <%s> for reading", file);

  if(bin)
  {
    xfread(&l, sizeof(int), 1, f);
    xfread(&w, sizeof(int), 1, f);
  }
  else
  {
    fscanf(f, "%d", &l);
    fscanf(f, "%d", &w);
  }

  if( (max_load > 0) && (max_load < l) )
  {
    l = max_load;
    message("IOTorch: loading only %d examples", l);
  }

  if( (n_inputs+n_targets) != w )
    error("IOTorch: %d inputs, %d targets != %d columns in the file", n_inputs, n_targets, w);

  internal_alloc = (real *)xalloc(w*l*sizeof(real));

  if(n_inputs)
  {
    data = (real **)xalloc(l*sizeof(real *));
    for(int i = 0; i < l; i++)
      data[i] = &internal_alloc[n_inputs*i];
  }

  if(n_targets)
  {
    y = (real **)xalloc(l*sizeof(real *));
    for(int i = 0; i < l; i++)
      y[i] = &internal_alloc[n_inputs*l+n_targets*i];
  }
  
  if(bin)
  {
    for(int i = 0; i < l; i++)
    {
      if(n_inputs)
        xfread(data[i], sizeof(real), n_inputs, f);
      if(n_targets)
        xfread(y[i], sizeof(real), n_targets, f);
    }
  }
  else
  {
    for(int i = 0; i < l; i++)
    {
      for(int j = 0; j < n_inputs; j++)
        fscanf(f, REAL_FORMAT, &data[i][j]);
      
      for(int j = 0; j < n_targets; j++)
        fscanf(f, REAL_FORMAT, &y[i][j]);
    }
  }

  fclose(f);

  *n_examples = l;

  if(n_inputs)
    *data_ = data;

  if(n_targets)
    *y_ = y;

  return(addBlkMem(internal_alloc, (void **)data, (void **)y, l, true, true));
}

int IOTorch::loadData(const char *file, sreal ***data_, real ***y_,
                       int n_inputs, int n_targets, int *n_examples, bool bin, int max_load)
{
  sreal **data = NULL;
  real **y = NULL;
  real *internal_alloc = NULL;
  int l, w;
  FILE *f;

  f = fopen(file, "r");
  if(!f)
    error("IOTorch: cannot open the file <%s> for reading", file);

  if(bin)
  {
    xfread(&l, sizeof(int), 1, f);
    xfread(&w, sizeof(int), 1, f);
  }
  else
  {
    fscanf(f, "%d", &l);
    fscanf(f, "%d", &w);
  }

  if( (max_load > 0) && (max_load < l) )
  {
    l = max_load;
    message("IOTorch: loading only %d examples", l);
  }

  if( (n_inputs+n_targets) != w )
    error("IOTorch: %d inputs, %d targets != %d columns in the file", n_inputs, n_targets, w);

  
  if(n_targets)
    internal_alloc = (real *)xalloc(n_targets*l*sizeof(real));

  if(n_inputs)
    data = (sreal **)xalloc(l*sizeof(sreal *));
  
  if(n_targets)
  {
    y = (real **)xalloc(l*sizeof(real *));
    for(int t = 0; t < l; t++)
      y[t] = &internal_alloc[n_targets*t];
  }

  sreal *temp = (sreal *)xalloc(w*sizeof(sreal));
  int nt_in;

  int w_on_line;
  for(int i = 0; i < l; i++)
  {
    if(bin)
    {
      xfread(&w_on_line, sizeof(int), 1, f);
      for(int j = 0; j < w_on_line; j++)
      {        
        xfread(&temp[j].index, sizeof(int),  1, f);
        xfread(&temp[j].value, sizeof(real), 1, f);
      }
    }
    else
    {
      fscanf(f, "%d", &w_on_line);
      for(int j = 0; j < w_on_line; j++)
      {
        fscanf(f, "%d", &temp[j].index);
        fscanf(f, REAL_FORMAT, &temp[j].value);
      }
    }
    
    nt_in = 0;
    for(int j = 0; j < w_on_line; j++)
    {
      if(temp[j].index < n_inputs)
        nt_in++;
      else
        break;
    }
    
    if(n_inputs)
    {
      data[i] = (sreal *)xalloc((nt_in+1)*sizeof(sreal));
      data[i][nt_in].index = -1;
    }
    
    for(int j = 0; j < nt_in; j++)
    {
      data[i][j].index = temp[j].index;
      data[i][j].value = temp[j].value;
    }
    
    
    for(int j = 0; j < n_targets; j++)
      y[i][j] = 0;
    
    for(int j = nt_in; j < w_on_line; j++)
      y[i][temp[j].index-n_inputs] = temp[j].value;
  }

  free(temp);
  fclose(f);

  *n_examples = l;

  if(n_inputs)
    *data_ = data;

  if(n_targets)
    *y_ = y;

  return(addBlkMem(internal_alloc, (void **)data, (void **)y, l, false, true));
}

//////////////// Save

void IOTorch::saveData(const char *file, real **data, real **y,
                       int l, int n_inputs, int n_targets, bool bin, int max_save)
{
  FILE *f;
  f = fopen(file, "w");

  if(!f)
    error("IOTorch: cannot open the file <%s> for writing", file);

  int w = n_inputs + n_targets;
  if(bin)
  {
    xfwrite(&l, sizeof(int), 1, f);
    xfwrite(&w, sizeof(int), 1, f);
  }
  else
    fprintf(f, "%d %d\n", l, w);
  
  if( (max_save > 0) && (max_save < l) )
  {
    l = max_save;
    message("IOTorch: saving only %d examples", l);
  }
  
  if(bin)
  {
    for(int i = 0; i < l; i++)
    {
      if(n_inputs)
        xfwrite(data[i], sizeof(real), n_inputs, f);
      if(n_targets)
        xfwrite(y[i], sizeof(real), n_targets, f);
    }
  }
  else
  {
    for(int i = 0; i < l; i++)
    {
      for(int j = 0; j < n_inputs; j++)
        fprintf(f, "%g ", data[i][j]);

      for(int j = 0; j < n_targets; j++)
        fprintf(f, "%g ", y[i][j]);

      fprintf(f, "\n");
    }
  }

  fclose(f);
}

void IOTorch::saveData(const char *file, sreal **data, real **y,
                       int l, int n_inputs, int n_targets, bool bin, int max_save)
{
  FILE *f;

  f = fopen(file, "w");
  if(!f)
    error("IOTorch: cannot open the file <%s> for writing", file);

  int w = n_inputs+n_targets;
  if(bin)
  {
    xfwrite(&l, sizeof(int), 1, f);
    xfwrite(&w, sizeof(int), 1, f);
  }
  else
    fprintf(f, "%d %d\n", l, w);

  if( (max_save > 0) && (max_save < l) )
  {
    l = max_save;
    message("IOTorch: saving only %d examples", l);
  }

  sreal *temp = NULL;
  if(n_targets)
    temp = (sreal *)xalloc(n_targets*sizeof(sreal));

  int w_on_line;
  int nn = 0;
  for(int i = 0; i < l; i++)
  {
    w_on_line = sparseVectorLength(data[i]);
    if(n_targets)
    {
      nn = 0;
      for(int j = 0; j < n_targets; j++)
      {
        real z = y[i][j];
        if(z != 0)
        {
          temp[nn].index = n_inputs+j;
          temp[nn].value = z;
          nn++;
        }
      }
    }
    w_on_line += nn;

    if(bin)
    {
      xfwrite(&w_on_line, sizeof(int), 1, f);

      for(int j = 0; j < w_on_line-nn; j++)
      {
        xfwrite(&data[i][j].index, sizeof(int),  1, f);
        xfwrite(&data[i][j].value, sizeof(real), 1, f);
      }

      for(int j = 0; j < nn; j++)
      {
        xfwrite(&temp[j].index, sizeof(int),  1, f);
        xfwrite(&temp[j].value, sizeof(real), 1, f);
      }      
    }
    else
    {
      fprintf(f, "%d ", w_on_line);

      for(int j = 0; j < w_on_line-nn; j++)
        fprintf(f, "%d %g ", data[i][j].index, data[i][j].value);

      for(int j = 0; j < nn; j++)
        fprintf(f, "%d %g ", temp[j].index, temp[j].value);

      fprintf(f, "\n");
    }
  }

  fclose(f);
}

IOTorch::~IOTorch()
{
  for(int i = 0; i < n_blocks; i++)
    destroyBlkMem(i);
}

}

