// Copyright (C) 2002 Ronan Collobert (collober@iro.umontreal.ca)
//                
//
// This file is part of Torch. Release II.
// [The Ultimate Machine Learning Library]
//
// Torch is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// Torch is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Torch; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#ifndef DATA_SET_INC
#define DATA_SET_INC

#include "Measurer.h"
#include "List.h"

namespace Torch {

/** Provides an interface to manipulate all kind of data.

    @author Ronan Collobert (collober@iro.umontreal.ca)
 */
class DataSet : public Object
{
  public:

    //--- internal ---
    int **subsets;
    int *n_examples_subsets;
    int n_subsets;
    //----------------

    // True if a subset of the examples is selected.
    bool select_examples;

    /** The indices of all selected examples.
        When #select_examples# is false, it contains
        the indices of all examples.
    */
    int *selected_examples;

    /** Index of the current example.
        Warning:
        \begin{itemize}
          \item don't forget to set it in #setRealExample()# !
          \item it's the \emph{real} index and not the index
                in the #selected_examples# table.
        \end{itemize}
    */
    int current_example;

    /// Targets size, for all examples.
    int n_targets;

    /// Pointer to the targets of the current example.
    void *targets;

    /// Input size, for all examples.
    int n_inputs;

    /// Pointer on the inputs of the current example.
    List *inputs;

    /** Number of examples in the dataset.
        If you're using #select_examples#, it's
        the number of selected examples.
    */
    int n_examples;
    
    /** Real number of examples in the dataset.
        It's the number of examples in memory.
        (= #n_examples# if #select_examples# is false)
    */
    int n_real_examples;

    //-----

    ///
    DataSet();

    /** Set #targets# and #inputs# to the targets and inputs
        of the example with the index #selected_examples[t]#.
    */
    void setExample(int t);

    /** Set #targets# and #inputs# to the targets and inputs
        of the example with the index #t#.
    */
    virtual void setRealExample(int t) = 0;

    /** Set a new subset.
        \begin{itemize}
          \item #subset_# (of size #n_examples_#) is a set
          of indices which define a subset of #data#.
          \item if a #pushSubset()# has been already called,
          the next #pushSubset()# defines a subset of the
          previous subset, and so on...
          \item this function set #select_examples# to #true#
          and set the read indices of the examples in
          #selected_examples#.
        \end{itemize}
    */
    void pushSubset(int *subset_, int n_examples_);

    /** Remove the last subset.
        \begin{itemize}
          \item recomputes "selected_examples".
          \item if it was the last subset, set #select_examples#
          to #false#.
        \end{itemize}
    */
    void popSubset();

    //-----

    virtual void init();
    virtual void reset();
    virtual ~DataSet();
};


}

#endif
