// Copyright (C) 2002 Samy Bengio (bengio@idiap.ch)
//                
//
// This file is part of Torch. Release II.
// [The Ultimate Machine Learning Library]
//
// Torch is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// Torch is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Torch; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#ifndef DIAGONAL_GMM_INC
#define DIAGONAL_GMM_INC

#include "Distribution.h"
#include "SeqDataSet.h"
#include "EMTrainer.h"

namespace Torch {

/** This class can be used to model Diagonal Gaussian Mixture Models.
    They can be trained using either EM (with EMTrainer) or gradient descent
    (with GMTrainer).

    @author Samy Bengio (bengio@idiap.ch)
*/
class DiagonalGMM : public Distribution
{
  public:
    /// number of Gaussians in the mixture
    int n_gaussians;

    /** prior weights of the Gaussians, used in EM to give 
        a small prior on each Gaussian
    */
    real prior_weights;

    /** optional initializations
        if nothing is given, then random, at your own risks...
        one can give a initial trainer containing a kmeans
    */
    EMTrainer* initial_kmeans_trainer;
    /// as well as a measurer of this trainer
    List* initial_kmeans_trainer_measurers;
    /// or one can give an initial parameter List
    List* initial_params;
    /// or one can give an initial file
    char* initial_file;

    /// the pointers to the parameters
    real* log_weights;
    real** means;
    real** var;

    /// the pointers to the derivative of the parameters
    real* dlog_weights;
    real** dmeans;
    real** dvar;

    /// this contains the minimal value of each variance
    real* var_threshold;

    /// for each frame, for each gaussian, keep its log probability
    real** log_probabilities_g;

    /** in order to faster the computation, we can do some "pre-computation"
        pre-computed sum_log_var + n_obs * log_2_pi
    */

    real* sum_log_var_plus_n_obs_log_2_pi;
    /// pre-computed -0.5 / var
    real** minus_half_over_var;

    /// accumulators for EM
    real** means_acc;
    real** var_acc;
    real*  weights_acc;

    ///
    DiagonalGMM(int n_observations_, int n_gaussians_, real* var_threshold_, real prior_weights_);

    virtual void reset();
    virtual int numberOfParams();
    virtual void allocateMemory();
    virtual void freeMemory();

    void generateObservation(real* observations);
    void generate(SeqDataSet* data,int n_);

    virtual void display();
    virtual void eMIterInitialize();
    virtual void iterInitialize();

    virtual real frameLogProbability(real *observations, real *inputs, int t);

    /// this method returns the log probability of the "g" Gaussian
    virtual real frameLogProbabilityOneGaussian(real *observations, real *inputs, int g);

    virtual void sequenceInitialize(List* inputs);
    virtual void eMSequenceInitialize(List* inputs);
    virtual void frameEMAccPosteriors(real *observations, real log_posterior, real *inputs, int t);
    virtual void eMUpdate();

    virtual void frameBackward(real *observations, real *alpha, real *inputs, int t);
    virtual void frameExpectation(real *observations, real *inputs, int t);

    virtual void setNGaussians(int n_gaussians_);

    virtual ~DiagonalGMM();
};


}

#endif
