-- Topal: GPG/GnuPG and Alpine/Pine integration
-- Copyright (C) 2001--2008  Phillip J. Brooke
--
-- This program is free software: you can redistribute it and/or modify
-- it under the terms of the GNU General Public License version 3 as
-- published by the Free Software Foundation.
--
-- This program is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
-- GNU General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program.  If not, see <http://www.gnu.org/licenses/>.

with Ada.Text_IO;
with Ada.Unchecked_Deallocation;

package body Expanding_Array is

   procedure Actual_Array_Free
   is new Ada.Unchecked_Deallocation (Actual_Array, Actual_Array_Pointer);

   procedure Natural_Free
   is new Ada.Unchecked_Deallocation (Natural, Natural_Pointer);

   procedure Positive_Free
   is new Ada.Unchecked_Deallocation (Positive, Positive_Pointer);

   procedure Initialize (Object : in out Big_Array) is
   begin
      Object.Ref_Count := new Natural;
      Object.Ref_Count.all := 1;
   end Initialize;

   procedure Adjust (Object : in out Big_Array) is
   begin
      Object.Ref_Count.all := Object.Ref_Count.all + 1;
   end Adjust;

   procedure Finalize (Object : in out Big_Array) is
   begin
      Object.Ref_Count.all := Object.Ref_Count.all - 1;
      begin
         if Object.Ref_Count.all = 0 then
            Actual_Array_Free(Object.Actual);
            Natural_Free(Object.Ref_Count);
            Positive_Free(Object.Array_Size);
         end if;
      exception
         when others =>
            Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                                 "AIEEE! Exception raised in Expanding_Array.Finalise");
      end;
   end Finalize;

   procedure Create (Left         : in out Big_Array;
                     Initial_Size : in     Positive) is
      A : Big_Array;
   begin
      A.Actual := new Actual_Array(1..Initial_Size);
      A.Array_Size := new Positive;
      A.Array_Size.all := Initial_Size;
      Left := A;
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Expanding_Array.Create");
         raise;
   end Create;

   procedure Copy (Left  : in out Big_Array;
                   Right : in     Big_Array) is
      A : Big_Array;
   begin
      -- Now, deep copy the array.
      A.Actual := new Actual_Array(1..Size(Right));
      A.Array_Size := new Positive;
      A.Array_Size.all := Size(Right);
      A.Actual.all(1..A.Array_Size.all) := Right.Actual.all(1..Right.Array_Size.all);
      Left := A;
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Expanding_Array.Copy");
         raise;
   end Copy;

   function Size (A : Big_Array) return Positive is
   begin
      return A.Array_Size.all;
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Expanding_Array.Size");
         raise;
   end Size;

   function Last (A : Big_Array) return Positive is
   begin
      return Size(A);
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Expanding_Array.Last");
         raise;
   end Last;

   procedure Expand(A : in out Big_Array;
                    S : in     Positive) is
      New_Actual : Actual_Array_Pointer;
   begin
      New_Actual := new Actual_Array(1..A.Array_Size.all + S);
      New_Actual.all(1..A.Array_Size.all) := A.Actual.all(1..A.Array_Size.all);
      Actual_Array_Free(A.Actual);
      A.Actual := New_Actual;
      A.Array_Size.all := A.Array_Size.all + S;
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Expanding_Array.Expand");
         raise;
   end Expand;

   function First (A : Big_Array) return Natural is
   begin
      return A.Actual.all'First;
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Expanding_Array.First");
         raise;
   end First;

   procedure Set(A     : in out Big_Array;
                 Index : in     Positive;
                 Value : in     Item) is
   begin
      -- Is the array large enough?
      while Last(A) < Index loop
         -- Double the size....
         Expand(A, A.Array_Size.all);
      end loop;
      -- Set the value.
      if A.Actual = null then
         raise Actual_Is_Null;
      else
         A.Actual(Index) := Value;
      end if;
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Expanding_Array.Set");
         raise;
   end Set;

   function Value(A     : Big_Array;
                  Index : Positive) return Item is
   begin
      return A.Actual(Index);
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Expanding_Array.Value");
         raise;
   end Value;


end Expanding_Array;
