/*
 * TLPR.C -- a Trivial LPR client.
 * Copyright (c) Wouter Verhelst, 2002 <wouter@debian.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or any
 * later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA.
 *
 * $Log: tlpr.c,v $
 * Revision 1.16  2003/05/23 14:34:56  wouter
 * i18n stuff
 *
 * Revision 1.14  2003/05/05 19:56:51  wouter
 * use '-l' instead of '-f' for form-feed. Think '^L'.
 * Check returnvalue for fopen(), too.
 *
 * Revision 1.13  2003/05/05 17:40:23  wouter
 * Added formfeed option; made sure the thing doesn't segfault on getpwuid.
 *
 * Revision 1.12  2003/04/01 17:51:49  wouter
 * philip-changes
 *
 * Revision 1.10  2002/12/28 13:51:12  wouter
 * memset() buf to zero before sending the file out to the printer.
 *
 * Revision 1.9  2002/12/11 22:14:21  wouter
 * fixed some minor bugs
 *
 * Revision 1.8  2002/12/01 13:49:24  wouter
 * * Read an environment variable for defaults wrt printer/ip
 * * check the size of the input before sending it
 *
 * Revision 1.7  2002/11/30 13:23:04  wouter
 * Code cleanup
 *
 * Revision 1.6  2002/11/29 23:21:31  wouter
 * Versie gecorrigeerd. CVS revisie != versie.
 *
 * Revision 1.5  2002/11/29 23:20:45  wouter
 * source autoconf'ed; terug naar non-filter versie van command file gegaan.
 *
 * Revision 1.4  2002/11/29 22:32:08  wouter
 * Final bugfixes
 *
 * Revision 1.3  2002/11/29 07:48:05  wouter
 * Some bugfixes. More are needed.
 *
 * Revision 1.2  2002/11/28 23:10:53  wouter
 * Some printf output fixed, and an exit(0) when only asking the version added.
 *
 * Revision 1.1.1.1  2002/11/28 19:41:27  wouter
 * Initial import
 *
 */

#ifndef lint
static const char rcsid[]= "$Id: tlpr.c,v 1.16 2003/05/23 14:34:56 wouter Exp $";
#endif /* !lint */

#include "config.h"

/*
 * struct sockaddr_in is defined in netinet/in.h on
 * BSD systems.  Probably the same elsewhere?
 */
#include <netinet/in.h>

#include <sys/types.h>
#include <sys/socket.h>

#include <errno.h>
#include <getopt.h>

#include <netdb.h>
#include <pwd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

/* i18n */
#include "gettext.h"
#define _(String) gettext(String)

#define TLPRVERSION "0.1.4, $Revision: 1.16 $"
#define LPR_DAEMON_PRINT_WAITING (unsigned char)1
#define LPR_DAEMON_RECEIVE_JOB (unsigned char)2
#define LPR_JOB_ABORT (unsigned char)1
#define LPR_JOB_CONTROL (unsigned char)2
#define LPR_JOB_DATA (unsigned char)3

int sendbuf(char* buf, int sock, int len) 
{
	int total = 0;		/* how many bytes we've sent		*/
	int bytesleft = len;	/* how many we have left to send 	*/
	int n;
	
	while(total < len) 
	{
		n = send(sock, buf+total, bytesleft, 0);
		if (n == -1) { break; }
		total += n;
		bytesleft -= n;
	}
	
	len = total;
	
	return n==-1?-1:0;	/*  return -1 on failure, 0 on success	*/
}

int construct_cmdfile(char* cmdfile, int jobnum, char** hname) 
{
	struct passwd passwd;
	struct passwd*ptr;
	char buf[1024];
	char myuname[32];
	uid_t uid;
	
	*hname=malloc(32);
	if (gethostname(*hname, 32)==-1) 
	{
		perror("gethostname()");
		fprintf(stderr, _("Not fatal; continuing\n"));
		strcpy(*hname, "localhost");
	}
	
	uid=getuid();
	if(getpwuid_r(uid, &passwd, buf, 1024, &ptr))
	{
		perror("getpwuid(getuid())");
		fprintf(stderr, _("Not fatal; continuing\n"));
		strcpy(myuname, "unknown");
	} 
	else
		strncpy(myuname, ptr->pw_name, 31);
	
	sprintf(cmdfile, "H%s\nP%s\nldfA%03d%s\n", *hname, myuname, jobnum, *hname);
}

/* Open a socket */
int initsocket(char* ip) {
	int sock;
	int port;
	int ok=0;
	struct sockaddr_in remote_addr;
	struct hostent*h;

	sock=socket(AF_INET, SOCK_STREAM, 0);
	for(port=721;!ok && port<=731;port++) 
	{
		struct sockaddr_in my_addr;
		my_addr.sin_family=AF_INET;
		my_addr.sin_port=htons(port);
		my_addr.sin_addr.s_addr=INADDR_ANY;
		memset(&(my_addr.sin_zero), '\0', 8);
		if(!bind(sock, (struct sockaddr*)&my_addr, sizeof(struct sockaddr))) 
			ok=1;
		else 
		{ 
			/* 
			 * We don't have permission to run on a privileged port. 
			 * Don't bind 
			 */
			if (errno==EACCES)
				ok=1;
		}
	}

	remote_addr.sin_family=AF_INET;
	remote_addr.sin_port=htons(515);
	if((h=gethostbyname(ip)) == NULL) {
		herror("gethostbyname");
		exit(1);
	}
	remote_addr.sin_addr.s_addr=((struct in_addr*)h->h_addr)->s_addr;
	memset(&(remote_addr.sin_zero), '\0', 8);
	
	if(connect(sock, (struct sockaddr*)&remote_addr,
				sizeof(struct sockaddr))==-1) 
	{
		perror("connect()");
		exit(1);
	}
	return sock;
}

int kick_printer(char* printer, char* ip) {
	int sock;
	char buf[256];

	sock=initsocket(ip);
	snprintf(buf,255,"%c%s\n", LPR_DAEMON_PRINT_WAITING, printer);
	sendbuf(buf, sock, strlen(buf));
}

int sndfile(FILE* in, char* printer, char* ip, char ff)
{
	char* myhname;
	int jobnum=random()/(RAND_MAX/999);
	char* cmdfile=malloc(76);
	int cfsize;
	
	int sock;
	char* buf;

	/* initialize the connection */
	sock=initsocket(ip);
	/* Prepare the commandfile */
	construct_cmdfile(cmdfile, jobnum, &myhname);
	cfsize=strlen(cmdfile);
	
	buf=malloc(256);
	snprintf(buf, 255, "%c%s\n", LPR_DAEMON_RECEIVE_JOB, printer);
	
	if(sendbuf(buf, sock, strlen(buf)))
		perror(_("sending print command"));
	
	read(sock, buf, 256);

	if ((*buf)!='\0') 
	{
		fprintf(stderr,_("Creating a printjob failed. Giving up.\n"));
		fprintf(stderr,_("Output from lpd was: %s"), buf);
		exit(1);
	}

	snprintf(buf, 255, "%c%d cfA%03d%s\n", LPR_JOB_CONTROL, cfsize,
			jobnum, myhname);

	if(sendbuf(buf, sock, strlen(buf)))
		perror(_("sending command file subcommand"));

	read(sock, buf, 256);

	if ((*buf)!='\0') 
	{
		fprintf(stderr, _("Opening connection to send command file failed.\n"));
		fprintf(stderr, _("Output from lpd was: %s"), buf);
		exit(1);
	}

	strcpy(buf,cmdfile);

	if(sendbuf(buf, sock, strlen(buf)))
		perror(_("sending commandfile"));

	*buf='\0';
	sendbuf(buf, sock, 1);
	read(sock, buf, 256);

	if ((*buf)!='\0') 
	{
		fprintf(stderr, _("Sending command file failed.\n"));
		fprintf(stderr, _("Output from lpd was: %s"), buf);
		exit(1);
	}

	snprintf(buf, 255, "%c0 dfA%03d%s\n", LPR_JOB_DATA, jobnum, myhname);

	if(sendbuf(buf, sock, strlen(buf)))
		perror(_("sending data file subcommand"));
	
	read(sock, buf, 256);

	if ((*buf)!='\0') 
	{
		fprintf(stderr, _("Opening connection to send data file failed.\n"));
		fprintf(stderr, _("Output from lpd was: %s"), buf);
		exit(1);
	}

	while(!feof(in)) 
	{
		int n;
		memset(buf, '\0', 256);
		n=fread(buf, 1, 256, in);
		/*printf("%d", n);*/
	
		if(sendbuf(buf, sock, n))
			perror(_("sending data file"));
	}

	*buf=ff;
        buf[1]='\0';
	sendbuf(buf, sock, ff?1:2);
}

int main(int argc, char** argv) 
{
	/* i18n*/
#ifdef ENABLE_NLS
        setlocale(LC_ALL,"");
        bindtextdomain(PACKAGE, LOCALEDIR);
        textdomain(PACKAGE);
#endif
	struct option long_options[] = 
	{
		{ "printer", 1, 0, 'P' },
		{ "help", 0, 0, 'h' },
		{ "ip-address", 1, 0, 'i' },
		{ "kick", 0, 0, 'k' },
		{ "version", 0, 0, 'v' },
                { "add-closing-ff", 0, 0, 'l' },
		{0, 0, 0, 0}
	};

	char* pname=malloc(sizeof("lp")+1);
	char* ip=malloc(sizeof("127.0.0.1")+1);
	FILE* inputfile=stdin;
	char* tmp;
	int option_index = 0;
	int c;
        char ff='\0';
	int want_kick=0;

	strcpy(pname, "lp");
	strcpy(ip, "127.0.0.1");

	if(tmp=getenv("TLPR_DEFAULT")) 
	{
		c=strcspn(tmp,":");
		if(c==strlen(tmp)) {
			fprintf(stderr,_("Error: environment variable TLPR_DEFAULT does not contain a correct value!\n"));
			exit(1);
		}
		free(ip);
		ip=malloc(strcspn(tmp,":")+1);
		strncpy(ip, tmp, strcspn(tmp, ":"));
		tmp+=strcspn(tmp, ":")+1;
		free(pname);
		pname=malloc(sizeof(tmp)+1);
		strcpy(pname, tmp);
	}
	
	while ((c = getopt_long(argc, argv, "P:hi:klv", long_options,
				&option_index)) != -1) 
	{
		switch(c) 
			{
			case 'P':
				/* Define the printer */
				free(pname);
				pname=malloc(strlen(optarg)+1);
				strcpy(pname, optarg);
				break;
			case 'i':
				/* Choose a host address */
				free(ip);
				ip=malloc(sizeof(optarg)+1);
				strcpy(ip, optarg);
				break;
			case 'k':
				want_kick=1;
				break;
			case 'l':
				ff=12;
				break;
			case 'v':
				printf(_("Trivial lpr version %s\n"), TLPRVERSION);
				printf(_("Copyright (c) 2002, Wouter Verhelst\n"));
				exit(0);
				break;
			default:
			case 'h':
				printf(_("Trivial lpr version %s\n"), TLPRVERSION);
				printf(_("Copyright (c) 2002, Wouter Verhelst\n"));
				printf("%s lpr [-P|--printer <%s>]\n\t[-h|--help]\n\t[-i|--ip-address <%s>]\n\t[-k|--kick]\n\t[-v|--version]\n\t[-l|--add-closing-ff]\n", _("Usage:"), _("printername"), _("IP address"));
				exit(0);
				break;
			}
	}
	
	if (optind < argc)
		if((inputfile=fopen(argv[optind], "r"))==NULL)
                {
                        perror(_("opening inputfile"));
                        exit(1);
                }

	if(!want_kick) {
		return sndfile(inputfile, pname, ip, ff);
	} else {
		return kick_printer(pname,ip);
	}
}

