#
# This script is part of TkKasse. You may use, modify and distribute it under the terms
# of the GNU General Public License, Version 2.
# See details at http://www.fsf.org/licenses/gpl.txt
# (C) by Jan Kandziora <tkkasse@users.sf.net> 
#


## This is no apperance options but an internal command used on arbitrary BillView widgets. Do not change.
option add *main.desktop.bills*TkKasseCommand BillViewHandler 99



## Application main function. 
proc main {} \
{
	## Create user and article query connection:
	uplevel #0 client::queryconnection query

	## Connect at startup.
	after 3000 \
	{
		global CONNECT_ON_STARTUP 
		if $CONNECT_ON_STARTUP queryServer
	}
}



## Handler for all errors not catched.
proc bgerror {args} \
{
	## Acoustic feedback.
	bell	
	
	## Activate the fatal error dialog.
	.oops activate
}



## Program exit handler.
rename exit tcl_exit
proc exit {} \
{
	## Activate exit dialogue.
	.exit activate
}

proc realExit {} \
{
	## Work around a bug causing a SIGSEGV at exit in Tcl 8.x .
	foreach IMAGE [ image names ] \
	{
		image delete $IMAGE
	}

	## Exit for real.
	tcl_exit
}


## Display some help on a topic.
proc help { URL_ARG } \
{
	## Put the URL into the command string.
	global HELP_COMMAND HELP_URL
	regsub -all {\\|&} $HELP_COMMAND {\\\0} SUBST
	regsub -all "%u" $SUBST "$HELP_URL$URL_ARG" SUBST

	## Call the help display application .
	catch { eval exec $SUBST & }
}


## Call for support.
proc support { args } \
{
	## Support form:
	set SUBJECT [ msgcat::mc <support.email.subject> ]
	set ADDRESS [ msgcat::mc <support.email.address> ]
	set BODY [ msgcat::mc <support.email.text> ]

	## Put the problem description into the body.
	regsub -all {\\|&} $BODY {\\\0} SUBST
	regsub -all "%s" $SUBST $args BODY

	## Put the data into the command.
	global MAIL_COMMAND
	regsub -all {\\|&} $MAIL_COMMAND {\\\0} SUBST
	regsub -all "%s" $SUBST $SUBJECT SUBST
	regsub -all "%a" $SUBST $ADDRESS SUBST

	## Have to use temporary file.
	if [ string match "*%t*" $SUBST ] \
	{
		## Yes: The email program will not read from stdin.
		set TEMPFILE "/tmp/tkkasse-client-error.[ pid ]"
		set FD [ open $TEMPFILE w ]
		puts $FD $BODY
		close $FD
		regsub -all "%t" $SUBST $TEMPFILE SUBST

		## Call the external program.
		catch { eval exec $SUBST & }
	} \
	{
		## The pipe is ok. Call the external program.
		catch { eval exec $SUBST << $BODY & }
	}
}



## Format currency values.
proc formatCurrency { PRICE_ARG } \
{
	## Get the localized currency format.
	set DECIMAL_WIDTH [ msgcat::mc <currency.format.decimals> ]
	set DECIMAL_SEPARATOR [ msgcat::mc <currency.format.decimalSeparator> ]
	set EYECATCH_WIDTH [ msgcat::mc <currency.format.eyecatch> ]
	set EYECATCH_SEPARATOR [ msgcat::mc <currency.format.eyecatchSeparator> ]

	## Delete leading minus and all leading zeroes from PRICE_ARG.
	set PRICE [ regsub {^-*0*} $PRICE_ARG {} ]
	if { $PRICE == {} } { set PRICE 0 }

	## Seperate the decimals from the non-broken part and format them.
	if { $DECIMAL_WIDTH > 0 } \
	{
		set DECIMALS [ format "$DECIMAL_SEPARATOR%0${DECIMAL_WIDTH}d" [ expr $PRICE % round(pow(10,$DECIMAL_WIDTH)) ] ]
	} \
	{
		set DECIMALS {}
	}
	
	## Get the value of the non-broken part.
	set PRICE_NONBROKEN [ expr $PRICE / round(pow(10,$DECIMAL_WIDTH)) ]

	## Insert eyecatches, if neccesary.
	set EYECATCHES {}
	set EYECATCH_COUNTER 0 
	for { set I [ string length $PRICE_NONBROKEN ] } { $I >= 0 } { set I [ expr $I-1 ] } \
	{
		set EYECATCHES "[ string index $PRICE_NONBROKEN $I ]$EYECATCHES"

		incr EYECATCH_COUNTER
		if { ($EYECATCH_COUNTER > $EYECATCH_WIDTH) && ( $I > 0 ) } \
		{
			set EYECATCH_COUNTER 1
			set EYECATCHES "$EYECATCH_SEPARATOR$EYECATCHES"
		}
	}

	## Return the formatted negative value, if the value is negative.
	if { $PRICE_ARG < 0 } { return "-$EYECATCHES$DECIMALS" } 

	## Return the formatted value.
	return " $EYECATCHES$DECIMALS"
}

## Check if the string value is a valid price in this currency.
proc checkCurrency { STRING_ARG } \
{
	## Get the localized currency format.
	set DECIMAL_WIDTH [ msgcat::mc <currency.format.decimals> ]
	set DECIMAL_SEPARATOR [ msgcat::mc <currency.format.decimalSeparator> ]
	set EYECATCH_WIDTH [ msgcat::mc <currency.format.eyecatch> ]
	set EYECATCH_SEPARATOR [ msgcat::mc <currency.format.eyecatchSeparator> ]

	## Check the string.	
	regexp [ format {^-*[[:digit:]]{1,%d}\%s?([[:digit:]]{%d}\%s?)*\%s[[:digit:]]{%d}$} $EYECATCH_WIDTH $EYECATCH_SEPARATOR $EYECATCH_WIDTH $EYECATCH_SEPARATOR $DECIMAL_SEPARATOR $DECIMAL_WIDTH ] $STRING_ARG
}

## Convert broken currency values to plain ones.
proc plainCurrency { BROKEN_VALUE_ARG } \
{
	## Get the localized currency format.
	set DECIMAL_SEPARATOR [ msgcat::mc <currency.format.decimalSeparator> ]
	set EYECATCH_SEPARATOR [ msgcat::mc <currency.format.eyecatchSeparator> ]

	## Convert the string.	
	regsub -all -- "\\$DECIMAL_SEPARATOR" $BROKEN_VALUE_ARG "" PLAIN_VALUE
	regsub -all -- "\\$EYECATCH_SEPARATOR" $PLAIN_VALUE "" PLAIN_VALUE
	if [ catch { expr [ regsub -all -- {^(-*)0*} $PLAIN_VALUE {\1} ] } RESULT ] { set RESULT 0 }

	## Return the value.
	return $RESULT
}



## Generic client connection class:
itcl::class client::connection \
{
	constructor { args } { eval configure $args } {}
	destructor {}

	method AliveHandler {}
	method connect {}
	method disconnect {}
	method ErrorHandler { CODE MESSAGE }
	method processLine {}
	method send { COMMAND }

	public \
	{
		variable host {}
		variable port {}
		variable disconnectcommand {}
	}

	protected \
	{
		variable alive_timer {}
		variable fd {}
		variable interpreter {}
	}
}

## Generic client connection class constructor: 
itcl::body client::connection::constructor { args } \
{
	## Create a safe interpreter and set some command aliases in it.
	set interpreter [ interp create -safe ]
	$interpreter alias AliveHandler $this AliveHandler
	$interpreter alias ErrorHandler $this ErrorHandler
}

## Generic client connection class destructor: 
itcl::body client::connection::destructor {} \
{
	## Cut the connection.
	disconnect

	## Delete the interpreter associated with the connection.
	interp delete $interpreter
}

## Connect to the server.
itcl::body client::connection::connect {} \
{
	## Check if we are already connected.
	if { $fd != {} } { return -code error -errorcode { TKKASSE CONNECTION CONNECTED } "already connected" }
	
	## Actually connect.
	set fd [ socket [ cget -host ] [ cget -port ] ]

	## Configure the socket for line-by-line operation.
	fconfigure $fd -buffering line

	## Register a handler for read events.
	fileevent $fd readable "if { \[ itcl::find objects $this \] != {} } { $this processLine }"

	## Register a handler for errors on the server side.
	send { setErrorHandler { ErrorHandler %c %i } }

	## Register a handler for connection
	send { doEcho AliveHandler }
}

## Disconnect form the server.
itcl::body client::connection::disconnect {} \
{
	## Cancel the alive check.
	after cancel $alive_timer

	## Close the channel.
	catch { close $fd }
	set fd {}

	## Call the handler for disconnection.
	eval [ cget -disconnectcommand ]	
}

## Send a command to the server.
itcl::body client::connection::send { COMMAND } \
{
	## Actually send the command.
	## Disconnect on fail.
	if [ catch { puts $fd $COMMAND } ] { disconnect }
}
	
## Process a line of text:
itcl::body client::connection::processLine {} \
{
	## Read the line and check if the server has closed the connection.
	if { [ gets $fd INPUT ] < 0 } \
	{
		## Yes. Close this connection.
		disconnect
	} \
	{
		## Process the read data.
		$interpreter eval $INPUT
	}
}

## Handler for connection checking.
itcl::body client::connection::AliveHandler {} \
{
	## Reset the alive check timer.
	set alive_time [ after 3000 "catch { $this send { doEcho AliveHandler } }" ]
}

## Handler for errors with the server.
itcl::body client::connection::ErrorHandler { CODE MESSAGE } \
{
	## Set code and message of the error
	global errorCode errorInfo
	set errorCode [ concat SERVER $CODE ]
	set errorInfo $MESSAGE

	## Show the fatal error dialog.
	.oops activate
}
	


## Connection class for retrieving user names and the article list:
itcl::class client::queryconnection \
{
	inherit client::connection

	constructor {args} {}

	method ArticlesUpdateHandler { ARTICLES_LIST }
	method connect {}
	method disconnect {}
	method UsersUpdateHandler { USERS_LIST }
}

## Connection class for retrieving user names and the article list constructor:
itcl::body client::queryconnection::constructor { args } \
{
	## Set handlers for callbacks by the server.
	$interpreter alias ArticlesUpdateHandler $this ArticlesUpdateHandler
	$interpreter alias UsersUpdateHandler $this UsersUpdateHandler
}

## Connect with the server.
itcl::body client::queryconnection::connect {} \
{
	## Actually connect.
	chain

	## Set handler callback commands in the server.
	send "setArticlesUpdateHandler { ArticlesUpdateHandler %a }"
	send "setUsersUpdateHandler { UsersUpdateHandler %u }"
}

## Disconnect:
itcl::body client::queryconnection::disconnect {} \
{
	## Actually disconnect.
	chain

	## Clear the article and user view.
	.main.desktop.articles configure -articles {}
	.main.desktop.users configure -users {}
}

## Handler for retrieving article list updates:
itcl::body client::queryconnection::ArticlesUpdateHandler { ARTICLES_LIST } \
{
	## Apply the new data to the article view.
	.main.desktop.articles configure -articles $ARTICLES_LIST
}

## Handler for retrieving user list updates:
itcl::body client::queryconnection::UsersUpdateHandler { USERS_LIST } \
{
	## Apply the new data to the user view.
	.main.desktop.users configure -users $USERS_LIST
}



## User connection class
## TODO: Clean up.
itcl::class client::userconnection \
{
	inherit client::connection

	constructor { args } { eval configure $args } {}

	method addArticleItemToBill { TAG KEY }
	method addArticleItemToBillAgain { TAG KEY }
	method addExtraToBillItem { TAG ITEM_UNIQUE NAME PRICE }
	method BillsUpdateHandler { BILLS_INFO_LIST SEE_TAG }
	method BillUpdateHandler { BILL_INFO }
	method cancelItemFromBill { TAG ITEM_UNIQUE }
	method changeOrderOfItemFromBill { TAG ITEM_UNIQUE }
	method changeServiceOfItemFromBill { TAG ITEM_UNIQUE }
	method changeTaxOfItemFromBill { TAG ITEM_UNIQUE }
	method cleanUpCashRegister {}
	method closeBill { TAG PAYED_BY COMMENT }
	method createBill { TABLE }
	method connect { LOGIN PASSWORD }
	method decrementBillItemMultiplier { TAG ITEM_UNIQUE }
	method deliverItemFromBill { TAG ITEM_UNIQUE }
	method disconnect {}
	method ErrorHandler { CODE MESSAGE }
	method incrementBillItemMultiplier { TAG ITEM_UNIQUE }
	method LoginHandler { USER_INFO }	
	method orderItemFromBill { TAG ITEM_UNIQUE }
	method printCashRegisterReceipt {}
	method printGuestReceiptForBill { TAG HOW }
	method printWaiterReceipt {}
	method removeExtraFromBillItem { TAG ITEM_UNIQUE EXTRA_INDEX }

	protected \
	{
		variable view {}
		variable view_key {}
	}
}

## User connection constructor:
itcl::body client::userconnection::constructor { args } \
{
	## Set handlers for callbacks by the server.
	$interpreter alias BillsUpdateHandler $this BillsUpdateHandler
	$interpreter alias BillUpdateHandler $this BillUpdateHandler
	$interpreter alias LoginHandler $this LoginHandler
}

## Add an article item to a bill.
itcl::body client::userconnection::addArticleItemToBill { TAG KEY } \
{
	send [ list addArticleItemToBill $TAG $KEY ]
}

## Add an article item to a bill again.
itcl::body client::userconnection::addArticleItemToBillAgain { TAG KEY } \
{
	send [ list addArticleItemToBillAgain $TAG $KEY ]
}

## Add an extra to a bill item.
itcl::body client::userconnection::addExtraToBillItem { TAG ITEM_UNIQUE NAME PRICE } \
{
	send [ list addExtraToBillItem $TAG $ITEM_UNIQUE $NAME $PRICE ]
}

## Handler for an update of the bills of the current user.
itcl::body client::userconnection::BillsUpdateHandler { BILLS_INFO_LIST SEE_TAG } \
{
	## Get the list of tags in the current bills view.
	set CURRENT_TAGS [ $view getViewTags ]

	## Get the list of the tags in the new bills view.
	array set BILLS_INFO_ARRAY {}
	set NEW_TAGS {}
	foreach BILL_INFO $BILLS_INFO_LIST \
	{
		set TAG [ lindex $BILL_INFO [ expr [ lsearch -exact $BILL_INFO "-tag" ] +1 ] ]
		lappend NEW_TAGS $TAG

		## Put the bill info into an array, indexed with the tag.
		set BILLS_INFO_ARRAY($TAG) $BILL_INFO
	}

	## Remove all bill views from the current bills view that are not present in the new one.
	foreach TAG $CURRENT_TAGS \
	{
		if { [ lsearch -exact $NEW_TAGS $TAG ] == -1 } \
		{
			## This bill view has to be removed.
			$view removeView $TAG
		}
	}

	## Now process all the bill views that should be present in the future.
	foreach TAG $NEW_TAGS \
	{
		## Add new bill views into the new bills view for each tag that is not present in the old one.
		if { [ lsearch -exact $CURRENT_TAGS $TAG ] == -1 } \
		{
			## This bill view has to be added.
			$view addView $TAG [ lindex $BILLS_INFO_ARRAY($TAG) [ expr [ lsearch -exact $BILLS_INFO_ARRAY($TAG) "-table" ] + 1 ] ]
		}
	
		## Now configure each bill view.
		$view configureView $TAG $BILLS_INFO_ARRAY($TAG)
	}

	## Set a tag to be on top if desired.
	if { $SEE_TAG != {} } { $view seeView $SEE_TAG }
}

## Handler for an update of a specific bill.
itcl::body client::userconnection::BillUpdateHandler { BILL_INFO } \
{
	## Configure this bills view.
	$view configureView [ lindex $BILL_INFO [ expr [ lsearch -exact $BILL_INFO "-tag" ] +1 ] ] $BILL_INFO
}

## Cancel a bill item.
itcl::body client::userconnection::cancelItemFromBill { TAG ITEM_UNIQUE } \
{
	send [ list cancelItemFromBill $TAG $ITEM_UNIQUE ]
}

## Change the order counter of a bill item.
itcl::body client::userconnection::changeOrderOfItemFromBill { TAG ITEM_UNIQUE } \
{
	send [ list changeOrderOfItemFromBill $TAG $ITEM_UNIQUE ]
}

## Change the service of a bill Item.
itcl::body client::userconnection::changeServiceOfItemFromBill { TAG ITEM_UNIQUE } \
{
	send [ list changeServiceOfItemFromBill $TAG $ITEM_UNIQUE ]
}

## Change the tax of a bill Item.
itcl::body client::userconnection::changeTaxOfItemFromBill { TAG ITEM_UNIQUE } \
{
	send [ list changeTaxOfItemFromBill $TAG $ITEM_UNIQUE ]
}

## Clean up the cash register.
itcl::body client::userconnection::cleanUpCashRegister {} \
{
	send [ list cleanUpCashRegister ]
}

## Close a bill in the cash register.
itcl::body client::userconnection::closeBill { TAG PAYED_BY COMMENT } \
{
	send [ list closeBill $TAG $PAYED_BY $COMMENT ]
}

## Create a new bill in the cash register.
itcl::body client::userconnection::createBill { TABLE } \
{
	send [ list createBill $TABLE ]
}

## Connect a users connection.
itcl::body client::userconnection::connect { LOGIN PASSWORD } \
{
	global DISPLAY

	## Actually connect.
	chain

	## Set the display for this client.
	send [ list setClientDisplay $DISPLAY ]

	## Set the login handler.
	send "setLoginHandler { LoginHandler %u }"

	## Login.
	send [ list authenticateUser $LOGIN $PASSWORD ]
}

## Decrement the multiplier of a bill item.
itcl::body client::userconnection::decrementBillItemMultiplier { TAG ITEM_UNIQUE } \
{
	send [ list decrementBillItemMultiplier $TAG $ITEM_UNIQUE ]
}

## Deliver a bill item.
itcl::body client::userconnection::deliverItemFromBill { TAG ITEM_UNIQUE } \
{
	send [ list deliverItemFromBill $TAG $ITEM_UNIQUE ]
}

## Disconnect:
itcl::body client::userconnection::disconnect {} \
{
	## Actually disconnect.
	chain

	## Close the bills view of this user. 
	catch { .main.desktop.bills delete $view_key }
}

## Handler for Errors associated with the server.
itcl::body client::userconnection::ErrorHandler { CODE MESSAGE } \
{
	## Check the error code.
	switch $CODE \
	{
		{ TKKASSE USER AUTHENTICATION_FAILURE } \
		{
			## Configure the error dialog.
			.error configure -text [ msgcat::mc <error.text.user.authenticationFailure> ]

			## Show the error dialog.
			.error activate

			## Authentication failed. Close the connection.
			itcl::delete object $this
		} \
		{ TKKASSE USER RIGHTS_INSUFFICIENT } \
		{
			## Configure the error dialog.
			.error configure -text [ msgcat::mc <error.text.user.rightsInsufficient> ]

			## Show the error dialog.
			.error activate

			## Authentication failed. Close the connection.
			itcl::delete object $this
		} \
		{ TKKASSE BILL TABLE_OCCUPIED } \
		{
			## Configure the error dialog.
			.error configure -text [ msgcat::mc <error.text.bill.tableOccupied> ]

			## Show the error dialog.
			.error activate
		}
		{ TKKASSE BILL NOT_COMPLETE } \
		{
			## Configure the error dialog.
			.error configure -text [ msgcat::mc <error.text.bill.notComplete> ]

			## Show the error dialog.
			.error activate
		}
		{ TKKASSE BILL UNKNOWN } \
		{
			## Ignore. 
		}
		{ TKKASSE BILL NOT_PRINTED } \
		{
			## Configure the error dialog.
			.error configure -text [ msgcat::mc <error.text.bill.notPrinted> ]

			## Show the error dialog.
			.error activate
		}
		{ TKKASSE BILL NO_PAY_COMMENT } \
		{
			## Configure the error dialog.
			.error configure -text [ msgcat::mc <error.text.bill.noPayComment> ]

			## Show the error dialog.
			.error activate
		}
		{ TKKASSE BILL NO_MATCH } \
		{
			## Configure the error dialog.
			.error configure -text [ msgcat::mc <error.text.bill.noMatch> ]

			## Show the error dialog.
			.error activate
		}
		default \
		{
			## Unknown error. Show the oops dialog.
			global errorCode errorInfo
			set errorCode [ concat SERVER $CODE ]
			set errorInfo $MESSAGE
			.oops activate
		}
	}
}

## Increment the multiplier of a bill item.
itcl::body client::userconnection::incrementBillItemMultiplier { TAG ITEM_UNIQUE } \
{
	send [ list incrementBillItemMultiplier $TAG $ITEM_UNIQUE ]
}

## Login handler.
itcl::body client::userconnection::LoginHandler { USER_INFO } \
{
	## Set the view key to the login name.
	set LOGIN [ lindex $USER_INFO [ expr [ lsearch -exact $USER_INFO "-login" ] +1 ] ]
	set view_key $LOGIN
	set view [ .main.desktop.bills add $view_key ].frame

	## Add a bill window for this user and select it.
	BillsView $view -userlogin $LOGIN -username [ lindex $USER_INFO [ expr [ lsearch -exact $USER_INFO "-name" ] +1 ] ]
	pack $view -fill both -expand 1
	.main.desktop.bills select $view_key

	## Set the handlers for bill updates.
	send { setBillsUpdateHandler { BillsUpdateHandler %b %s } }
	send { setBillUpdateHandler { BillUpdateHandler %b } }
}

## Print a receipt of the cash register contents.
itcl::body client::userconnection::printCashRegisterReceipt {} \
{
	send printCashRegisterReceipt
}

## Print a receipt for the guest.
itcl::body client::userconnection::printGuestReceiptForBill { TAG HOW } \
{
	send [ list printGuestReceiptForBill $TAG $HOW ]
}

## Print a receipt of the cash register contents for a single waiter.
itcl::body client::userconnection::printWaiterReceipt {} \
{
	send printWaiterReceipt
}

## Order a bill item.
itcl::body client::userconnection::orderItemFromBill { TAG ITEM_UNIQUE } \
{
	send [ list orderItemFromBill $TAG $ITEM_UNIQUE ]
}

## Remove an extra from a bill item.
itcl::body client::userconnection::removeExtraFromBillItem { TAG ITEM_UNIQUE EXTRA_INDEX } \
{
	send [ list removeExtraFromBillItem $TAG $ITEM_UNIQUE $EXTRA_INDEX ]
}








## Initialize the query connection.
proc queryServer {} \
{
	## Configure query connection.
	global SERVER_HOST SERVER_PORT
	query configure \
		-host $SERVER_HOST \
		-port $SERVER_PORT \
		-disconnectcommand \
		{
			## Configure the error dialog.
			global SERVER_HOST SERVER_PORT
			.error configure -text [ msgcat::mc <error.text.connectionLost> "$SERVER_HOST:$SERVER_PORT" ]

			## Show the error dialog.
			.error activate

			## Close all connections.
			disconnectServer
		}

	## Query server.
	if [ catch { query connect } ] \
	{
		## Check if we are already connected.
		global errorCode
		if { $errorCode != [ list CONNECTION CONNECTED ] } \
		{
			## No. Configure the error dialog.
			.error configure -text [ msgcat::mc <error.text.noConnection> "$SERVER_HOST:$SERVER_PORT" ]

			## Show the error dialog.
			.error activate

			## Close all connections.
			disconnectServer
		}
	}
}

## Close all connections to the server.
proc disconnectServer {} \
{
	## Close query connection.
	query configure -disconnectcommand {}
	query disconnect 

	## Close user connections
	foreach CONNECTION [ itcl::find objects -isa client::userconnection ] \
	{
		itcl::delete object $CONNECTION
	}
}

## Login an user.
proc loginUser {} \
{
	## Connect to the server
	global SERVER_HOST SERVER_PORT
	if [ catch { set CONNECTION [ uplevel #0 client::userconnection [ list user[ .main.desktop.users getLogin active ] ] ] } ] {} \
	{
		$CONNECTION configure \
			-host $SERVER_HOST \
			-port $SERVER_PORT
		$CONNECTION connect [ .main.desktop.users getLogin active ] [ .login get ]
	}
}

## Clean up the cash register.
proc cleanupCashRegister {} \
{
	catch { user[ .main.desktop.users getLogin active ] cleanUpCashRegister }
}

## Log out an user.
proc logoutUser {} \
{
	catch { itcl::delete object user[ .main.desktop.users getLogin active ] }
}
	
## Create a new bill for this user
proc createBill {} \
{
	set BILL [ .newbill get ]
	if { $BILL != "" } \
	{
		catch { user[ .main.desktop.users getLogin active ] createBill $BILL }
	}
}

## Close the current bill.
proc closeBill {} \
{
	## Get the current tabset, as the bills are divided by users.
	if [ catch { getCurrentBillViewTag } TAG ] \
	{
		## The current user is not logged in, no bill view object.
		return
	}

	## Let the associated user connection send the command.
	[ getCurrentUserConnectionObject ] closeBill $TAG [ .closebill getpayedby ] [ .closebill getdetail ]
}

## Add an article to the bill.
proc addArticleItemToBill {} \
{
	## Get the current tabset, as the bills are divided by users.
	if [ catch { getCurrentBillViewTag } TAG ] \
	{
		## The current user is not logged in, no bill view object.
		return
	}

	## Get the selected article.
	set ARTICLE [ .main.desktop.articles get ]

	## Return if no article is selected.
	if { $ARTICLE == "" } return

	## Let the associated user connection send the command.
	[ getCurrentUserConnectionObject ] addArticleItemToBill $TAG $ARTICLE

	## Clear the article selection.
	.main.desktop.articles deselect
}

## Add an article to the bill again.
proc addArticleItemToBillAgain {} \
{
	## Get the current tabset, as the bills are divided by users.
	if [ catch { getCurrentBillViewTag } TAG ] \
	{
		## The current user is not logged in, no bill view object.
		return
	}

	## Get the selected article.
	set ARTICLE [ .main.desktop.articles get ]

	## Return if no article is selected.
	if { $ARTICLE == "" } return

	## Let the associated user connection send the command.
	[ getCurrentUserConnectionObject ] addArticleItemToBillAgain $TAG $ARTICLE
}

## Add an extra to the last article.
proc addExtraToBillItem {} \
{
	set NAME  [ lindex [ .extraArticle get ] 0 ]
	set PRICE [ lindex [ .extraArticle get ] 1 ]

	if { ( $NAME != "" ) && [ checkCurrency $PRICE ] } \
	{
		## Throw the event.
		throwBillViewEvent addExtra $NAME [ plainCurrency $PRICE ]
	} \
	{
		## Wrong input format for the extra.
		bell
	}
}

## Print a receipt of the cash register contents.
proc printCashRegisterReceipt {} \
{
	catch { user[ .main.desktop.users getLogin active ] printCashRegisterReceipt }
}

## Print a receipt for the guest.
proc printGuestReceipt { HOW } \
{
	## Get the current tabset, as the bills are divided by users.
	if [ catch { getCurrentBillViewTag } TAG ] \
	{
		## The current user is not logged in, no bill view object.
		return
	}

	## Print.	
	user[ .main.desktop.users getLogin active ] printGuestReceiptForBill $TAG $HOW
}

## Print a receipt of the cash register contents for a single waiter.
proc printWaiterReceipt {} \
{
	catch { user[ .main.desktop.users getLogin active ] printWaiterReceipt }
}

## Handler for events generated by the main menu, but associated to a bill view.
proc throwBillViewEvent { EVENT_TYPE_ARG args } \
{
	## Handle the event as if it was generated in the context of the current bill view.
	set BILL_VIEW [ getCurrentBillViewObject ]
	if { $BILL_VIEW != {} } { eval $BILL_VIEW handleEvent $EVENT_TYPE_ARG $args }
}	

## Handler for events generated in the context of a bill view.
proc BillViewHandler { COMMAND_ARG TAG_ARG args } \
{
	## Let the associated user connection send the command.
	eval [ getCurrentUserConnectionObject ] $COMMAND_ARG $TAG_ARG $args
}

## Get the current bill view tag.
proc getCurrentBillViewTag {} \
{
	## Get the current tabset tag, as the bills are divided by users.
	[ [ .main.desktop.bills childsite [ .main.desktop.bills get ] ].frame component bills ].tabset get select
}

## Get the current bill view object.
proc getCurrentBillViewObject {} \
{
	## Get the current tabset, as the bills are divided by users.
	if [ catch { getCurrentBillViewTag } TAG ] \
	{
		## The current user is not logged in, no bill view object.
		return
	}

	## Get the bill view object.
	if [ catch { [ .main.desktop.bills childsite [ .main.desktop.bills get ] ].frame getView $TAG } BILL_VIEW_OBJECT ] \
	{
		## No bill for this waiter.
		return
	} \
	{
		## Return the object descriptor.
		return $BILL_VIEW_OBJECT
	}
}

## Get the connection object for the current user.
proc getCurrentUserConnectionObject {} \
{
	return "user[ .main.desktop.users getLogin active ]"
}

