/*
 *  ISEM - Instructional Sparc EMulator and tkisem
 *  Copyright (C) 1993, 1994, 1995, 1996
 *	Department of Computer Science,
 *      The University of New Mexico
 *
 *  Please send questions, comments, and bug reports to: isem@cs.unm.edu
 *
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

//
// $Id: Instruct.h 1.1 Fri, 25 Oct 1996 18:04:04 -0600 maccabe $
//

//== Instruct.h =========================================================
//
// V1.0 - 15 JUN 92
//
//  Include File for class "Instruction"
//

#ifndef _INSTRUCT_H
#define _INSTRUCT_H

#include "InstrFmt.h"		/* Bitfields in Instruction format */

class Instruction {
public:

    // The order of these enums is critical and is defined in the SPARC
    // architecture manual
    //
    enum Op {FORMAT2=0, FORMAT1=1, FORMAT3=2, FORMAT3LS=3};

    enum Op2 {UNIMP, Bicc = 2, SETHI = 4, FBfcc = 6, CBccc = 7};

    // The order of these instructions is critical. For more details
    // on the arrangment of the OP3 field decoding, see the SPARC
    // architecture manual
    //
    // Any instructions which cause a gap in the sequence, are started
    // on a new line.
    //
    enum Op3 {ADD=0, AND, OR, XOR, SUB, ANDN, ORN, XNOR, ADDX,
	 UMUL = 10, SMUL, SUBX,
	 UDIV = 14, SDIV,
         ADDcc, ANDcc, ORcc, XORcc, SUBcc, ANDNcc, ORNcc,
         XNORcc, ADDXcc,
	 UMULcc = 26, SMULcc, SUBXcc,
         UDIVcc = 30, SDIVcc,
         TADDcc, TSUBcc, TADDccTV, TSUBccTV, MULScc,
         SLL, SRL, SRA, RDY, RDPSR, RDWIM, RDTBR,
         WRY = 48, WRPSR, WRWIM, WRTBR, FPop1, FPop2, CPop1, CPop2,
         JMPL, RETT, Ticc, IFLUSH, SAVE, RESTORE,
         LD = 64, LDUB, LDUH, LDD, ST, STB, STH, STD,
         LDSB = 73, LDSH,
         LDSTUB = 77,
         SWAP = 79, LDA, LDUBA, LDUHA, LDDA, STA, STBA, STHA, STDA,
         LDSBA = 89, LDSHA,
         LDSTUBA = 93,
         SWAPA = 95, LDF, LDFSR,
         LDDF = 99, STF, STFSR, STDFQ, STDF,
         LDC = 112, LDCSR,
         LDDC = 115, STC, STCSR, STDCQ, STDC };

    // The order of these instructions is critical. For more details
    // on the arrangment of the COND field decoding, see the SPARC
    // architecture manual
    //
    enum Cond { N, E, LE, L, LEU, CS, NEG, VS,
                A, NE, G, GE, GU, CC, POS, VC};


    Instruction(UInt32);

    // Member functions to return specific bit fields of the instruction
    //
    Op	 	op() const;
    Int32  	disp30() const;
    UInt32 	rd() const;
    UInt32 	rs1() const;
    Op2 	op2() const;
    Int32  	imm22() const;
    UInt32 	a() const;
    Cond 	cond() const;
    Int32  	disp22() const;
    Op3 	op3() const;
    UInt32 	i() const;
    UInt32 	asi() const;
    UInt32 	rs2() const;
    Int32  	simm13() const;
    UInt32 	opf() const;
    UInt32	shcnt() const;


private:
    UInt32  f0;		// access to entire 32bits
}; 


inline Instruction::Instruction(UInt32 i) {
    f0 = i;
}

inline Instruction::Op Instruction::op() const {
    return (Instruction::Op)(f0 >> 30);
}

inline Int32  Instruction::disp30() const {
    Int32 tmp = f0 & 0x3fffffff;
    if ((tmp & 0x20000000) != 0) {	// negative ?
        tmp |= Int32(0xC0000000);       // yep, sign extend to 32 bits
    }
    return(tmp);
}

inline UInt32 Instruction::rd() const {
    return (f0 >> 25) & 0x1f;
}

inline Instruction::Op2 Instruction::op2() const {
    return (Instruction::Op2)((f0 >> 22) & 0x7);
}

inline UInt32 Instruction::rs1() const {
    return (f0 >> 14) & 0x1f;
}


inline Int32 Instruction::imm22() const {
    Int32 tmp = f0 & 0x3fffff;
    if ((tmp & 0x200000)!= 0) {		// negative ?
        tmp |= Int32(0xFFC00000);       // yep, sign extend to 32 bits
    }
    return(tmp);
}

inline UInt32 Instruction::a() const {
    return (f0 >> 29) & 0x1;
}

inline Instruction::Cond Instruction::cond() const {
    return (Instruction::Cond)((f0 >> 25) & 0xf);
}

inline Int32 Instruction::disp22() const {
    Int32 tmp = f0 & 0x3fffff;
    if ((tmp & 0x200000)!= 0) {		// negative ?
        tmp |= Int32(0xFFC00000);       // yep, sign extend to 32 bits
    }
    return tmp;
}

inline Instruction::Op3 Instruction::op3() const {
    return (Instruction::Op3)(((f0 >> 19) & 0x3f) | ((f0 >> 24) & 0x40));
}

inline UInt32 Instruction::i() const {
    return (f0 >> 13) & 0x1;
}

inline UInt32 Instruction::asi() const {
    return (f0 >> 5) & 0xff;
}

inline UInt32 Instruction::rs2() const {
    return (f0 & 0x1f);
}

inline Int32 Instruction::simm13() const {
    Int32 tmp = f0 & 0x1fff;
    if ((tmp & 0x1000) != 0 ) {		// negative ?
	tmp |= Int32(0xFFFFE000);       // yep, sign extend to 32 bits
    }
    return(tmp);
}

inline UInt32 Instruction::opf() const {
    return (f0 >> 5) & 0x1ff;
}

inline UInt32 Instruction::shcnt() const {
    return (f0 & 0x1f);
}


#endif
