# $Id: iroster.tcl,v 1.5 2004/07/10 20:05:04 aleksey Exp $

foreach {k v} [list background       white     \
		    foreground       black     \
		    activeBackground cyan \
		    troughColor      cyan] {
    if {[cequal [set $k [option get . $k widgetDefault]] ""]} {
	set $k $v
    }
}

option add *Roster.cbackground           $background       widgetDefault
option add *Roster.groupindent           0                widgetDefault
option add *Roster.jidindent             2                widgetDefault
option add *Roster.jidmultindent         2                widgetDefault
option add *Roster.subjidindent          4                widgetDefault
option add *Roster.subitemtype            1                widgetDefault
option add *Roster.foreground            $foreground       widgetDefault
option add *Roster.jidfill               $background       widgetDefault
option add *Roster.jidhlfill             $activeBackground widgetDefault
option add *Roster.jidborder             $background       widgetDefault
option add *Roster.groupfill             black      widgetDefault
option add *Roster.groupcfill            $troughColor      widgetDefault
option add *Roster.grouphlfill           $activeBackground widgetDefault
option add *Roster.groupborder           $foreground       widgetDefault
option add *Roster.stalkerforeground     black           widgetDefault
option add *Roster.unavailableforeground black           widgetDefault
option add *Roster.dndforeground         red           widgetDefault
option add *Roster.xaforeground          black           widgetDefault
option add *Roster.awayforeground        black           widgetDefault
option add *Roster.availableforeground   black           widgetDefault
option add *Roster.chatforeground        blue           widgetDefault

unset background foreground activeBackground troughColor

namespace eval roster {
    set roster(jids) {}

    custom::defgroup Roster [::msgcat::mc "Roster options."] -group Tkabber
    variable show_only_online 0
    set menu_item_idx 0
    variable id ""
    set use_aliases 1
}


proc roster::process_item {jid name groups subsc ask category subtype} {
    variable roster
    debugmsg roster "ROSTER_ITEM: $jid; $name; $groups; $subsc; $ask; $category; $subtype"

    set jid [tolower_node_and_domain $jid]


    if {$subsc != "remove"} {
	if {[lsearch $roster(jids) $jid] == -1} {
	    lappend roster(jids) $jid
	}

	set roster(group,$jid)    $groups
	set roster(name,$jid)     $name
	set roster(subsc,$jid)    $subsc
	set roster(ask,$jid)      $ask
	set roster(category,$jid) $category
	set roster(subtype,$jid)  $subtype

	lassign [get_category_and_subtype $jid] \
	    roster(ccategory,$jid) roster(csubtype,$jid)
	set roster(isuser,$jid) [cequal $roster(ccategory,$jid) user]
	catch {unset roster(cached_category_and_subtype,$jid)}
    } else {
	lvarpop roster(jids) [lsearch $roster(jids) $jid]

	unset roster(group,$jid)
	unset roster(name,$jid)
	unset roster(subsc,$jid)
	unset roster(ask,$jid)
	unset roster(category,$jid)
	unset roster(subtype,$jid)
	unset roster(ccategory,$jid)
	unset roster(csubtype,$jid)
	unset roster(isuser,$jid)
	catch {unset roster(cached_category_and_subtype,$jid)}
    }
    after cancel ::update_chat_titles
    after idle ::update_chat_titles
}

proc client:roster_item {jid name groups subsc ask category subtype} {
    roster::process_item $jid $name $groups $subsc $ask $category $subtype
}

proc client:roster_push {jid name groups subsc ask category subtype} {
    roster::process_item $jid $name $groups $subsc $ask $category $subtype
    roster::redraw_after_idle
}

proc client:roster_cmd {status} {
    debugmsg roster "ROSTER_CMD: $status"
    
    if {[cequal $status END_ROSTER]} {
	roster::redraw
    } else {
	global roster::roster
	set roster::roster(jids) {}
	#roster::clear .roster
    }
}


proc roster::get_groups {} {
    variable roster

    set groups {}
    foreach jid $roster(jids) {
	#debugmsg roster [array get roster]
	set groups [concat $groups $roster(group,$jid)]
    }

    set groups [lrmdups $groups]
    return $groups
}

proc roster::itemconfig {jid args} {
    variable roster

    if {[llength $args] == 1} {
	lassign $args attr
	switch -- $attr {
	    -group    {set param group}
	    -name     {set param name}
	    -subsc    {set param subsc}
	    -ask      {set param ask}
	    -category {set param category}
	    -subtype  {set param subtype}
	    default {return}
	}
	if {[info exists roster($param,$jid)]} {
	    return $roster($param,$jid)
	} else {
	    return ""
	}
    } else {
	foreach {attr val} $args {
	    switch -- $attr {
		-group    {set param group}
		-name     {set param name}
		-subsc    {set param subsc}
		-ask      {set param ask}
		-category {set param category}
		-subtype  {set param subtype}
		default   {set param ""}
	    }
	    set roster($param,$jid) $val
	}
    }
}

#load_source roster_nested.tcl

proc roster::redraw {} {
    variable options
#    if {$options(nested)} {
#	roster::redraw_nested
#	return
#    }

    variable roster
    variable config
    variable show_only_online
    variable aliases
    variable use_aliases

    clear .roster 0
    
    set groups {}

    set undef_group_name [::msgcat::mc Undefined]

    if {$use_aliases} {
	foreach jid [array names aliases] {
	    foreach alias $aliases($jid) {
		set ignore_jid($alias) ""
	    }
	}
    }

    foreach jid $roster(jids) {
	if {[info exists ignore_jid($jid)]} continue
	if {![lempty $roster(group,$jid)]} {
	    set groups [concat $groups $roster(group,$jid)]
	    
	    foreach group $roster(group,$jid) {
	        lappend jidsingroup($group) $jid
	    }
	} else {
	    lappend jidsingroup($undef_group_name) $jid
	    lappend groups $undef_group_name
	}
    }
    set groups [lrmdups $groups]
    foreach group $groups {
	set jidsingroup($group) [lrmdups $jidsingroup($group)]
	if {![info exists roster(collapsed,$group)]} {
	    set roster(collapsed,$group) 0
	}
	set online 0
	set users 0
	set not_users 0
	foreach jid $jidsingroup($group) {
	    if {$roster(isuser,$jid)} {
		incr users
		set status [get_user_aliases_status $jid]
		set jstat($jid) $status
		if {$status != "unavailable"} {
		    incr online
		    set useronline($jid) 1
		} else {
		    set useronline($jid) 0
		}
	    } else {
		incr not_users
	    }
	}
	if {!$show_only_online || $online + $not_users > 0} {
	    if {$users} {
		addline .roster group "$group ($online/$users)" $group
	    } else {
		addline .roster group $group $group
	    }
	}
	if {!$roster(collapsed,$group)} {
	    set jid_names {}
	    foreach jid $jidsingroup($group) {
		lappend jid_names [list $jid [get_label $jid]]
	    }
	    set jid_names [lsort -index 1 -dictionary $jid_names]
	    foreach jid_name $jid_names {
		lassign $jid_name jid name
		if {!$show_only_online || \
			![info exists useronline($jid)] || $useronline($jid)} {
		    lassign [get_category_and_subtype $jid] category type
		    set jids [get_jids_of_user $jid]
		    set numjids [llength $jids]
		    if {($numjids > 1) && ($config(subitemtype) > 0) && \
			    $category == "user"} {
		    	if {$config(subitemtype) & 1} {
			    if {$category == "conference"} {
			    	set numjids [expr {$numjids - 1}]
			    }
			    addline .roster jid "$name ($numjids)" $jid $jids
			} else {
			    addline .roster jid "$name" $jid $jids
			}
			changeforeground .roster $jid [get_jid_foreground $jid]

			if {[info exists roster(collapsed,$jid)] && \
				!$roster(collapsed,$jid)} {
			    foreach subjid $jids {
				set subjid_resource [resource_from_jid $subjid]
				if {$subjid_resource != ""} {
				    addline .roster jid2 \
					$subjid_resource $subjid [list $subjid]
				    changeforeground .roster \
					$subjid [get_jid_foreground $subjid]
				}
			    }
			}
		    } else {
			if {$numjids <= 1 && $category == "user"} {
			    set status $jstat($jid)

			    if {([cequal $roster(subsc,$jid) from] || \
				     [cequal $roster(subsc,$jid) none]) && \
				    $status == "unavailable"} {
				set status stalker
			    }
			    addline .roster jid $name $jid $jids \
				$config(${status}foreground)
			} else {
			    addline .roster jid $name $jid $jids
			    changeforeground .roster $jid [get_jid_foreground $jid]
			}
		    }
		}
	    }
	}
    }
    #debugmsg roster [array get roster collapsed*]
    update_scrollregion .roster
}

proc roster::redraw_after_idle {} {
    variable afterid

    if {[info exists afterid]} \
	return

    set afterid [after idle {
	ifaceck::roster::redraw
	unset ifaceck::roster::afterid
    }]
}

# Callback
proc ::redraw_roster {args} {
    ifaceck::roster::redraw_after_idle
}

proc roster::get_jids_of_user {user} {
    variable aliases
    variable use_aliases

    if {$use_aliases && [info exists aliases($user)]} {
	set jids [::get_jids_of_user $user]
	foreach alias $aliases($user) {
	    set jids [concat $jids [::get_jids_of_user $alias]]
	}
	return $jids
    } else {
	return [::get_jids_of_user $user]
    }
}

proc roster::get_user_aliases_status {user} {
    variable aliases
    variable use_aliases

    if {$use_aliases && [info exists aliases($user)]} {
	set status [get_user_status $user]

	foreach alias $aliases($user) {
	    set status [max_status $status [get_user_status $alias]]
	}
	return $status
    } else {
	return [get_user_status $user]
    }
}

proc roster::get_jid_foreground {jid} {
    lassign [get_category_and_subtype $jid] category type

    switch -- $category {
	"" -
	user {
	    return [get_user_foreground $jid]
	}
	conference {
	    set status [get_jid_status $jid]
	    if {$status != "unavailable"} {
		return available
	    } else {
		return unavailable
	    }
	}
	service {
	    return [get_service_foreground $jid $type]
	}
	default {
	    return ""
	}
    }
}

proc roster::get_service_foreground {service type} {
    variable roster

    switch -- $type {
	jud {return ""}
	}
    if {![cequal $roster(subsc,$service) none]} {
	return [get_user_status $service]
    } else {
	return stalker
    }
}

proc roster::get_user_foreground {user} {
    variable roster

    set status [get_user_aliases_status $user]

    if {[info exists roster(subsc,$user)]} {
	if {([cequal $roster(subsc,$user) from] || \
		 [cequal $roster(subsc,$user) none]) && \
		$status == "unavailable"} {
	    return stalker
	} else {
	    return $status
	}
    } else {
	set user_without_resource [node_and_server_from_jid $user]
	if {([cequal $roster(subsc,$user_without_resource) from] || \
		 [cequal $roster(subsc,$user_without_resource) none]) && \
		$status == "unavailable"} {
	    return stalker
	} else {
	    return $status
	}
    }
}

proc roster::on_change_jid_presence {jid} {
    variable roster
    
    set rjid [find_jid $jid]
    debugmsg roster "$jid $rjid"

    if {$rjid != ""} {
	lassign [get_category_and_subtype $rjid] category subtype
	
	if {$category == "user"} {
	    set_status [cconcat [get_label $rjid] " " \
			    [::msgcat::mc "is now"] " " \
			    [::msgcat::mc [get_user_status $rjid]]]
	    
	    hook::run on_change_user_presence_hook \
		[get_label $rjid] [get_user_status $rjid]
	}
    }
    redraw_after_idle
}

proc roster::find_jid {jid} {
    variable roster

    if {[lcontain $roster(jids) $jid]} {
	return $jid
    }

    lassign [heuristically_get_category_and_subtype $jid] category subtype
    debugmsg roster "$category $subtype"
    foreach rjid $roster(jids) {
	lassign [get_category_and_subtype $rjid] rcategory rsubtype
	if {[string equal $category $rcategory]} {
	    switch -- $category {
		user {
		    if {[string equal [node_and_server_from_jid $jid] $rjid]} {
			return $rjid
		    }
		}
		default {
		}
	    }
	}
    }
    return ""
}

proc roster::get_category_and_subtype {jid} {
    variable roster

    if {[info exists roster(cached_category_and_subtype,$jid)]} {
	return $roster(cached_category_and_subtype,$jid)
    }

    if {[info exists roster(category,$jid)]} {
	if {$roster(category,$jid) != ""} {
	    return [list $roster(category,$jid) $roster(subtype,$jid)]
	}
    }
    
    return [heuristically_get_category_and_subtype $jid]
}

proc roster::heuristically_get_category_and_subtype {jid} {
    variable roster

    if {[info exists roster(cached_category_and_subtype,$jid)]} {
	return $roster(cached_category_and_subtype,$jid)
    }

    if {[node_from_jid $jid] == ""} {
	set category service

	set updomain [lindex [split [server_from_jid $jid] .] 0]
	if {[lcontain {aim icq irc jabber jud msn pager rss serverlist \
			   sms smtp yahoo} $updomain]} {
	    set subtype $updomain
	} elseif {[cequal icqv7 $updomain]} {
	    set subtype icq
	} else {
	    set subtype ""
	}

	set roster(cached_category_and_subtype,$jid) [list $category $subtype]
	return [list $category $subtype]
    }

    if {[resource_from_jid $jid] == ""} {
	set updomain [lindex [split [server_from_jid $jid] .] 0]
	switch -- $updomain {
	    conference {
		set category conference
		set subtype ""
	    }
	    default {
		set category user
		set subtype ""
	    }
	}
	set roster(cached_category_and_subtype,$jid) [list $category $subtype]
	return [list $category $subtype]
    }
    set roster(cached_category_and_subtype,$jid) {user client}
    return {user client}
}

proc roster::changeforeground {w jid fg} {
    variable config
    set t $w.text
    set tag [jid_to_tag $jid]
    $t tag configure $tag -foreground $config(${fg}foreground)
}

proc roster::create {w args} {
    variable roster
    variable config

    set t $w.text
    
    set width 12
    set height 15
    set popupproc {}
    set grouppopupproc {}
    foreach {attr val} $args {
	switch -- $attr {
	    -width {set width $val}
	    -height {set height $val}
	    -popup {set popupproc $val}
	    -grouppopup {set grouppopupproc $val}
	}
    }

    ScrolledWindow $w -class Roster

    set config(background)  [option get $w cbackground Roster] 
    set config(groupindent) [option get $w groupindent Roster] 
    set config(jidindent)   [option get $w jidindent   Roster]
    set config(jidmultindent)   [option get $w jidmultindent   Roster]
    set config(jid2indent)  [option get $w subjidindent Roster]
    set config(subitemtype)  [option get $w subitemtype  Roster]
    set config(foreground)  [option get $w foreground  Roster]
    set config(jidfill)	    [option get $w jidfill     Roster]
    set config(jidhlfill)   [option get $w jidhlfill   Roster]
    set config(jidborder)   [option get $w jidborder   Roster]
    set config(jid2fill)    $config(jidfill)
    set config(jid2hlfill)  $config(jidhlfill)
    set config(jid2border)  $config(jidborder)
    set config(groupfill)   [option get $w groupfill   Roster]
    set config(groupcfill)  [option get $w groupcfill  Roster]
    set config(grouphlfill) [option get $w grouphlfill Roster]
    set config(groupborder) [option get $w groupborder Roster]
    set config(stalkerforeground) [option get $w stalkerforeground Roster]
    set config(unavailableforeground) [option get $w unavailableforeground Roster]
    set config(dndforeground) [option get $w dndforeground Roster]
    set config(xaforeground) [option get $w xaforeground Roster]
    set config(awayforeground) [option get $w awayforeground Roster]
    set config(availableforeground) [option get $w availableforeground Roster]
    set config(chatforeground) [option get $w chatforeground Roster]

    text $w.text -bg $config(background) -state disabled \
	-width $width -height $height -wrap none
    $w setwidget $w.text

    set roster($w,width) 0
    set roster($w,popup) $popupproc
    set roster($w,grouppopup) $grouppopupproc

#    bindscroll $w.canvas

#    if {$w == ".roster"} {
#	DropSite::register .roster.canvas -dropcmd roster::dropcmd \
#	    -droptypes {JID}
#	DragSite::register .roster.canvas -draginitcmd roster::draginitcmd
#    }

}

proc roster::addline {w type text jid {jids {}} {foreground ""}} {
    global font
    variable roster
    variable config
    variable aliases
    variable use_aliases

    set t $w.text

    set tag [jid_to_tag $jid]
    set background $config(${type}fill)

    if {[cequal $type jid]} {
	set isuser \
	    [expr {![info exists roster(isuser,$jid)] || $roster(isuser,$jid)}]

	if {[llength $jids] > 1} {
	    if {[info exists roster(collapsed,$jid)] && !$roster(collapsed,$jid)} {
	    } else {
		set roster(collapsed,$jid) 1
	    }
	} else {
	    set roster(collapsed,$jid) 1
	}
    } elseif {[cequal $type group]} {
	if {[info exists roster(collapsed,$jid)] && $roster(collapsed,$jid)} {
	    set group_state closed
	    set group_icon ">"
	} else {
	    set group_state opened
	    set group_icon "v"
	}
    }

    if {([cequal $type jid]) && ($config(subitemtype) > 0) && ($config(subitemtype) & 2)} {
	if {$isuser && ([llength $jids] > 1)} {
	    set x $config(jidmultindent)
	} else {
	    set x $config(jidindent)
	}
    } else {
	set x $config(${type}indent)
    }

    if {$foreground == ""} {
	if {[cequal $type jid] || [cequal $type jid2]} {
	    set foreground $config(unavailableforeground)
	} else {
	    set foreground $config(foreground)
	}
    }
    $t configure -state normal
    if {[cequal $type group]} {
	$t insert end "$group_icon " [list icon $tag]
	$t tag configure icon -attributes bold
    }
    $t insert end "[string repeat " " $x]$text\n" $tag
    $t tag configure $tag -foreground $foreground -background $background
    $t configure -state disabled

    set roster($w,width) [max $roster($w,width) \
			      [expr {$x + [string length $text]}]]


    set doubledjid  [double% $jid]

if {0} {
    $t tag bind $tag <Double-Button-1> \
	[list roster::jid_doubleclick $doubledjid]
}
    if {[cequal $type jid] || [cequal $type jid2]} {
if {0} {
	if {[llength $jids] > 0} {
	    set doubledjids [double% $jids]
	    $c bind jid$tag <Any-Enter> \
		+[list eval balloon::set_text \
		    \[roster::jids_popup_info [list $doubledjids]\]]

	    $c bind jid$tag <Any-Motion> \
		[list eval balloon::on_mouse_move \
		    \[roster::jids_popup_info [list $doubledjids]\] %X %Y]
	} else {
	    set jids [list $jid]
	    if {$use_aliases && [info exists aliases($jid)]} {
		set jids [concat $jids $aliases($jid)]
	    }
	    set doubledjids [double% $jids]
	    $c bind jid$tag <Any-Enter> \
		+[list eval balloon::set_text \
		    \[roster::jids_popup_info [list $doubledjids]\]]

	    $c bind jid$tag <Any-Motion> \
		[list eval balloon::on_mouse_move \
		     \[roster::jids_popup_info [list $doubledjids]\] %X %Y]
	}
	
	$c bind jid$tag <Any-Leave> {+
	    balloon::destroy
	}
	
}
	if {![cequal $roster($w,popup) {}]} {
	    $t tag bind $tag <3> [list $roster($w,popup) $doubledjid %X %Y]
	}
    } else {
	if {$w == ".roster"} {
	    $t tag bind $tag <Button-1> \
		[list roster::group_doubleclick $doubledjid]
	}

	if {![cequal $roster($w,grouppopup) {}]} {
	    $t tag bind $tag <3> \
		[list $roster($w,grouppopup) $doubledjid %X %Y]
	}
    }
if {0} {
    if {[cequal $type jid]} {
	if {$isuser \
		&& ([llength $jids] > 1)} {
	    if {$w == ".roster"} {
		$c bind jid$tag <Button-1> \
		    [list roster::user_singleclick $jid]
	    }
	}
    }
}
}

proc roster::clear {w {updatescroll 1}} {
    variable roster

    set t $w.text
    $t configure -state normal
    $t delete 0.1 end
    $t configure -state disabled

    set roster($w,width) 0
    if {$updatescroll} {
	update_scrollregion $w
    }
}

proc roster::clean {} {
    variable roster

    array unset roster group,*
    array unset roster name,*
    array unset roster subsc,*
    array unset roster ask,*
    array unset roster category,*
    array unset roster subtype,*
    set roster(jids) {}
    redraw
}

proc roster::update_scrollregion {w} {
}

proc roster::jid_doubleclick {jid} {
    variable roster

    lassign [get_category_and_subtype $jid] category subtype

    switch -- $category {
	conference {
	    global gr_nick
	
	    client:presence $jid "" "" {}
	    join_group $jid [get_group_nick $jid $gr_nick]
	}
	user -
	default {
	    if {[cequal $chat::options(default_message_type) chat]} {
		chat::open_to_user $jid
	    } else {
		message::send_dialog -to $jid
	    }
	}
    }
}

proc roster::group_doubleclick {group} {
    variable roster

    set roster(collapsed,$group) [expr {!$roster::roster(collapsed,$group)}]
    redraw
}

proc roster::jids_popup_info {jids} {
    set text {}
    foreach j $jids {
	append text "\n[roster::user_popup_info $j]"
    }
    set text [string trimleft $text "\n"]
    return $text
}

proc roster::user_popup_info {user} {
    variable user_popup_info

    lassign [get_category_and_subtype $user] category subtype

    switch -- $category {
	conference {
	    set status [get_jid_status $user]
	    set desc ""
	}
	user -
	default {
	    set status [get_user_status $user]
	    set desc   [get_user_status_desc $user]
	}
    }

    #set desc   [get_user_status_desc $user]
    if {(![cequal $status [string tolower $desc]]) && (![cequal $desc ""])} {
	append status " ($desc)"
    }

    set user_popup_info "$user: $status"

    hook::run roster_user_popup_info_hook \
	[namespace which -variable user_popup_info] $user

    return $user_popup_info
}


proc roster::get_label {jid} {
    variable roster

    if {[lempty $roster(name,$jid)]} {
	return $jid
    } else {
	return $roster(name,$jid)
    }
}

proc roster::switch_only_online {} {
    variable show_only_online
    set show_only_online [expr !$show_only_online]
    changed_only_online
}

proc roster::changed_only_online {} {

    redraw
}

proc roster::is_online {jid} {
    if {[is_user $jid]} {
	switch -- [get_user_aliases_status $jid] {
	    unavailable {return 0}
	    default {return 1}
	}
    } else {
	return 1
    }
}

proc roster::is_user_online {jid} {
    return [expr {![cequal [get_user_aliases_status $jid] unavailable]}]
}

proc roster::is_user {jid} {
    return [cequal [lindex [get_category_and_subtype $jid] 0] "user"]
}


proc roster::item_to_xml {jid} {
    variable roster

    set grtags {}
    foreach group $roster(group,$jid) {
	lappend grtags [jlib::wrapper:createtag group -chdata $group]
    }

    set vars [list jid $jid]

    if {$roster(name,$jid) != ""} {
	lappend vars name $roster(name,$jid)
    }

    if {$roster(category,$jid) != ""} {
	lappend vars category $roster(category,$jid)
	if {$roster(subtype,$jid) != ""} {
	    lappend vars type $roster(subtype,$jid)
	}
    }

    return [jlib::wrapper:createtag item \
		-vars $vars \
		-subtags $grtags]
}

proc roster::send_item {jid} {
    jlib::send_iq set \
	[jlib::wrapper:createtag query \
	     -vars {xmlns jabber:iq:roster} \
	     -subtags [list [roster::item_to_xml $jid]]]
}

proc roster::remove_item {jid} {
    jlib::send_iq set \
	[jlib::wrapper:createtag query \
	     -vars {xmlns jabber:iq:roster} \
	     -subtags [list [jlib::wrapper:createtag item \
				 -vars [list jid $jid \
					    subscription remove]]]]
	
    jlib::send_presence -to $jid -type unsubscribe

    lassign [get_category_and_subtype $jid] category subtype

    if {$category == "service"} {
	jlib::send_iq set \
	    [jlib::wrapper:createtag query \
		-vars {xmlns jabber:iq:register} \
		-subtags [list [jlib::wrapper:createtag remove]]] \
	    -to $jid
    }
}

proc roster::remove_item_dialog {jid} {
    set res [MessageDlg .remove_item -aspect 50000 -icon question -type user \
	-buttons {yes no} -default 0 -cancel 1 \
	-message [format [::msgcat::mc "Are you sure to remove %s from roster?"] $jid]]
    if {$res == 0} {
	roster::remove_item $jid
    }
}

###############################################################################

proc roster::dropcmd {target source X Y op type data} {
    variable roster
    debugmsg roster "$target $source $X $Y $op $type $data"

    set c .roster.canvas

    set x [expr {$X-[winfo rootx $c]}]
    set y [expr {$Y-[winfo rooty $c]}]
    set xc [$c canvasx $x]
    set yc [$c canvasy $y]

    set tags [$c gettags [lindex [$c find closest $xc $yc] 0]]
    if {[lcontain $tags group]} {
	set tag [crange [lindex $tags 0] 3 end]
	set gr [tag_to_jid $tag]
    } else {
	set gr {}
    }

    debugmsg roster "GG: $gr; $tags"

    lassign $data jid category type name version

    if {![lcontain $roster(jids) $jid]} {
	if {$gr != {}} {
	    itemconfig $jid -category $category -subtype $type \
		-name $name -group [list $gr]
	} else {
	    itemconfig $jid -category $category -subtype $type \
		-name $name -group {}
	}
    } else {
	set groups [itemconfig $jid -group]
	if {$gr != ""} {
	    lappend groups $gr
	    set groups [lrmdups $groups]
	    debugmsg roster $groups
	}
	itemconfig $jid -category $category -subtype $type \
	    -name $name -group $groups
    }
    send_item $jid
}

proc roster::draginitcmd {target x y top} {
    variable roster
    debugmsg roster "$target $x $y $top"

    balloon::destroy
    set c .roster.canvas

    set tags [$c gettags current]
    if {[lcontain $tags jid]} {
	set tag [crange [lindex $tags 0] 3 end]
	set jid [tag_to_jid $tag]

	set data [list $jid \
		      [itemconfig $jid -category] \
		      [itemconfig $jid -subtype] \
		      [itemconfig $jid -name] {}]

	debugmsg roster $data
	return [list JID {copy} $data]
    } else {
	return {}
    }
}

###############################################################################
###############################################################################

proc roster::popup_menu {jid x y} {
    global curuser
    set curuser $jid

    lassign [get_category_and_subtype $jid] category subtype

    switch -- $category {
	conference {set menu .confpopupmenu}
	user {set menu [create_user_menu $jid]}
	service {set menu .servicepopupmenu}
	default {set menu .jidpopupmenu}
    }
    ck_popup $menu $x $y
}


proc roster::group_popup_menu {name x y} {
    set m [create_group_popup_menu $name]
    ck_popup $m $x $y
}


proc roster::groupchat_popup_menu {jid} {
    global curgroupuser
    set curgroupuser $jid
    tk_popup [create_groupchat_user_menu $jid] \
	[winfo pointerx .] [winfo pointery .]
}

proc roster::add_menu_item {m label command jids} {
    variable menu_item_idx
    if {[llength $jids] == 0} {
	$m add command -label $label -command $command
    } elseif {[llength $jids] == 1} {
	set curuser $jids
	set com [subst -nobackslashes -nocommands $command]
	$m add command -label $label -command $com
    } else {
	set m2 [menu $m.[incr menu_item_idx] -tearoff 0]
	$m add cascad -label $label -menu $m2
	foreach jid $jids {
	    set curuser [list $jid]
	    set com [subst -nobackslashes -nocommands $command]
	    $m2 add command -label $jid -command $com
	}
    }
}

proc roster::collapse_item {jid} {
    variable roster
    variable id

    set id ""
    set roster(collapsed,$jid) [expr !$roster::roster(collapsed,$jid)]
    redraw
}

proc roster::user_singleclick {jid} {
    variable id

    if {$id == ""} {
	set id [after 300 [list roster::collapse_item $jid]]
    } else {
	after cancel $id
	set id ""
    }
}

proc roster::create_user_menu {user} {
    set jids [get_jids_of_user $user]
    if {[winfo exists [set m .jidpopupmenu]]} {
	destroy $m
    }
    menu $m -tearoff 0
    add_menu_item $m [::msgcat::mc "Start chat"] \
	{chat::open_to_user $curuser} $jids
    add_menu_item $m [::msgcat::mc "Send message..."] \
	{message::send_dialog -to $curuser} $jids
    add_menu_item $m [::msgcat::mc "Invite to conference..."] \
	{chat::invite_dialog $curuser} $jids
    $m add command -label [::msgcat::mc "Resubscribe"] -command {
	jlib::send_presence -to $curuser -type subscribe
    }

    hook::run roster_create_user_menu_hook $m $jids

    $m add separator
    add_custom_presence_menu $m $jids
    add_menu_item $m [::msgcat::mc "Send users..."] \
	{roster::send_users_dialog $curuser} $jids
    add_menu_item $m [::msgcat::mc "Send file..."] \
	{ft::send_file_dialog $curuser} $jids
    add_menu_item $m [::msgcat::mc "Send file via Jidlink..."] \
	{ftjl::send_file_dialog $curuser} $jids
    

    $m add separator
    add_menu_item $m [::msgcat::mc "Show info..."] \
	{userinfo::open $curuser} $jids
    $m add command -label [::msgcat::mc "Show history..."] \
	-command {logger::show_log $curuser}
    $m add separator
    $m add command -label [::msgcat::mc "Edit item..."] \
	-command {itemedit::show_dialog $curuser}
    $m add command -label [::msgcat::mc "Edit security..."] \
	-command {ssj::prefs $curuser} \
	-state [lindex {normal disabled} \
		    [cequal [info commands ::ssj::prefs] ""]]
    $m add separator
    $m add command -label [::msgcat::mc "Remove..."] \
	-command {roster::remove_item_dialog $curuser}

    return $m
}


set stdmenu {
    $m add separator
    $m add command -label [::msgcat::mc "Show info..."] \
	-command {userinfo::open $curuser}
    $m add command -label [::msgcat::mc "Show history..."] \
	-command {logger::show_log $curuser}
    $m add separator
    $m add command -label [::msgcat::mc "Edit item..."] \
	-command {itemedit::show_dialog $curuser}
    $m add command -label [::msgcat::mc "Edit security..."] \
	-command {ssj::prefs $curuser} \
	-state [lindex {normal disabled} \
		    [cequal [info commands ::ssj::prefs] ""]]
    $m add separator
    $m add command -label [::msgcat::mc "Remove..."] \
	-command {roster::remove_item_dialog $curuser}
}


if {[winfo exists [set m .jidpopupmenu]]} {
    destroy $m
}
menu $m
$m add command -label [::msgcat::mc "Start chat"] \
    -command {chat::open_to_user $curuser}
$m add command -label [::msgcat::mc "Send message..."] -command \
    {message::send_dialog -to $curuser}
$m add command -label [::msgcat::mc "Invite to conference..."] -command \
    {chat::invite_dialog $curuser}
$m add command -label [::msgcat::mc "Resubscribe"] -command {
    jlib::send_presence -to $curuser -type subscribe
}
$m add separator
$m add command -label [::msgcat::mc "Send users..."] \
    -command {roster::send_users_dialog $curuser}
$m add command -label [::msgcat::mc "Send file..."] \
    -command {ft::send_file_dialog $curuser}
$m add command -label [::msgcat::mc "Send file via Jidlink..."] \
    -command {ftjl::send_file_dialog $curuser}
eval $stdmenu


if {[winfo exists [set m .confpopupmenu]]} {
    destroy $m
}
menu $m
$m add command -label [::msgcat::mc "Join..."] -command {
    join_group_dialog -server [server_from_jid $curuser] \
	    -group [node_from_jid $curuser]
}
eval $stdmenu

if {[winfo exists [set m .servicepopupmenu]]} {
    destroy $m
}
menu $m
$m add command -label [::msgcat::mc "Log in"] -command {
    switch -- $userstatus {
       available { jlib::send_presence -to $curuser }
       invisible { jlib::send_presence -to $curuser -type $userstatus }
       default   { jlib::send_presence -to $curuser -show $userstatus }
    }
}

$m add command -label [::msgcat::mc "Log out"] -command {
    jlib::send_presence -to $curuser -type unavailable
}
eval $stdmenu
unset stdmenu

proc roster::create_groupchat_user_menu {jid} {
    if {[winfo exists [set m .groupchatpopupmenu]]} {
	destroy $m
    }
    menu $m -tearoff 0
    $m add command -label [::msgcat::mc "Start chat"] \
	-command [list chat::open_to_user $jid]
    #$m add command -label "Send message..." -command {}
    hook::run roster_create_groupchat_user_menu_hook $m $jid
    $m add separator
    $m add command -label [::msgcat::mc "Send users..."] \
	-command [list roster::send_users_dialog $jid]
    $m add command -label [::msgcat::mc "Send file..."] \
	-command [list ft::send_file_dialog $jid]
    $m add command -label [::msgcat::mc "Send file via Jidlink..."] \
	-command [list ftjl::send_file_dialog $jid]
    $m add command -label [::msgcat::mc "Invite to conference..."] \
	-command [list chat::invite_dialog $jid]
    $m add separator
    $m add command -label [::msgcat::mc "Show info..."] \
	-command [list userinfo::open $jid]
    $m add command -label [::msgcat::mc "Show history..."] \
	-command {} -state disabled
    return $m
}


proc roster::create_group_popup_menu {name} {
    if {[winfo exists [set m .grouppopupmenu]]} {
	destroy $m
    }
    menu $m -tearoff 0
    $m add command -label [::msgcat::mc "Rename..."] \
	-command [list roster::rename_group_dialog $name]
    $m add command \
	-label [::msgcat::mc "Resubscribe to all users in group..."] \
	-command [list roster::resubscribe_group $name]
    $m add command -label [::msgcat::mc "Remove..."] \
	-command [list roster::remove_group_dialog $name]
    return $m
}

###############################################################################

proc roster::remove_group_dialog {name} {
    set res [MessageDlg .remove_item -aspect 50000 -icon question -type user \
		 -buttons {yes no} -default 0 -cancel 1 \
		 -message [format [::msgcat::mc "Are you sure to remove group '%s' from roster?"] $name]]

    if {$res == 0} {
	send_rename_group $name ""
    }
}

proc roster::rename_group_dialog {name} {
    global new_roster_group_name

    set new_roster_group_name $name

    set w .roster_group_rename
    if {[winfo exists $w]} {
	destroy $w
    }
    
    Dialog $w -title [::msgcat::mc "Rename roster group"] \
	-separator 1 -anchor e -default 0 -cancel 1

    $w add -text [::msgcat::mc "OK"] -command "
	destroy [list $w]
	roster::send_rename_group [list $name] \$new_roster_group_name
    "
    $w add -text [::msgcat::mc "Cancel"] -command [list destroy $w]


    set p [$w getframe]
    
    label $p.lgroupname -text [::msgcat::mc "New group name:"]
    ecursor_entry [entry $p.groupname -textvariable new_roster_group_name]

    grid $p.lgroupname  -row 0 -column 0 -sticky e
    grid $p.groupname   -row 0 -column 1 -sticky ew

    focus $p.groupname
    $w draw
}

proc roster::send_rename_group {name new_name} {
    variable roster

    if {[string equal $new_name $name]} return

    set items {}

    foreach jid $roster(jids) {
	if {[lcontain $roster(group,$jid) $name] || \
		($name == [::msgcat::mc "Undefined"] && \
		     $roster(group,$jid) == {})} {
	    set idx [lsearch -exact $roster(group,$jid) $name]
	    if {$new_name != ""} {
		set roster(group,$jid) \
		    [lreplace $roster(group,$jid) $idx $idx $new_name]
	    } else {
		set roster(group,$jid) \
		    [lreplace $roster(group,$jid) $idx $idx]
	    }
	    set roster(group,$jid) [lrmdups $roster(group,$jid)]
	    lappend items [item_to_xml $jid]
	}
    }

    if {$items != {}} {
	jlib::send_iq set \
	    [jlib::wrapper:createtag query \
		 -vars {xmlns jabber:iq:roster} \
		 -subtags $items]
    }
}

proc roster::resubscribe_group {name} {
    variable roster

    foreach jid $roster(jids) {
	if {[lcontain $roster(group,$jid) $name]} {
	    lassign [get_category_and_subtype $jid] category type
	    if {$category == "user"} {
		jlib::send_presence -to $jid -type subscribe
	    }
	}
    }
}


proc roster::add_group_by_jid_regexp_dialog {} {
    global new_roster_group_rname
    global new_roster_group_regexp

    set w .roster_group_add_by_jid_regexp
    if {[winfo exists $w]} {
	destroy $w
    }
    
    Dialog $w -title [::msgcat::mc "Add roster group by JID regexp"] \
	-separator 1 -anchor e -default 0 -cancel 1

    $w add -text [::msgcat::mc "OK"] -command "
	destroy [list $w]
	roster::add_group_by_jid_regexp \
	    \$new_roster_group_rname \$new_roster_group_regexp
    "
    $w add -text [::msgcat::mc "Cancel"] -command [list destroy $w]


    set p [$w getframe]
    
    label $p.lgroupname -text [::msgcat::mc "New group name:"]
    ecursor_entry [entry $p.groupname -textvariable new_roster_group_rname]
    label $p.lregexp -text [::msgcat::mc "JID regexp:"]
    ecursor_entry [entry $p.regexp -textvariable new_roster_group_regexp]

    grid $p.lgroupname -row 0 -column 0 -sticky e
    grid $p.groupname  -row 0 -column 1 -sticky ew
    grid $p.lregexp    -row 1 -column 0 -sticky e
    grid $p.regexp     -row 1 -column 1 -sticky ew

    focus $p.groupname
    $w draw
}

proc roster::add_group_by_jid_regexp {name regexp} {
    variable roster

    if {$name == ""} return

    set items {}

    foreach jid $roster(jids) {
	if {[regexp $regexp $jid]} {
	    set idx [lsearch -exact $roster(group,$jid) $name]
	    lappend roster(group,$jid) $name
	    set roster(group,$jid) [lrmdups $roster(group,$jid)]
	    lappend items [item_to_xml $jid]
	}
    }

    if {$items != {}} {
	jlib::send_iq set \
	    [jlib::wrapper:createtag query \
		 -vars {xmlns jabber:iq:roster} \
		 -subtags $items]
    }
}



###############################################################################

proc roster::send_users_dialog {user} {
    global send_uc

    set jid [get_jid_of_user $user]

    if {[cequal $jid ""]} {
        set jid $user
    }

    set gw .contacts
    catch { destroy $gw }

    if {[catch { set nick [get_label $user] }]} {
	if {[catch { set nick [chat::get_nick $user groupchat] }]} {
	    set nick $user
	}
    }

    Dialog $gw \
        -title "[::msgcat::mc {Send contacts to}] $nick" \
        -separator 1 -anchor e -default 0 -cancel 1

    set gf [$gw getframe]

    set sw [ScrolledWindow $gf.sw]
    set sf [ScrollableFrame $sw.sf]
    pack $sw -expand yes -fill both
    $sw setwidget $sf
    set sff [$sf getframe]

    bindscroll $sf

    $gw add -text [::msgcat::mc "Send"] -command [list roster::send_users $gw $jid]
    $gw add -text [::msgcat::mc "Cancel"] -command "destroy $gw"

    catch { unset send_uc }

    set choices {}
    foreach choice [lsort -dictionary $roster::roster(jids)] {
	if {![cequal $roster::roster(category,$choice) conference]} {
            lappend choices [list $choice [roster::get_label $choice]]
	}
    }

    set i 0
    foreach choice [lsort -index 1 $choices] {
        set uc [lindex $choice 0]

        set cb [checkbutton $sff.$i -variable send_uc($uc) \
                    -text [lindex $choice 1]]
        bind $cb <Any-Enter>  [list balloon::default_balloon $cb enter  %X %Y \
                                    $uc]
        bind $cb <Any-Motion> [list balloon::default_balloon $cb motion %X %Y \
                                    $uc]
        bind $cb <Any-Leave>  [list balloon::default_balloon $cb leave  %X %Y]
        pack $cb -anchor w
        incr i
    }
    if {$i == 0} {
        MessageDlg ${gw}_err -aspect 50000 -icon info \
                -message [::msgcat::mc "No users in roster..."] -type user \
		-buttons ok -default 0 -cancel 0
        return
    }

    $gw draw
}

proc roster::send_users {gw jid} {
    variable roster
    global send_uc

    set sf [$gw getframe].sw.sf
    set choices {}
    foreach uc [array names send_uc] {
        if {$send_uc($uc)} {
            lappend choices $uc
        }
    }

    destroy $gw

    set subtags {}
    set body [::msgcat::mc "Contact Information"]
    foreach choice $choices {
	lappend subtags [item_to_xml $choice]
	set nick [roster::get_label $choice]
        append body "\n$nick - jabber:$choice"
    }

    message::send_msg $jid -type normal -body $body \
	        -xlist [list [jlib::wrapper:createtag x \
	                          -vars [list xmlns jabber:x:roster] \
	                          -subtags $subtags]]
}

###############################################################################

proc roster::export_to_file {} {
    variable roster

    set filename [tk_getSaveFile \
		      -initialdir ~/.tkabber/ \
		      -initialfile $::loginconf(user).roster \
		      -filetypes [list \
				      [list [::msgcat::mc "Roster Files"] \
					   .roster] \
				      [list [::msgcat::mc "All Files"] *]]]
    if {$filename != ""} {
	set items {}

	foreach jid $roster(jids) {
	    lappend items [item_to_xml $jid]
	}

	set fd [open $filename w]
	fconfigure $fd -encoding utf-8
	puts $fd $items
	close $fd
    }
}

proc roster::import_from_file {} {
    variable roster

    set filename [tk_getOpenFile \
		      -initialdir ~/.tkabber/ \
		      -initialfile $::loginconf(user).roster \
		      -filetypes [list \
				      [list [::msgcat::mc "Roster Files"] \
					   .roster] \
				      [list [::msgcat::mc "All Files"] *]]]
    if {$filename != ""} {
	set fd [open $filename r]
	fconfigure $fd -encoding utf-8
	set items [read $fd]
	close $fd

	if {$items != {}} {
	    jlib::send_iq set \
		[jlib::wrapper:createtag query \
		     -vars [list xmlns "jabber:iq:roster"] \
		     -subtags $items]
	}
    }
}

###############################################################################

proc roster::add_custom_presence_menu {m jids} {
    set mm [menu $m.custom_presence -tearoff 0]

    add_menu_item $mm [::msgcat::mc "Online"] \
	{send_custom_presence $curuser available} $jids
    add_menu_item $mm [::msgcat::mc "Free to chat"] \
	{send_custom_presence $curuser chat} $jids
    add_menu_item $mm [::msgcat::mc "Away"] \
	{send_custom_presence $curuser away} $jids
    add_menu_item $mm [::msgcat::mc "Extended away"] \
	{send_custom_presence $curuser xa} $jids
    add_menu_item $mm [::msgcat::mc "Do not disturb"] \
	{send_custom_presence $curuser dnd} $jids
    add_menu_item $mm [::msgcat::mc "Offline"] \
	{send_custom_presence $curuser unavailable} $jids

    $m add cascad -label [::msgcat::mc "Send custom presence"] -menu $mm
}

###############################################################################

set roster_main_menu \
    [list cascad [::msgcat::mc "Roster"] {} roster 1 \
	 [list \
	      [list command [::msgcat::mc "Add user..."] {} {} {} \
		   -command message::send_subscribe_dialog] \
	      [list command [::msgcat::mc "Add conference..."] {} {} {} \
		   -command {add_group_dialog}] \
	      [list command [::msgcat::mc "Add group by regexp on JIDs..."] {} {} {} \
		   -command {roster::add_group_by_jid_regexp_dialog}] \
	      [list checkbutton [::msgcat::mc "Show online users only"] \
		   {} {} {} \
		   -variable roster::show_only_online \
		   -command roster::changed_only_online] \
	      [list checkbutton [::msgcat::mc "Use aliases"] {} {} {} \
		   -variable roster::use_aliases] \
	      [list command [::msgcat::mc "Export roster..."] {} {} {} \
		   -command roster::export_to_file] \
	      [list command [::msgcat::mc "Import roster..."] {} {} {} \
		   -command [list roster::import_from_file]] \
	     ]]

trace variable roster::use_aliases w "roster::redraw ; #"


