#ifndef _TINYSNMP_MANAGER_UDP_H
#define _TINYSNMP_MANAGER_UDP_H

/*
 * Copyright (c) Abraham vd Merwe <abz@blio.com>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *	  notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *	  notice, this list of conditions and the following disclaimer in the
 *	  documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the author nor the names of other contributors
 *	  may be used to endorse or promote products derived from this software
 *	  without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <sys/socket.h>
#include <netinet/in.h>

#include <abz/typedefs.h>

/* maximum UDP datagram size */
#define UDP_DATAGRAM_SIZE 65535

typedef struct
{
   struct sockaddr_in addr;
   int fd;
   uint32_t timeout;
} udp_t;

/*
 * initialize udp data structure.
 *
 *     addr          address/port of destination host
 *     timeout       timeout in milliseconds for any operations. if timeout
 *                   is 0, the timeout is disabled (i.e. we don't use
 *                   non-blocking I/O).
 *
 * return 0 if successful, -1 if some error occurred. check errno to
 * see what error occurred.
 */
extern void udp_open (udp_t *udp,const struct sockaddr_in *addr,uint32_t timeout);

/*
 * clean up. this also closes the connection if you're currently
 * connected to a host.
 */
extern void udp_close (udp_t *udp);

/*
 * connect to the host specified in udp_open(). return -1 if some
 * error occurred, 0 if successful. check errno to see what error
 * occurred.
 */
extern int udp_connect (udp_t *udp);

/*
 * disconnect.
 */
extern void udp_disconnect (udp_t *udp);

/*
 * send a single packet.
 *
 *     buf   where the packet data is stored
 *     len   length of packet (in bytes)
 *
 * return -1 if an error occurred, 0 if successful. check errno
 * to see what error occurred.
 */
extern int udp_send (const udp_t *udp,const uint8_t *buf,uint16_t len);

/*
 * receive a single packet.
 *
 *     buf   where the packet data is stored
 *     len   when the function is called len should be set to
 *           the size of the buffer (in bytes). when the function
 *           returns, len will be adjusted to the size of received
 *           packet (in bytes).
 *
 * return -1 if an error occurred, 0 if successful. check errno
 * to see what error occurred.
 */
extern int udp_receive (const udp_t *udp,uint8_t *buf,uint16_t *len);

#endif	/* #ifndef _TINYSNMP_MANAGER_UDP_H */
